package org.nakedobjects.nos.security.ldap;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import javax.naming.AuthenticationException;
import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.security.AuthenticationRequest;
import org.nakedobjects.noa.security.Authenticator;
import org.nakedobjects.nof.core.conf.Configuration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.security.PasswordAuthenticationRequest;
import org.nakedobjects.nof.core.util.Assert;


public class LdapAuthenticator implements Authenticator {

    private static final Logger LOG = Logger.getLogger(LdapAuthenticator.class);
    private static final boolean FAILED_AUTHENTICATION = false;
    private final String ldapProvider;
    private final String ldapDn;
    private static final String AUTH_LDAPSERVER = "security.ldap.server";
    private static final String AUTH_LDAPDN = "security.ldap.dn";

    public LdapAuthenticator() {
        ldapProvider = NakedObjectsContext.getConfiguration().getString(Configuration.ROOT + AUTH_LDAPSERVER);
        ldapDn = NakedObjectsContext.getConfiguration().getString(Configuration.ROOT + AUTH_LDAPDN);
    }

    public boolean canAuthenticate(AuthenticationRequest request) {
        return request instanceof PasswordAuthenticationRequest;
    }

    private void setRoles(final DirContext authContext, final AuthenticationRequest request, final String username)
            throws NamingException {
        final List roles = new ArrayList();
        final String filter = "(objectclass=organizationalRole)";
        final SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[] { "cn" });
        final NamingEnumeration answer = authContext.search("uid=" + username + ", " + ldapDn, filter, controls);
        while (answer.hasMore()) {
            final SearchResult result = (SearchResult) answer.nextElement();
            final String roleName = (String) result.getAttributes().get("cn").get(0);
            roles.add(roleName);
            LOG.debug("Adding role: " + roleName);
        }
        request.setRoles((String[]) roles.toArray(new String[roles.size()]));
    }

    public boolean isValid(final AuthenticationRequest request) {
        final PasswordAuthenticationRequest passwordRequest = (PasswordAuthenticationRequest) request;
        final String username = passwordRequest.getName();
        Assert.assertNotNull(username);
        if (username.equals("")) {
            LOG.debug("empty username");
            return FAILED_AUTHENTICATION;
        }
        final String password = passwordRequest.getPassword();
        Assert.assertNotNull(password);

        final Hashtable env = new Hashtable(4);
        env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
        env.put(Context.PROVIDER_URL, ldapProvider);
        env.put(Context.SECURITY_PRINCIPAL, "uid=" + username + ", " + ldapDn);
        env.put(Context.SECURITY_CREDENTIALS, password);

        DirContext authContext = null;
        try {
            authContext = new InitialDirContext(env);
            setRoles(authContext, request, username);
            return true;
        } catch (final AuthenticationException e) {
            return false;
        } catch (final NamingException e) {
            throw new NakedObjectRuntimeException("Failed to authenticate using LDAP", e);
        } finally {
            try {
                if (authContext != null) {
                    authContext.close();
                }
            } catch (final NamingException e) {
                throw new NakedObjectRuntimeException("Failed to authenticate using LDAP", e);
            }
        }
    }

}

// Copyright (c) Naked Objects Group Ltd.
