package org.nakedobjects.nos.client.dnd.basic;

import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.Vector;

import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.reflect.NakedObjectAction;
import org.nakedobjects.noa.reflect.OneToManyAssociation;
import org.nakedobjects.noa.spec.Features;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.conf.Configuration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.context.Perspective;
import org.nakedobjects.nof.core.persist.AllInstances;
import org.nakedobjects.nof.core.security.ExplorationSession;
import org.nakedobjects.nos.client.dnd.Click;
import org.nakedobjects.nos.client.dnd.CompositeViewSpecification;
import org.nakedobjects.nos.client.dnd.Content;
import org.nakedobjects.nos.client.dnd.ContentDrag;
import org.nakedobjects.nos.client.dnd.Drag;
import org.nakedobjects.nos.client.dnd.DragStart;
import org.nakedobjects.nos.client.dnd.ObjectContent;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.UserActionSet;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAxis;
import org.nakedobjects.nos.client.dnd.ViewDrag;
import org.nakedobjects.nos.client.dnd.Workspace;
import org.nakedobjects.nos.client.dnd.action.AbstractUserAction;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Padding;
import org.nakedobjects.nos.client.dnd.view.simple.CompositeView;


public final class ApplicationWorkspace extends CompositeView implements Workspace {
    private static final String NAKEDOBJECTS_USERS = Configuration.ROOT + "exploration.users";
    protected Vector serviceViews;
    protected Vector iconViews;

    public ApplicationWorkspace(final Content content, final CompositeViewSpecification specification, final ViewAxis axis) {
        super(content, specification, axis);
        serviceViews = new Vector();
        iconViews = new Vector();
    }

    public void addServiceIcon(View serviceIcon) {
        add(serviceViews, serviceIcon);
    }

    public void addObjectIcon(View serviceIcon) {
        add(iconViews, serviceIcon);
    }

    public void addView(final View view) {
        add(views, view);
        getViewManager().setKeyboardFocus(view);
        view.getFocusManager().focusFirstChildView();
    }

    public View addOpenViewFor(final Naked object, final Location at) {
        View window = openViewFor(object, at, false);
        return window;
    }

    private View openViewFor(final Naked object, final Location at, final boolean asIcon) {
        View view = createSubviewFor(object, asIcon);
        view.setLocation(at);
        if (asIcon) {

        } else {
            addView(view);
        }
        return view;
    }

    public Drag dragStart(DragStart drag) {
        View subview = subviewFor(drag.getLocation());
        if (subview != null) {
            drag.subtract(subview.getLocation());
            return subview.dragStart(drag);
        } else {
            return null;
        }
    }

    public View addIconFor(final Naked object, final Location at) {
        return openViewFor(object, at, true);
    }

    // TODO check the dragging in of objects, flag to user that object cannot be dropped
    public void drop(final ContentDrag drag) {
        getFeedbackManager().showDefaultCursor();

        if (!drag.getSourceContent().isObject()) {
            return;
        }

        NakedObject perspective = (NakedObject) getContent().getNaked();

        if (drag.getSourceContent().getNaked() == getPerspective()) {
            getFeedbackManager().setAction("can' drop self on workspace");
            return;
        }

        NakedObject source = ((ObjectContent) drag.getSourceContent()).getObject();
        if (Features.isService(source.getSpecification())) {
            OneToManyAssociation fld = (OneToManyAssociation) perspective.getSpecification().getField("services");
            fld.addElement(perspective, source);
            invalidateContent();
        } else {
            if (!drag.isShift()) {
                OneToManyAssociation fld = (OneToManyAssociation) perspective.getSpecification().getField("objects");
                fld.addElement(perspective, source);
            }
        }

        View newView;
        if (Features.isService(source.getSpecification())) {
            return;
        } else {
            if (drag.isShift()) {
                newView = createSubviewFor(source, false);
                drag.getTargetView().addView(newView);
            } else {
                // place object onto desktop as icon
                View icon = drag.getSource();
                if (!icon.getSpecification().isOpen()) {
                    View[] subviews = getSubviews();
                    for (int i = 0; i < subviews.length; i++) {
                        if (subviews[i] == icon) {
                            icon.markDamaged();
                            Location at = drag.getTargetLocation();
                            at.subtract(drag.getOffset());
                            icon.setLocation(at);
                            icon.markDamaged();
                            return;
                        }
                    }
                }
                newView = createSubviewFor(source, true);
                // drag.getTargetView().addView(newView);
                addObjectIcon(newView);
            }
        }
        Location location = drag.getTargetLocation();
        location.subtract(drag.getOffset());
        newView.setLocation(location);
    }

    public void entered() {
    // prevents status details about "Persective..."
    }

    private NakedObject getPerspective() {
        return (NakedObject) getContent().getNaked();
    }

    public View createSubviewFor(final Naked object, final boolean asIcon) {
        View view;
        Content content = Toolkit.getContentFactory().createRootContent(object);
        if (asIcon) {
            view = Toolkit.getViewFactory().createIcon(content);
        } else {
            view = Toolkit.getViewFactory().createWindow(content);
        }
        return view;
    }

    public void drop(final ViewDrag drag) {
        getFeedbackManager().showDefaultCursor();

        final View sourceView = drag.getSourceView();
        if (sourceView.getSpecification() != null && sourceView.getSpecification().isSubView()) {
            if (sourceView.getSpecification().isOpen() && sourceView.getSpecification().isReplaceable()) {
                // TODO remove the open view from the container and place on
                // workspace; replace the internal view with an icon
            } else {
                Location newLocation = drag.getViewDropLocation();
                addOpenViewFor(sourceView.getContent().getNaked(), newLocation);
                sourceView.getState().clearViewIdentified();
            }
        } else {
            sourceView.markDamaged();
            Location newLocation = drag.getViewDropLocation();
            sourceView.setLocation(newLocation);
            sourceView.limitBoundsWithin(getSize());
            sourceView.markDamaged();
        }
    }

    public Padding getPadding() {
        return new Padding();
    }

    public Workspace getWorkspace() {
        return this;
    }

    public void lower(final View view) {
        if (views.contains(view)) {
            views.removeElement(view);
            views.insertElementAt(view, 0);
            markDamaged();
        }
    }

    public void raise(final View view) {
        if (views.contains(view)) {
            views.removeElement(view);
            views.addElement(view);
            markDamaged();
        }
    }

    public void removeView(final View view) {
        view.markDamaged();
        if (iconViews.contains(view)) {
            iconViews.remove(view);
            getViewManager().removeFromNotificationList(view);
            removeObject((NakedObject) view.getContent().getNaked());
        } else {
            super.removeView(view);
        }
    }

    public void removeObject(final NakedObject object) {
        final NakedObject perspective = (NakedObject) getContent().getNaked();
        OneToManyAssociation fld = (OneToManyAssociation) perspective.getSpecification().getField("objects");
        fld.removeElement(perspective, object);
    }

    public void removeViewsFor(final NakedObject object) {
        View views[] = getSubviews();
        for (int i = 0; i < views.length; i++) {
            View view = views[i];
            if (view.getContent().getNaked() == object) {
                view.dispose();
            }
        }
    }

    public void secondClick(final Click click) {
        View subview = subviewFor(click.getLocation());
        if (subview != null) {
            // ignore double-click on self - don't open up new view
            super.secondClick(click);
        }
    }

    public String toString() {
        return "Workspace" + getId();
    }

    public void viewMenuOptions(final UserActionSet options) {
        options.setColor(Toolkit.getColor("background.workspace-menu"));

        options.add(new AbstractUserAction("Close all") {
            public void execute(final Workspace workspace, final View view, final Location at) {
                View views[] = getWindowViews();
                for (int i = 0; i < views.length; i++) {
                    View v = views[i];
                    // if (v.getSpecification().isOpen()) {
                    v.dispose();
                    // }
                }
                markDamaged();
            }
        });

        options.add(new AbstractUserAction("Tidy up windows") {
            public void execute(final Workspace workspace, final View view, final Location at) {
                tidyViews(getWindowViews());
            }
        });

        options.add(new AbstractUserAction("Tidy up icons") {
            public void execute(final Workspace workspace, final View view, final Location at) {
                tidyViews(getObjectIconViews());
            }
        });

        options.add(new AbstractUserAction("Tidy up all") {
            public void execute(final Workspace workspace, final View view, final Location at) {
                tidyViews(getObjectIconViews());
                tidyViews(getWindowViews());
            }
        });

        options.add(new AbstractUserAction("Perspectives...", NakedObjectAction.EXPLORATION) {
            public void execute(final Workspace workspace, final View view, final Location at) {
                NakedObjectSpecification spec = NakedObjectsContext.getReflector().loadSpecification(Perspective.class);
                NakedCollection collection = NakedObjectsContext.getObjectPersistor()
                        .findInstances(new AllInstances(spec, false));
                addOpenViewFor(collection, at);
            }
        });

        options.add(new AbstractUserAction("Services...") {
            public void execute(final Workspace workspace, final View view, final Location at) {
                Object services[] = NakedObjectsContext.getObjectLoader().getServices();
                NakedObjectSpecification spec = NakedObjectsContext.getReflector().loadSpecification(Object.class);
                NakedCollection collection = NakedObjectsContext.getObjectLoader().createAdapterForCollection(services, spec);
                addOpenViewFor(collection, at);
            }
        });

        menuForChangingUsers(options);
    }

    private void menuForChangingUsers(final UserActionSet options) {
        // TODO pick out users from the perspectives, but only show when in exploration mode
        String users = NakedObjectsContext.getConfiguration().getString(NAKEDOBJECTS_USERS);
        if (users != null) {
            StringTokenizer st = new StringTokenizer(users, ",");
            if (st.countTokens() > 0) {
                UserActionSet set = new UserActionSet("Change user", options, NakedObjectAction.EXPLORATION);
                while (st.hasMoreTokens()) {
                    String token = st.nextToken();
                    menuOptionForChangingUser(set, token);
                }
                options.add(set);
            }
        }
    }

    private void menuOptionForChangingUser(UserActionSet set, String token) {
        int end = token.indexOf(':');
        if (end == -1) {
            end = token.length();
        }
        final String tokenName = token.substring(0, end).trim();
        // TODO also extract role names
        set.add(new AbstractUserAction(tokenName) {
            public void execute(Workspace workspace, View view, Location at) {
                ExplorationSession session = (ExplorationSession) NakedObjectsContext.getSession();
                session.setName(tokenName);
            }
        });
    }

    protected View[] subviews() {
        View v[] = new View[views.size() + serviceViews.size() + iconViews.size()];
        int offset = 0;
        Object[] src = serviceViews.toArray();
        System.arraycopy(src, 0, v, offset, src.length);
        offset += src.length;
        src = iconViews.toArray();
        System.arraycopy(src, 0, v, offset, src.length);
        offset += src.length;
        src = views.toArray();
        System.arraycopy(src, 0, v, offset, src.length);
        return v;
    }

    public void clearServiceViews() {
        Enumeration e = serviceViews.elements();
        while (e.hasMoreElements()) {
            View view = (View) e.nextElement();
            view.markDamaged();
        }
        serviceViews.clear();
    }

    protected View[] getWindowViews() {
        return createArrayOfViews(views);
    }

    private View[] createArrayOfViews(Vector views) {
        View[] array = new View[views.size()];
        views.copyInto(array);
        return array;
    }

    protected View[] getServiceIconViews() {
        return createArrayOfViews(serviceViews);
    }

    protected View[] getObjectIconViews() {
        return createArrayOfViews(iconViews);
    }

    private void tidyViews(View[] views) {
        for (int i = 0; i < views.length; i++) {
            View v = views[i];
            v.setLocation(ApplicationWorkspaceBuilder.UNPLACED);
        }
        invalidateLayout();
        markDamaged();
    }

}
// Copyright (c) Naked Objects Group Ltd.
