package org.nakedobjects.nos.client.dnd.basic;

import java.util.Enumeration;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.spec.Features;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.util.NotImplementedException;
import org.nakedobjects.nos.client.dnd.CompositeViewSpecification;
import org.nakedobjects.nos.client.dnd.Content;
import org.nakedobjects.nos.client.dnd.ContentFactory;
import org.nakedobjects.nos.client.dnd.ObjectContent;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAxis;
import org.nakedobjects.nos.client.dnd.ViewFactory;
import org.nakedobjects.nos.client.dnd.builder.AbstractViewBuilder;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Size;


/**
 * WorkspaceBuilder builds a workspace view for an ObjectContent view by finding a collection of classes from
 * a field called 'classes' and adding an icon for each element. Similarly, if there is a collection called
 * 'objects' its elements are also added to the display.
 * 
 * <p>
 * During lay-out any icons that have an UNPLACED location (-1, -1) are given a location. Objects of type
 * NakedClass are added to the left-hand side, while all other icons are placed on the right-hand side of the
 * workspace view. Open windows are displayed in the centre.
 */
public class ApplicationWorkspaceBuilder extends AbstractViewBuilder {
    private static final Logger LOG = Logger.getLogger(ApplicationWorkspaceBuilder.class);
    private static final int PADDING = 10;
    public static final Location UNPLACED = new Location(-1, -1);

    public void build(final View view1) {
        ApplicationWorkspace view = (ApplicationWorkspace) view1;
        NakedObject object = ((ObjectContent) view.getContent()).getObject();

        // REVIEW is this needed?
        view.clearServiceViews();

        if (object != null) {
            NakedObjectField[] flds = object.getSpecification().getDynamicallyVisibleFields(object);
            ViewFactory viewFactory = Toolkit.getViewFactory();
            ContentFactory contentFactory = Toolkit.getContentFactory();

            if (view.getSubviews().length == 0) {
                for (int f = 0; f < flds.length; f++) {
                    NakedObjectField field = flds[f];
                    Naked attribute = field.get(object);

                    if (field.getId().equals("objects") && field.isCollection()) {
                        Enumeration elements = ((NakedCollection) attribute).elements();
                        while (elements.hasMoreElements()) {
                            NakedObject obj = (NakedObject) elements.nextElement();
                            Content content = contentFactory.createRootContent(obj);
                            View objectIcon = viewFactory.createIcon(content);
                            objectIcon.setLocation(ApplicationWorkspaceBuilder.UNPLACED);
                            view.addView(objectIcon);
                        }
                    }
                }
            }

            for (int f = 0; f < flds.length; f++) {
                NakedObjectField field = flds[f];
                Naked attribute = field.get(object);

                if (field.getId().equals("services") && field.isCollection()) {
                    NakedObjectsContext.getObjectPersistor().resolveField(object, field);

                    Enumeration elements = ((NakedCollection) attribute).elements();
                    while (elements.hasMoreElements()) {
                        NakedObject service = (NakedObject) elements.nextElement();
                        Content content = contentFactory.createServiceContent(service);
                        View serviceIcon = viewFactory.createIcon(content);
                        view.addServiceIcon(serviceIcon);
                    }
                }
            }
        }

    }

    public boolean canDisplay(final Naked object) {
        return object instanceof NakedObject && object != null;
    }

    public Size getRequiredSize(final View view) {
        return new Size(600, 400);
    }

    public String getName() {
        return "Simple Workspace";
    }

    public void layout(View view1, Size maximumSize) {
        ApplicationWorkspace view = (ApplicationWorkspace) view1;

        int widthUsed = layoutServiceIcons(maximumSize, view);
        layoutObjectIcons(maximumSize, view);
        layoutWindowViews(maximumSize, view, widthUsed);
    }

    private void layoutWindowViews(final Size maximumSize, final ApplicationWorkspace view, final int xOffset) {
        Size size = view.getSize();
        size.contract(view.getPadding());

        int maxHeight = size.getHeight();
        int maxWidth = size.getWidth();

        int xWindow = xOffset + PADDING;
        int yWindow = PADDING;

        int xMinimized = 1;
        int yMinimized = maxHeight - 1;

        View views[] = view.getWindowViews();

        for (int i = 0; i < views.length; i++) {
            View subview = views[i];
            subview.layout(new Size(maximumSize));
        }

        for (int i = 0; i < views.length; i++) {
            View v = views[i];
            Size componentSize = v.getRequiredSize(new Size(size));
            v.setSize(componentSize);
            if (v instanceof MinimizedView) {
                Size s = v.getMaximumSize();
                if (xMinimized + s.getWidth() > maxWidth) {
                    xMinimized = 1;
                    yMinimized -= s.getHeight() + 1;
                }
                v.setLocation(new Location(xMinimized, yMinimized - s.getHeight()));
                xMinimized += s.getWidth() + 1;

            } else if (v.getLocation().equals(UNPLACED)) {
                int height = componentSize.getHeight() + 6;
                v.setLocation(new Location(xWindow, yWindow));
                yWindow += height;

            }
            v.limitBoundsWithin(maximumSize);
        }
    }

    private int layoutServiceIcons(final Size maximumSize, final ApplicationWorkspace view) {
        Size size = view.getSize();
        size.contract(view.getPadding());

        int maxHeight = size.getHeight();

        int xService = PADDING;
        int yService = PADDING;
        int maxServiceWidth = 0;

        View views[] = view.getServiceIconViews();
        for (int i = 0; i < views.length; i++) {
            View v = views[i];
            Size componentSize = v.getRequiredSize(new Size(size));
            v.setSize(componentSize);
            int height = componentSize.getHeight() + 6;

            Naked object = v.getContent().getNaked();
            if (Features.isService(object.getSpecification())) {
                if (yService + height > maxHeight) {
                    yService = PADDING;
                    xService += maxServiceWidth + PADDING;
                    maxServiceWidth = 0;
                    LOG.debug("creating new column at " + xService + ", " + yService);
                }
                LOG.debug("service icon at " + xService + ", " + yService);
                v.setLocation(new Location(xService, yService));
                maxServiceWidth = Math.max(maxServiceWidth, componentSize.getWidth());
                yService += height;
            }
            v.limitBoundsWithin(maximumSize);
        }
        
        return xService + maxServiceWidth;
    }

    private void layoutObjectIcons(final Size maximumSize, final ApplicationWorkspace view) {
        Size size = view.getSize();
        size.contract(view.getPadding());

        int maxWidth = size.getWidth();

        int xObject = maxWidth - PADDING;
        int yObject = PADDING;

        View views[] = view.getObjectIconViews();
        for (int i = 0; i < views.length; i++) {
            View v = views[i];
            Size componentSize = v.getRequiredSize(new Size(size));
            v.setSize(componentSize);
            if (v.getLocation().equals(UNPLACED)) {
                int height = componentSize.getHeight() + 6;
                v.setLocation(new Location(xObject - componentSize.getWidth(), yObject));
                yObject += height;
            }
            v.limitBoundsWithin(maximumSize);
        }
    }

    public View createCompositeView(final Content content, final CompositeViewSpecification specification, final ViewAxis axis) {
        throw new NotImplementedException();
    }

}
// Copyright (c) Naked Objects Group Ltd.
