package org.nakedobjects.nos.client.dnd.border;

import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Click;
import org.nakedobjects.nos.client.dnd.ContentDrag;
import org.nakedobjects.nos.client.dnd.Drag;
import org.nakedobjects.nos.client.dnd.DragStart;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAreaType;
import org.nakedobjects.nos.client.dnd.ViewDrag;
import org.nakedobjects.nos.client.dnd.drawing.Bounds;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Padding;
import org.nakedobjects.nos.client.dnd.drawing.Size;


public class AbstractBorder extends AbstractViewDecorator {
    protected int bottom;
    protected int left;
    private boolean onBorder;
    protected int right;
    protected int top;

    protected AbstractBorder(final View view) {
        super(view);
    }

    protected Bounds contentArea() {
        return new Bounds(getLeft(), getTop(), getSize().getWidth() - getLeft() - getRight(), getSize().getHeight() - getTop()
                - getBottom());
    }

    public View dragFrom(final Location location) {
        location.subtract(getLeft(), getTop());

        return super.dragFrom(location);
    }

    public void dragIn(final ContentDrag drag) {
        drag.subtract(getLeft(), getTop());
        super.dragIn(drag);
    }

    public void dragOut(final ContentDrag drag) {
        drag.subtract(getLeft(), getTop());
        super.dragOut(drag);
    }

    public Drag dragStart(final DragStart drag) {
        if (overContent(drag.getLocation())) {
            drag.subtract(getLeft(), getTop());
            return super.dragStart(drag);
        } else {
            return null;
        }
    }

    public void draw(final Canvas canvas) {
        if (Toolkit.debug) {
            canvas.drawDebugOutline(new Bounds(getSize()), getBaseline(), Toolkit.getColor("debug.bounds.border"));
        }
        wrappedView.draw(canvas.createSubcanvas(getLeft(), getTop(), getSize().getWidth() - getRight(), getSize().getHeight()
                - getBottom()));
    }

    public void drop(final ContentDrag drag) {
        drag.subtract(getLeft(), getTop());
        super.drop(drag);
    }

    public void drop(final ViewDrag drag) {
        drag.subtract(getLeft(), getTop());
        super.drop(drag);
    }

    public void firstClick(final Click click) {
        if (overContent(click.getLocation())) {
            click.subtract(getLeft(), getTop());
            wrappedView.firstClick(click);
        }
    }

    public int getBaseline() {
        return wrappedView.getBaseline() + getTop();
    }

    protected int getBottom() {
        return bottom;
    }

    protected int getLeft() {
        return left;
    }

    public Padding getPadding() {
        Padding padding = wrappedView.getPadding();
        padding.extendTop(getTop());
        padding.extendLeft(getLeft());
        padding.extendBottom(getBottom());
        padding.extendRight(getRight());

        return padding;
    }

    public Size getRequiredSize(final Size maximumSize) {
        maximumSize.contract(getLeft() + getRight(), getTop() + getBottom());
        Size size = wrappedView.getRequiredSize(maximumSize);
        size.extend(getLeft() + getRight(), getTop() + getBottom());

        return size;
    }

    protected int getRight() {
        return right;
    }

    public Size getSize() {
        Size size = wrappedView.getSize();
        size.extend(getLeft() + getRight(), getTop() + getBottom());

        return size;
    }

    protected int getTop() {
        return top;
    }

    protected void debugDetails(final DebugString debug) {
        super.debugDetails(debug);
        debug.appendln("border", getTop() + "/" + getBottom() + " " + getLeft() + "/" + getRight()
                + " (top/bottom left/right)");
        debug.appendln("contents", contentArea());
    }

    protected boolean overBorder(final Location location) {
        return !contentArea().contains(location);
    }

    protected boolean overContent(final Location location) {
        return contentArea().contains(location);
    }

    protected boolean isOnBorder() {
        return onBorder;
    }

    public View identify(final Location location) {
        getViewManager().getSpy().addTrace(this, "mouse location within border", location);
        getViewManager().getSpy().addTrace(this, "non border area", contentArea());

        if (overBorder(location)) {
            getViewManager().getSpy().addTrace(this, "over border area", contentArea());
            return getView();
        } else {
            location.add(-getLeft(), -getTop());
            return super.identify(location);
        }

    }

    public void mouseDown(final Click click) {
        if (overContent(click.getLocation())) {
            click.subtract(getLeft(), getTop());
            wrappedView.mouseDown(click);
        }
    }

    public void mouseMoved(final Location at) {
        boolean on = overBorder(at);
        if (onBorder != on) {
            markDamaged();
            onBorder = on;
        }

        if (!on) {
            at.move(-getLeft(), -getTop());
            wrappedView.mouseMoved(at);
        }
    }

    public void mouseUp(final Click click) {
        if (overContent(click.getLocation())) {
            click.subtract(getLeft(), getTop());
            wrappedView.mouseUp(click);
        }
    }

    public void exited() {
        onBorder = false;
        super.exited();
    }

    public View pickupContent(final Location location) {
        location.subtract(getLeft(), getTop());
        return super.pickupContent(location);
    }

    public View pickupView(final Location location) {
        if (overBorder(location)) {
            return Toolkit.getViewFactory().createDragViewOutline(getView());
        } else {
            location.subtract(getLeft(), getTop());
            return super.pickupView(location);
        }
    }

    public void secondClick(final Click click) {
        if (overContent(click.getLocation())) {
            click.subtract(getLeft(), getTop());
            wrappedView.secondClick(click);
        }
    }

    public void setMaximumSize(final Size size) {
        Size wrappedSize = new Size(size);
        wrappedSize.contract(getLeft() + getRight(), getTop() + getBottom());
        wrappedView.setMaximumSize(wrappedSize);
    }

    public void setSize(final Size size) {
        Size wrappedViewSize = new Size(size);
        wrappedViewSize.contract(getLeft() + getRight(), getTop() + getBottom());
        wrappedView.setSize(wrappedViewSize);
    }

    public void setBounds(final Bounds bounds) {
        Bounds wrappedViewBounds = new Bounds(bounds);
        wrappedViewBounds.contract(getLeft() + getRight(), getTop() + getBottom());
        wrappedView.setBounds(wrappedViewBounds);
    }

    public void thirdClick(final Click click) {
        if (overContent(click.getLocation())) {
            click.subtract(getLeft(), getTop());
            wrappedView.thirdClick(click);
        }
    }

    public ViewAreaType viewAreaType(final Location mouseLocation) {
        Size size = wrappedView.getSize();
        Bounds bounds = new Bounds(getLeft(), getTop(), size.getWidth(), size.getHeight());

        if (bounds.contains(mouseLocation)) {
            mouseLocation.subtract(getLeft(), getTop());

            return wrappedView.viewAreaType(mouseLocation);
        } else {
            return ViewAreaType.VIEW;
        }
    }

}
// Copyright (c) Naked Objects Group Ltd.
