package org.nakedobjects.nos.client.dnd.border;

import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Click;
import org.nakedobjects.nos.client.dnd.Drag;
import org.nakedobjects.nos.client.dnd.DragStart;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewDrag;
import org.nakedobjects.nos.client.dnd.ViewState;
import org.nakedobjects.nos.client.dnd.Workspace;
import org.nakedobjects.nos.client.dnd.action.WindowControl;
import org.nakedobjects.nos.client.dnd.drawing.Bounds;
import org.nakedobjects.nos.client.dnd.drawing.Color;
import org.nakedobjects.nos.client.dnd.drawing.DrawingUtil;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Offset;
import org.nakedobjects.nos.client.dnd.drawing.Size;
import org.nakedobjects.nos.client.dnd.drawing.Text;


public abstract class AbstractWindowBorder extends AbstractBorder {
    final protected static int LINE_THICKNESS = 5;
    private final static Text TITLE_STYLE = Toolkit.getText("small-title");
    private final int baseline;
    private final int titlebarHeight;
    protected WindowControl controls[];

    public AbstractWindowBorder(final View enclosedView) {
        super(enclosedView);

        titlebarHeight = Math.max(WindowControl.HEIGHT + VPADDING + TITLE_STYLE.getDescent(), TITLE_STYLE.getTextHeight());
        baseline = LINE_THICKNESS + WindowControl.HEIGHT;

        left = LINE_THICKNESS;
        right = LINE_THICKNESS;
        top = LINE_THICKNESS + titlebarHeight;
        bottom = LINE_THICKNESS;
    }

    public void debugDetails(final DebugString debug) {
        debug.append("WindowBorder " + left + " pixels\n");
        debug.append("           titlebar " + (top - titlebarHeight) + " pixels");
        super.debugDetails(debug);
    }

    public Drag dragStart(final DragStart drag) {
        if (overBorder(drag.getLocation())) {
            Location location = drag.getLocation();
            View dragOverlay = Toolkit.getViewFactory().createDragViewOutline(getView());
            return new ViewDrag(this, new Offset(location.getX(), location.getY()), dragOverlay);
        } else {
            return super.dragStart(drag);
        }
    }

    protected void setControls(final WindowControl[] controls) {
        this.controls = controls;
    }

    public void setSize(final Size size) {
        super.setSize(size);
        layoutControls(size.getWidth());
    }

    public void setBounds(final Bounds bounds) {
        super.setBounds(bounds);
        layoutControls(bounds.getWidth());
    }

    private void layoutControls(final int width) {
        int x = width - getPadding().getRight() - (WindowControl.WIDTH + HPADDING) * controls.length;
        int y = LINE_THICKNESS + VPADDING;

        for (int i = 0; i < controls.length; i++) {
            controls[i].setSize(controls[i].getRequiredSize(new Size()));
            controls[i].setLocation(new Location(x, y));
            x += controls[i].getSize().getWidth() + HPADDING;
        }
    }

    public void draw(final Canvas canvas) {
        Size s = getSize();
        int x = left;
        int width = s.getWidth();
        int height = s.getHeight();

        // blank background
        Bounds bounds = getBounds();
        canvas.drawSolidRectangle(3, 3, bounds.getWidth() - 6, bounds.getHeight() - 6, Toolkit.getColor("background.window." + getSpecification().getName()));

        boolean hasFocus = containsFocus();
        Color titleBarBackgroundColor = hasFocus ? Toolkit.getColor("primary3") : Toolkit.getColor("secondary3");
        Color titleBarTextColor = hasFocus ? Toolkit.getColor("black") : Toolkit.getColor("secondary1");
        Color borderColor = hasFocus ? Toolkit.getColor("primary1") : Toolkit.getColor("secondary1");
        Color insetColorLight = hasFocus ? Toolkit.getColor("primary2") : Toolkit.getColor("secondary2");
        Color insetColorDark = hasFocus ? Toolkit.getColor("black") : Toolkit.getColor("black");

        // slightly rounded grey border
        canvas.drawRectangle(1, 0, width - 2, height, borderColor);
        canvas.drawRectangle(0, 1, width, height - 2, borderColor);

        for (int i = 2; i < left; i++) {
            canvas.drawRectangle(i, i, width - 2 * i, height - 2 * i, borderColor);
        }

        ViewState state = getState();
        if (state.isActive()) {
            int i = left;
            canvas.drawRectangle(i, top, width - 2 * i, height - 2 * i - top, Toolkit.getColor("active"));
        }

        // vertical lines within border
        canvas.drawLine(2, 15, 2, height - 15, insetColorDark);
        canvas.drawLine(3, 16, 3, height - 14, insetColorLight);
        canvas.drawLine(width - 3, 15, width - 3, height - 15, insetColorDark);
        canvas.drawLine(width - 2, 16, width - 2, height - 14, insetColorLight);

        // horizontal lines within border
        canvas.drawLine(15, 2, width - 15, 2, insetColorDark);
        canvas.drawLine(16, 3, width - 14, 3, insetColorLight);
        canvas.drawLine(15, height - 3, width - 15, height - 3, insetColorDark);
        canvas.drawLine(16, height - 2, width - 14, height - 2, insetColorLight);

        // title bar
        canvas.drawSolidRectangle(left, LINE_THICKNESS, width - left - right, titlebarHeight, titleBarBackgroundColor);
        int y = LINE_THICKNESS + titlebarHeight - 1;
        canvas.drawLine(x, y, width - right - 1, y, borderColor);

        canvas.drawText(title(), x + HPADDING, baseline, titleBarTextColor, TITLE_STYLE);

        Color white = Toolkit.getColor("white");
        int hatchX = HPADDING + TITLE_STYLE.stringWidth(title()) + 10;
        int hatchWidth = controls[0].getBounds().getX() - hatchX - 10;
        int hatchY = LINE_THICKNESS + 2;
        int hatchHeight = titlebarHeight - 6;
        DrawingUtil.drawHatching(canvas, hatchX, hatchY, hatchWidth , hatchHeight, borderColor, white);
        
        // controls
        for (int i = 0; controls != null && i < controls.length; i++) {
            Canvas controlCanvas = canvas.createSubcanvas(controls[i].getBounds());
            controls[i].draw(controlCanvas);
        }

        super.draw(canvas);
    }

    

    protected abstract String title();

    public Size getRequiredSize(final Size maximumSize) {
        Size size = super.getRequiredSize(maximumSize);
        int width = getLeft() + HPADDING + TITLE_STYLE.stringWidth(title()) + HPADDING + controls.length
                * (WindowControl.WIDTH + HPADDING) + HPADDING + getRight();
        size.ensureWidth(width);
        return size;
    }

    public void secondClick(final Click click) {
        View button = overControl(click.getLocation());
        if (button == null) {
            super.secondClick(click);
        }
    }

    public void thirdClick(final Click click) {
        View button = overControl(click.getLocation());
        if (button == null) {
            super.thirdClick(click);
        }
    }

    public void firstClick(final Click click) {
        View button = overControl(click.getLocation());
        if (button == null) {
            if (overBorder(click.getLocation())) {
                Workspace workspace = getWorkspace();
                if (workspace != null) {
                    if (click.button2()) {
                        workspace.lower(getView());
                    } else if (click.button1()) {
                        workspace.raise(getView());
                    }
                }
            } else {
                super.firstClick(click);
            }

        } else {
            button.firstClick(click);
        }

    }

    private View overControl(final Location location) {
        for (int i = 0; i < controls.length; i++) {
            WindowControl control = controls[i];
            if (control.getBounds().contains(location)) {
                return control;
            }
        }
        return null;
    }

}
// Copyright (c) Naked Objects Group Ltd.
