package org.nakedobjects.nos.client.dnd.border;

import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Click;
import org.nakedobjects.nos.client.dnd.ContentDrag;
import org.nakedobjects.nos.client.dnd.Drag;
import org.nakedobjects.nos.client.dnd.DragStart;
import org.nakedobjects.nos.client.dnd.InternalDrag;
import org.nakedobjects.nos.client.dnd.SimpleInternalDrag;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.UserActionSet;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAreaType;
import org.nakedobjects.nos.client.dnd.Workspace;
import org.nakedobjects.nos.client.dnd.action.AbstractUserAction;
import org.nakedobjects.nos.client.dnd.drawing.Bounds;
import org.nakedobjects.nos.client.dnd.drawing.Color;
import org.nakedobjects.nos.client.dnd.drawing.DrawingUtil;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Offset;
import org.nakedobjects.nos.client.dnd.drawing.Size;
import org.nakedobjects.nos.client.dnd.view.simple.NullView;


/**
 * A scroll border provides a window on a larger view, providing scrollbars as a way of moving the visible
 * part of that view around the actual visible viewing area. To achieve this the view is divided up into five
 * main areas, not all of which are used. In the centre is the viewing area of the underlying view. At the
 * bottom and to the right... At the top and to the left are headers that
 */
public class ScrollBorder extends AbstractViewDecorator {
    private static final int CENTER = 3;
    private static final int NORTH = 1;
    private static final int SOUTH = 5;
    private static final int CORNER = 0;
    private static final int SCROLLBAR_WIDTH = 16;
    private static final int WEST = 2;
    private static final int EAST = 4;

    private ScrollBar horizontalScrollBar = new ScrollBar();
    private ScrollBar verticalScrollBar = new ScrollBar();
    protected int bottom;
    protected int left;
    private View leftHeader;
    protected int right;
    private Size size = new Size();
    protected int top;
    private View topHeader;
    private int dragArea = CENTER;
    private int offsetToThumbEdge;

    public ScrollBorder(final View view) {
        this(view, new NullView(), new NullView());
    }

    /**
     * Note - the leftHeader, if it is specified, view must be the same height as the content view and the
     * rightHeader, if it is specified, must be the same width.
     */
    public ScrollBorder(final View content, final View leftHeader, final View topHeader) {
        super(content);
        bottom = right = SCROLLBAR_WIDTH;
        horizontalScrollBar.setPostion(0);
        verticalScrollBar.setPostion(0);
        setLeftHeader(leftHeader);
        setTopHeader(topHeader);
    }

    public void setTopHeader(final View topHeader) {
        this.topHeader = topHeader;
        topHeader.setParent(getView());
        top = topHeader.getRequiredSize(new Size()).getHeight();
    }

    public void setLeftHeader(final View leftHeader) {
        this.leftHeader = leftHeader;
        leftHeader.setParent(getView());
        left = leftHeader.getRequiredSize(new Size()).getWidth();
    }

    private int adjust(final Click click) {
        return adjust(click.getLocation());
    }

    private int adjust(final ContentDrag drag) {
        return adjust(drag.getTargetLocation());
    }

    private int adjust(final Location location) {
        Bounds contentArea = viewportArea();
        Offset offset = offset();
        int yOffset = offset.getDeltaY();
        int xOffset = offset.getDeltaX();
        if (contentArea.contains(location)) {
            location.subtract(left, top);
            location.add(xOffset, yOffset);
            return CENTER;
        } else {
            int x = location.getX();
            int y = location.getY();

            if (x > contentArea.getX2() && y >= contentArea.getY() && y <= contentArea.getY2()) {
                // vertical scrollbar
                location.subtract(0, contentArea.getY());
                return EAST;
            } else if (y > contentArea.getY2() && x >= contentArea.getX() && x <= contentArea.getX2()) {
                // horzontal scrollbar
                location.subtract(contentArea.getX(), 0);
                return SOUTH;
            } else if (y < contentArea.getY() && x >= contentArea.getX() && x <= contentArea.getX2()) {
                // top border
                location.subtract(left, 0);
                location.add(xOffset, 0);
                return NORTH;
            } else if (x < contentArea.getX() && y >= contentArea.getY() && y <= contentArea.getY2()) {
                // left border
                location.subtract(0, top);
                location.add(0, yOffset);
                return WEST;
            } else {
                // ignore;
                location.setX(-1);
                location.setY(-1);
                return CORNER;
            }
        }

    }

    protected Bounds viewportArea() {
        return new Bounds(left, top, getSize().getWidth() - left - right, getSize().getHeight() - top - bottom);
    }

    protected void debugDetails(final DebugString debug) {
        super.debugDetails(debug);
        debug.append("\n           Top header: " + (topHeader == null ? "none" : topHeader.toString()));
        debug.append("\n           Left header: " + (leftHeader == null ? "none" : leftHeader.toString()));

        debug.append("\n           Vertical scrollbar ");
        debug.append("\n             offset " + top);
        debug.append("\n             position " + verticalScrollBar.getPosition());
        debug.append("\n             minimum " + verticalScrollBar.getMinimum());
        debug.append("\n             maximum " + verticalScrollBar.getMaximum());
        debug.append("\n             visible amount " + verticalScrollBar.getVisibleAmount());

        debug.append("\n           Horizontal scrollbar ");
        debug.append("\n             offset " + left);
        debug.append("\n             position " + horizontalScrollBar.getPosition());
        debug.append("\n             minimum " + horizontalScrollBar.getMinimum());
        debug.append("\n             maximum " + horizontalScrollBar.getMaximum());
        debug.append("\n             visible amount " + horizontalScrollBar.getVisibleAmount());
        debug.append("\n           Viewport area " + viewportArea());
        debug.append("\n           Offset " + offset());
    }

    public void drag(final InternalDrag drag) {
        switch (dragArea) {
        case NORTH:
            drag.getLocation().subtract(offset().getDeltaX(), top);
            topHeader.drag(drag);
            break;

        case WEST:
            drag.getLocation().subtract(left, offset().getDeltaY());
            leftHeader.drag(drag);
            break;

        case CENTER:
            drag.getLocation().subtract(offset());
            wrappedView.drag(drag);
            break;

        case SOUTH:
            int x = drag.getLocation().getX() - left;
            horizontalScrollBar.setPostion(x - offsetToThumbEdge);
            markDamaged();
            break;

        case EAST:
            int y = drag.getLocation().getY() - top;
            verticalScrollBar.setPostion(y - offsetToThumbEdge);
            markDamaged();
            break;

        default:
            return;
        }
    }

    public Drag dragStart(final DragStart drag) {
        int area = adjust(drag);
        dragArea = area;
        switch (dragArea) {
        case NORTH:
            return topHeader.dragStart(drag);

        case WEST:
            return leftHeader.dragStart(drag);

        case CENTER:
            return wrappedView.dragStart(drag);

        case SOUTH:
            return dragStartSouth(drag);

        case EAST:
            return dragStartEast(drag);

        default:
            return null;
        }
    }

    public void dragCancel(final InternalDrag drag) {
        switch (dragArea) {
        case NORTH:
            drag.getLocation().subtract(offset().getDeltaX(), top);
            topHeader.dragCancel(drag);
            break;

        case WEST:
            drag.getLocation().subtract(left, offset().getDeltaY());
            leftHeader.dragCancel(drag);
            break;

        case CENTER:
            drag.getLocation().subtract(offset());
            wrappedView.dragCancel(drag);
            break;
        }
    }

    public void dragTo(final InternalDrag drag) {
        switch (dragArea) {
        case NORTH:
            drag.getLocation().subtract(offset().getDeltaX(), top);
            topHeader.dragTo(drag);
            break;

        case WEST:
            drag.getLocation().subtract(left, offset().getDeltaY());
            leftHeader.dragTo(drag);
            break;

        case CENTER:
            drag.getLocation().subtract(offset());
            wrappedView.dragTo(drag);
            break;

        case SOUTH:
        case EAST:
        default:
            // ignore

        }
    }

    public View dragFrom(final Location location) {
        adjust(location);
        switch (dragArea) {
        case NORTH:
            return topHeader.dragFrom(location);

        case WEST:
            return leftHeader.dragFrom(location);

        case CENTER:
            return wrappedView.dragFrom(location);
        }

        return null;
    }

    public void dragIn(final ContentDrag drag) {
        adjust(drag);
        switch (dragArea) {
        case NORTH:
            topHeader.dragIn(drag);
            break;

        case WEST:
            leftHeader.dragIn(drag);
            break;

        case CENTER:
            wrappedView.dragIn(drag);
            break;

        case SOUTH:
        case EAST:
        default:
            System.out.println(this + " ignored");

            // ignore
        }
    }

    public void dragOut(final ContentDrag drag) {
        adjust(drag);
        switch (dragArea) {
        case NORTH:
            topHeader.dragOut(drag);
            break;

        case WEST:
            leftHeader.dragOut(drag);
            break;

        case CENTER:
            wrappedView.dragOut(drag);
            break;

        case SOUTH:
        case EAST:
        default:
            // ignore
        }
    }

    private Drag dragStartEast(final DragStart drag) {
        Location location = drag.getLocation();
        int y = location.getY();
        if (verticalScrollBar.isOnThumb(y)) {
            // offset is the distance from the left/top of the thumb to the pointer
            offsetToThumbEdge = y - verticalScrollBar.getPosition();
            return new SimpleInternalDrag(this, new Offset(super.getAbsoluteLocation()));
        } else {
            return null;
        }
    }

    private Drag dragStartSouth(final DragStart drag) {
        Location location = drag.getLocation();
        int x = location.getX();
        if (horizontalScrollBar.isOnThumb(x)) {
            offsetToThumbEdge = x - horizontalScrollBar.getPosition();
            return new SimpleInternalDrag(this, new Offset(super.getAbsoluteLocation()));
        } else {
            return null;
        }
    }

    private int adjust(final DragStart drag) {
        return adjust(drag.getLocation());
    }

    public void draw(final Canvas canvas) {
        Bounds contents = viewportArea();
        Offset offset = offset();
        int x = offset.getDeltaX();
        int y = offset.getDeltaY();

        int contentWidth = contents.getWidth();
        int contentHeight = contents.getHeight();

        Canvas headerCanvasLeft = canvas.createSubcanvas(0, top, left, contentHeight);
        headerCanvasLeft.offset(0, -y);
        leftHeader.draw(headerCanvasLeft);

        Canvas headerCanvasRight = canvas.createSubcanvas(left, 0, contentWidth, top);
        headerCanvasRight.offset(-x, 0);
        topHeader.draw(headerCanvasRight);

        Color thumbColor = Toolkit.getColor("primary2");
        drawHorizontalScrollBar(canvas, contentWidth, contentHeight, thumbColor);
        drawVerticalScrollBar(canvas, contentWidth, contentHeight, thumbColor);

        Canvas contentCanvas = canvas.createSubcanvas(left, top, contentWidth, contentHeight);
        contentCanvas.offset(-x, -y);

        if (Toolkit.debug) {
            canvas.drawRectangle(contents.getX(), contents.getY(), contents.getWidth(), contents.getHeight(), Toolkit
                    .getColor("debug.bounds.draw"));
        }

        // drawContent(canvas, contentWidth, contentHeight);
        wrappedView.draw(contentCanvas);

        if (Toolkit.debug) {
            Size size = getSize();
            canvas.drawRectangle(0, 0, size.getWidth(), size.getHeight(), Toolkit.getColor("debug.bounds.view"));
            canvas.drawLine(0, size.getHeight() / 2, size.getWidth() - 1, size.getHeight() / 2, Toolkit
                    .getColor("debug.bounds.view"));
            canvas.drawLine(0, getBaseline(), size.getWidth() - 1, getBaseline(), Toolkit.getColor("debug.baseline"));
        }

    }

    private void drawVerticalScrollBar(final Canvas canvas, final int contentWidth, final int contentHeight, final Color color) {
        int verticalVisibleAmount = verticalScrollBar.getVisibleAmount();
        int verticalScrollPosition = verticalScrollBar.getPosition();
        if (right > 0 && (verticalScrollPosition > top || verticalVisibleAmount < contentHeight)) {
            int x = contentWidth + left;
            canvas.drawSolidRectangle(x + 1, top, SCROLLBAR_WIDTH - 1, contentHeight, Toolkit.getColor("secondary3"));
            canvas.drawSolidRectangle(x + 1, top + verticalScrollPosition, SCROLLBAR_WIDTH - 2, verticalVisibleAmount, color);
            canvas.drawRectangle(x, top, SCROLLBAR_WIDTH, contentHeight, Toolkit.getColor("secondary2"));
            canvas.drawRectangle(x + 1, top + verticalScrollPosition, SCROLLBAR_WIDTH - 2, verticalVisibleAmount, Toolkit
                    .getColor("secondary1"));

            DrawingUtil.drawHatching(canvas, x + 3, top + verticalScrollPosition + 4, SCROLLBAR_WIDTH - 6,
                    verticalVisibleAmount - 8, Toolkit.getColor("primary1"), Toolkit.getColor("primary3"));
        }
    }

    // TODO merge these two methods
    /*
     * private void drawScrollBar(Canvas canvas, int contentWidth, int contentHeight, Color color) { int y =
     * top; int x = contentWidth + left; int verticalVisibleAmount = verticalScrollBar.getVisibleAmount(); int
     * verticalScrollPosition = verticalScrollBar.getPosition();
     * 
     * int height = verticalVisibleAmount; if (right > 0 && (verticalScrollPosition > y || height <
     * contentHeight)) { canvas.drawSolidRectangle(x + 1, y, SCROLLBAR_WIDTH - 2, contentHeight,
     * Toolkit.getColor("secondary3")); canvas.drawSolidRectangle(x + 1, y + verticalScrollPosition,
     * SCROLLBAR_WIDTH - 3, height, color); canvas.drawRectangle(x, y, SCROLLBAR_WIDTH - 1, contentHeight,
     * Toolkit.getColor("secondary2")); canvas.drawRectangle(x + 1, y + verticalScrollPosition,
     * SCROLLBAR_WIDTH - 3, height, Toolkit.getColor("secondary1")); } }
     */

    private void drawHorizontalScrollBar(final Canvas canvas, final int contentWidth, final int contentHeight, final Color color) {
        int horizontalScrollPosition = horizontalScrollBar.getPosition();
        int horizontalVisibleAmount = horizontalScrollBar.getVisibleAmount();
        if (bottom > 0 && (horizontalScrollPosition > left || horizontalVisibleAmount < contentWidth)) {
            int y = contentHeight + top;
            canvas.drawSolidRectangle(left, y + 1, contentWidth, SCROLLBAR_WIDTH - 1, Toolkit.getColor("secondary3"));
            int x = left + horizontalScrollPosition;
            canvas.drawSolidRectangle(x, y + 1, horizontalVisibleAmount, SCROLLBAR_WIDTH - 2, color);
            canvas.drawRectangle(left, y, contentWidth, SCROLLBAR_WIDTH, Toolkit.getColor("secondary2"));
            canvas.drawRectangle(x, y + 1, horizontalVisibleAmount, SCROLLBAR_WIDTH - 2, Toolkit.getColor("secondary1"));

            DrawingUtil.drawHatching(canvas, x + 5, y + 3, horizontalVisibleAmount - 10, SCROLLBAR_WIDTH - 6, Toolkit
                    .getColor("primary1"), Toolkit.getColor("primary3"));
        }
    }

    public void firstClick(final Click click) {
        int area = adjust(click);
        switch (area) {
        case NORTH:
            topHeader.firstClick(click);
            break;

        case WEST:
            leftHeader.firstClick(click);
            break;

        case CENTER:
            wrappedView.firstClick(click);
            break;

        case SOUTH:
            // TODO allow modified click to move thumb to the pointer, rather than paging.
            horizontalScrollBar.firstClick(click.getLocation().getX(), click.button3());
            break;

        case EAST:
            verticalScrollBar.firstClick(click.getLocation().getY(), click.button3());
            break;

        default:
            break;
        }
    }

    public Location getAbsoluteLocation() {
        Location location = super.getAbsoluteLocation();
        location.subtract(offset());
        return location;
    }

    public Bounds getBounds() {
        return new Bounds(getLocation(), getSize());
    }

    public Size getRequiredSize(final Size maximumSize) {
        Size size = wrappedView.getRequiredSize(new Size(maximumSize));
        if (size.getWidth() > maximumSize.getWidth()) {
            size.extendHeight(SCROLLBAR_WIDTH);
        }
        if (size.getHeight() > maximumSize.getHeight()) {
            size.extendWidth(SCROLLBAR_WIDTH);
        }
        size.extend(left, top);
        size.limitSize(maximumSize);
        return size;
    }

    public Size getSize() {
        return new Size(size);
    }

    public View identify(final Location location) {
        getViewManager().getSpy().addTrace(this, "mouse location within border", location);
        getViewManager().getSpy().addTrace(this, "non border area", viewportArea());

        int area = adjust(location);
        switch (area) {
        case NORTH:
            return topHeader.identify(location);

        case WEST:
            return leftHeader.identify(location);

        case CENTER:
            return wrappedView.identify(location);

        case SOUTH:
            getViewManager().getSpy().addTrace(this, "over scroll bar area", viewportArea());
            return getView();

        case EAST:
            getViewManager().getSpy().addTrace(this, "over scroll bar area", viewportArea());
            return getView();

        default:
            return null;
        }
    }

    public void limitBoundsWithin(final Size size) {
        super.limitBoundsWithin(size);
        verticalScrollBar.limit();
        horizontalScrollBar.limit();
    }

    public void markDamaged(final Bounds bounds) {
        /*
         * TODO this only works for the main content area, not for the headers. how do we figure out which
         * area to adjust for?
         */
        Offset offset = offset();
        bounds.translate(-offset.getDeltaX(), -offset.getDeltaY());
        bounds.translate(left, top);
        super.markDamaged(bounds);
    }

    public void mouseMoved(final Location location) {
        int area = adjust(location);
        switch (area) {
        case NORTH:
            topHeader.mouseMoved(location);
            break;

        case WEST:
            leftHeader.mouseMoved(location);
            break;

        case CENTER:
            // location.add(offset());
            // location.move(-left, -top);
            wrappedView.mouseMoved(location);
            break;

        case SOUTH:
        case EAST:
        default:
            break;
        }
    }

    private Offset offset() {
        Bounds contents = viewportArea();
        int width = contents.getWidth();
        int x = width == 0 ? 0 : horizontalScrollBar.getPosition() * wrappedView.getRequiredSize(new Size()).getWidth() / width;
        int height = contents.getHeight();
        int y = height == 0 ? 0 : verticalScrollBar.getPosition() * wrappedView.getRequiredSize(new Size()).getHeight() / height;
        return new Offset(x, y);
    }

    protected boolean overContent(final Location location) {
        return viewportArea().contains(location);
    }

    public void reset() {
        horizontalScrollBar.reset();
        verticalScrollBar.reset();
    }

    /**
     * Moves the scrollbar to beginning or the end when a double click occurs on that side.
     */
    public void secondClick(final Click click) {
        int area = adjust(click);
        switch (area) {
        case NORTH:
            topHeader.secondClick(click);
            break;

        case WEST:
            leftHeader.secondClick(click);
            break;

        case CENTER:
            wrappedView.secondClick(click);
            break;

        case SOUTH:
            horizontalScrollBar.secondClick(click.getLocation().getX());
            break;

        case EAST:
            verticalScrollBar.secondClick(click.getLocation().getY());
            break;

        default:
            break;
        }
    }

    public void thirdClick(final Click click) {
        int area = adjust(click);
        switch (area) {
        case NORTH:
            topHeader.thirdClick(click);
            break;

        case WEST:
            leftHeader.thirdClick(click);
            break;

        case CENTER:
            wrappedView.thirdClick(click);
            break;

        case SOUTH:
        case EAST:
        default:
            // ignore
            break;
        }
    }

    public void setBounds(final Bounds bounds) {
        setLocation(bounds.getLocation());
        setSize(bounds.getSize());
    }

    public void setMaximumSize(final Size size) {
        Size wrappedSize = new Size(size);
        wrappedSize.contract(left, top);
        wrappedView.setMaximumSize(wrappedSize);
    }

    public void setSize(final Size size) {
        // TODO need to restore the offset after size change - see limitBounds
        // float verticalRatio = ((float) verticalScrollPosition) /
        // contentArea().getHeight();

        this.size = new Size(size);

        Size contentSize = wrappedView.getRequiredSize(new Size());
        wrappedView.setSize(contentSize);

        int availableHeight2 = size.getHeight() - top;
        int contentHeight2 = contentSize.getHeight();
        right = availableHeight2 >= contentHeight2 ? 0 : SCROLLBAR_WIDTH;

        int availableWidth2 = size.getWidth() - left;
        int contentWidth2 = contentSize.getWidth();
        bottom = availableWidth2 >= contentWidth2 ? 0 : SCROLLBAR_WIDTH;

        Bounds viewport = viewportArea();

        int viewportHeight = viewport.getHeight();
        int maxContentHeight = Math.max(viewportHeight, contentSize.getHeight());

        verticalScrollBar.setSize(viewportHeight, maxContentHeight);
        if (leftHeader != null) {
            leftHeader.setSize(new Size(left, maxContentHeight));
        }

        int viewportWidth = viewport.getWidth();
        int maxContentWidth = Math.max(viewportWidth, contentSize.getWidth());

        horizontalScrollBar.setSize(viewportWidth, maxContentWidth);
        if (topHeader != null) {
            topHeader.setSize(new Size(maxContentWidth, top));
        }
    }

    public int getVerticalPosition() {
        return verticalScrollBar.getPosition();
    }

    public int getHorizontalPosition() {
        return horizontalScrollBar.getPosition();
    }

    public ViewAreaType viewAreaType(final Location location) {
        int area = adjust(location);
        switch (area) {
        case NORTH:
            return topHeader.viewAreaType(location);

        case WEST:
            return leftHeader.viewAreaType(location);

        case CENTER:
            return wrappedView.viewAreaType(location);

        case SOUTH:
        case EAST:
        default:
            return ViewAreaType.INTERNAL;
        }
    }

    public void viewMenuOptions(final UserActionSet menuOptions) {
        super.viewMenuOptions(menuOptions);
        menuOptions.add(new AbstractUserAction("Reset scroll border") {
            public void execute(final Workspace workspace, final View view, final Location at) {
                reset();
                invalidateLayout();
            }
        });
    }
}
// Copyright (c) Naked Objects Group Ltd.
