package org.nakedobjects.nos.client.dnd.content;

import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.reflect.Consent;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.reflect.OneToOneAssociation;
import org.nakedobjects.noa.spec.Features;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.reflect.Allow;
import org.nakedobjects.nof.core.reflect.Veto;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nos.client.dnd.OneToOneField;
import org.nakedobjects.nos.client.dnd.UserAction;
import org.nakedobjects.nos.client.dnd.UserActionSet;


public class OneToOneFieldImpl extends AbstractObjectContent implements OneToOneField {
    private static final UserAction CLEAR_ASSOCIATION = new ClearOneToOneAssociationOption();
    private final ObjectField field;
    private final NakedObject object;

    public OneToOneFieldImpl(final NakedObject parent, final NakedObject object, final OneToOneAssociation association) {
        field = new ObjectField(parent, association);
        this.object = object;
    }

    public Consent canClear() {
        NakedObject parentObject = getParent();
        OneToOneAssociation association = getOneToOneAssociation();
        // NakedObject associatedObject = getObject();

        Consent isEditable = isEditable();
        if (isEditable.isVetoed()) {
            return isEditable;
        }

        Consent isValid = association.isAssociationValid(parentObject, null);
        if (isValid.isAllowed()) {
            String status = "Clear the association to this object from '" + parentObject.titleString() + "'";
            return new Allow(status);
        } else {
            return new Veto(isValid.getReason());
        }
    }

    public Consent canSet(final NakedObject object) {
        NakedObjectSpecification targetType = getOneToOneAssociation().getSpecification();
        NakedObjectSpecification spec = object.getSpecification();

        if (isEditable().isVetoed()) {
            return isEditable();
        }

        if (!spec.isOfType(targetType)) {
            return new Veto("Can only drop objects of type " + targetType.getSingularName());
        }

        if (getParent().getResolveState().isPersistent() && object.getResolveState().isTransient()) {
            return new Veto("Can't drop a non-persistent into this persistent object");
        }

        // if (object instanceof Aggregated) {
        // Aggregated aggregated = ((Aggregated) object);
        // if (aggregated.isAggregated() && aggregated.parent() != getParent()) {
        if (Features.isAggregated(spec) && spec.getAggregate(object) != getParent()) {
            return new Veto("Object is already associated with another object: " + spec.getAggregate(object));
        }
        // }

        Consent perm = getOneToOneAssociation().isAssociationValid(getParent(), object);
        return perm;
    }

    public void clear() {
        getOneToOneAssociation().clearAssociation(getParent(), object);
    }

    public void debugDetails(final DebugString debug) {
        field.debugDetails(debug);
        debug.appendln("object", object);
    }

    public String getFieldName() {
        return field.getName();
    }

    public NakedObjectField getField() {
        return field.getFieldReflector();
    }

    public Consent isEditable() {
        Consent usable = getField().isUsable();
        if (usable.isVetoed()) {
            return usable;
        }
        return getField().isUsable(getParent());
    }

    public Naked getNaked() {
        return object;
    }

    public NakedObject getObject() {
        return object;
    }

    private OneToOneAssociation getOneToOneAssociation() {
        return (OneToOneAssociation) getField();
    }

    public Naked[] getOptions() {
        return (NakedObject[]) getOneToOneAssociation().getOptions(getParent());
    }

    public NakedObject getParent() {
        return field.getParent();
    }

    public NakedObjectSpecification getSpecification() {
        return getOneToOneAssociation().getSpecification();
    }

    public boolean isMandatory() {
        return getOneToOneAssociation().isMandatory();
    }

    public boolean isPersistable() {
        return getObject() != null && super.isPersistable();
    }

    public boolean isObject() {
        return true;
    }

    public boolean isOptionEnabled() {
        return getOneToOneAssociation().isOptionEnable();
    }

    public boolean isTransient() {
        return object != null && object.getResolveState().isTransient();
    }

    public void contentMenuOptions(final UserActionSet options) {
        super.contentMenuOptions(options);
        if (getObject() != null && !getOneToOneAssociation().isMandatory()) {
            options.add(CLEAR_ASSOCIATION);
        }
    }

    public void setObject(final NakedObject object) {
        getOneToOneAssociation().setAssociation(getParent(), object);
    }

    public String title() {
        return object == null ? "" : object.titleString();
    }

    public String toString() {
        return getObject() + "/" + getField();
    }

    public String windowTitle() {
        return field.getName() + " for " + field.getParent().titleString();
    }

    public String getId() {
        return getOneToOneAssociation().getName();
    }

    public String getDescription() {
        String name = getFieldName();
        String type = getField().getSpecification().getSingularName();
        type = name.indexOf(type) == -1 ? " (" + type + ")" : "";
        String description = getOneToOneAssociation().getDescription();
        return name + type + " " + description;
    }

    public String getHelp() {
        return getOneToOneAssociation().getHelp();
    }
}
// Copyright (c) Naked Objects Group Ltd.
