package org.nakedobjects.nos.client.dnd.image;

import java.awt.Toolkit;
import java.awt.image.FilteredImageSource;
import java.awt.image.RGBImageFilter;
import java.util.Hashtable;

import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.image.TemplateImage;
import org.nakedobjects.nof.core.image.TemplateImageLoader;
import org.nakedobjects.nos.client.dnd.drawing.Color;
import org.nakedobjects.nos.client.dnd.drawing.Image;
import org.nakedobjects.nos.client.dnd.util.Properties;


public class ImageFactory {

    private class Filter extends RGBImageFilter {

        public int filterRGB(final int x, final int y, final int rgb) {
            return 0xFFFFFF - rgb;
        }
    }
    
    private static final String DEFAULT_IMAGE_NAME = "Default";
    private static final String DEFAULT_IMAGE_PROPERTY = Properties.PROPERTY_BASE + "default-image";
    private static ImageFactory instance;
    private static final String SEPARATOR = "_";

    public static ImageFactory getInstance() {
        if (instance == null) {
            throw new IllegalStateException("Instance not set up yet");
        }
        return instance;
    }

    private TemplateImageLoader loader;
    /**
     * Keyed list of icons (each derived from an image, of a specific size etc), where the key is the name of
     * the icon and its size.
     */
    private Hashtable templateImages = new Hashtable();

    public ImageFactory(TemplateImageLoader imageLoader) {
        loader = imageLoader;
        instance = this;
    }

    private Image findIcon(final NakedObjectSpecification specification, final int iconHeight, final Color tint) {
        String className = specification.getFullName().replace('.', '_');
        Image loadIcon = loadIcon(className, iconHeight, tint);
        if (loadIcon == null) {
            loadIcon = findIconWithShortName(specification, iconHeight, tint);
        }
        return loadIcon;
    }

    private Image findIconForSuperClass(final NakedObjectSpecification specification, final int iconHeight, final Color tint) {
        NakedObjectSpecification superclassSpecification = specification.superclass();
        Image loadIcon;
        if (superclassSpecification == null) {
            loadIcon = null;
        } else {
            loadIcon = findIcon(superclassSpecification, iconHeight, tint);
        }
        return loadIcon;
    }

    private Image findIconWithShortName(final NakedObjectSpecification specification, final int iconHeight, final Color tint) {
        String className = specification.getShortName().replace('.', '_');
        Image loadIcon = loadIcon(className, iconHeight, tint);
        if (loadIcon == null) {
            loadIcon = findIconForSuperClass(specification, iconHeight, tint);
        }
        return loadIcon;
    }

    /**
     * Loads the fall back icon image, for use when no specific image can be found
     */
    public Image loadDefaultIcon(final int height, final Color tint) {
        String fallbackImage = NakedObjectsContext.getConfiguration().getString(DEFAULT_IMAGE_PROPERTY, DEFAULT_IMAGE_NAME);
        Image icon = loadIcon(fallbackImage, height, tint);
        if (icon == null) {
            icon = loadIcon("unknown", height, tint);
        }
        if (icon == null) {
            throw new NakedObjectRuntimeException("Failed to find default icon: " + fallbackImage);
        }
        return icon;
    }

    public Image loadIcon(final NakedObjectSpecification specification, final int iconHeight, final Color tint) {
        Image icon = findIcon(specification, iconHeight, null);
        return icon;
    }

    /**
     * Loads an icon of the specified size, and with the specified tint. If color is null then no tint is
     * applied.
     */
    public Image loadIcon(final String name, final int height, final Color tint) {
        final String id = name + SEPARATOR + height + SEPARATOR + tint;

        if (templateImages.containsKey(id)) {
            return (Image) templateImages.get(id);

        } else {
            TemplateImage template = templateImage(name);
            if (template == null) {
                return null;
            } else {
                java.awt.Image iconImage = template.getIcon(height);
                if (tint != null) {
                    Toolkit.getDefaultToolkit().createImage(new FilteredImageSource(iconImage.getSource(), new Filter()));
                }
                Image icon = new AwtImage(iconImage);
                
                templateImages.put(id, icon);
                return icon;
            }
        }
    }

    /**
     * Load an image with the given name.
     */
    public Image loadImage(final String path) {
        TemplateImage template = templateImage(path);
        if (template == null) {
            return null;
        }
        return new AwtImage(template.getImage());
    }

    private TemplateImage templateImage(final String name) {
        TemplateImage template = loader.getTemplateImage(name);
        return template;
    }

}
// Copyright (c) Naked Objects Group Ltd.
