package org.nakedobjects.nos.client.dnd.table;

import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Click;
import org.nakedobjects.nos.client.dnd.CollectionContent;
import org.nakedobjects.nos.client.dnd.Content;
import org.nakedobjects.nos.client.dnd.Drag;
import org.nakedobjects.nos.client.dnd.DragStart;
import org.nakedobjects.nos.client.dnd.InternalDrag;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAreaType;
import org.nakedobjects.nos.client.dnd.ViewAxis;
import org.nakedobjects.nos.client.dnd.basic.ResizeDrag;
import org.nakedobjects.nos.client.dnd.drawing.Bounds;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Shape;
import org.nakedobjects.nos.client.dnd.drawing.Size;
import org.nakedobjects.nos.client.dnd.view.simple.AbstractView;


public class TableHeader extends AbstractView {
    private int height;
    private int resizeColumn;

    public TableHeader(final Content content, final ViewAxis axis) {
        super(content, null, axis);
        height = VPADDING + Toolkit.getText("label").getTextHeight() + VPADDING;
    }

    public void firstClick(final Click click) {
        if (click.getLocation().getY() <= height) {
            TableAxis axis = ((TableAxis) getViewAxis());

            int column = axis.getColumnAt(click.getLocation().getX()) - 1;
            if (column == -2) {
                super.firstClick(click);
            } else if (column == -1) {
                ((CollectionContent) getContent()).setOrderByElement();
                invalidateContent();
            } else {
                NakedObjectField field = axis.getFieldForColumn(column);
                ((CollectionContent) getContent()).setOrderByField(field);
                invalidateContent();
            }
        } else {
            super.firstClick(click);
        }
    }

    public void invalidateContent() {
        getParent().invalidateContent();
    }

    public Size getMaximumSize() {
        return new Size(-1, height);
    }

    public Drag dragStart(final DragStart drag) {
        if (isOverColumnBorder(drag.getLocation())) {
            TableAxis axis = ((TableAxis) getViewAxis());
            resizeColumn = axis.getColumnBorderAt(drag.getLocation().getX());
            Bounds resizeArea = new Bounds(getView().getAbsoluteLocation(), getSize());
            resizeArea.translate(getView().getPadding().getLeft(), getView().getPadding().getTop());
            if (resizeColumn == 0) {
                resizeArea.setWidth(axis.getHeaderOffset());
            } else {
                resizeArea.translate(axis.getLeftEdge(resizeColumn - 1), 0);
                resizeArea.setWidth(axis.getColumnWidth(resizeColumn - 1));
            }

            Size minimumSize = new Size(70, 0);
            return new ResizeDrag(this, resizeArea, ResizeDrag.RIGHT, minimumSize, null);
        } else if (drag.getLocation().getY() <= height) {
            return null;
        } else {
            return super.dragStart(drag);
        }
    }

    public void dragTo(final InternalDrag drag) {
        if (drag.getOverlay() == null) {
            throw new NakedObjectRuntimeException("No overlay for drag: " + drag);
        }
        int newWidth = drag.getOverlay().getSize().getWidth();
        newWidth = Math.max(70, newWidth);
        getViewManager().getSpy().addAction("Resize column to " + newWidth);

        TableAxis axis = ((TableAxis) getViewAxis());
        if (resizeColumn == 0) {
            axis.setOffset(newWidth);
        } else {
            axis.setWidth(resizeColumn - 1, newWidth);
        }
        axis.invalidateLayout();
    }

    public void draw(final Canvas canvas) {
        super.draw(canvas.createSubcanvas());

        int y = VPADDING + Toolkit.getText("label").getAscent();

        TableAxis axis = ((TableAxis) getViewAxis());
        int x = axis.getHeaderOffset() - 2;

        if (((CollectionContent) getContent()).getOrderByElement()) {
            drawOrderIndicator(canvas, axis, x - 10);
        }

        canvas.drawLine(0, height - 1, getSize().getWidth() - 1, height - 1, Toolkit.getColor("secondary1"));
        canvas.drawLine(x, 0, x, getSize().getHeight() - 1, Toolkit.getColor("secondary1"));
        x++;
        int columns = axis.getColumnCount();
        NakedObjectField fieldSortOrder = ((CollectionContent) getContent()).getFieldSortOrder();
        for (int i = 0; i < columns; i++) {
            if (fieldSortOrder == axis.getFieldForColumn(i)) {
                drawOrderIndicator(canvas, axis, x + axis.getColumnWidth(i) - 10);
            }

            canvas.drawLine(x, 0, x, getSize().getHeight() - 1, Toolkit.getColor("black"));
            Canvas headerCanvas = canvas.createSubcanvas(x, 0, axis.getColumnWidth(i) - 1, height);
            headerCanvas.drawText(axis.getColumnName(i), HPADDING, y, Toolkit.getColor("secondary1"), Toolkit.getText("label"));
            x += axis.getColumnWidth(i);
        }
        canvas.drawLine(x, 0, x, getSize().getHeight() - 1, Toolkit.getColor("secondary2"));
        canvas.drawRectangle(0, height, getSize().getWidth() - 1, getSize().getHeight() - height - 1, Toolkit.getColor("secondary2"));
    }

    private void drawOrderIndicator(final Canvas canvas, final TableAxis axis, final int x) {
        Shape arrow;
        arrow = new Shape();
        if (((CollectionContent) getContent()).getReverseSortOrder()) {
            arrow.addVertex(0, 7);
            arrow.addVertex(3, 0);
            arrow.addVertex(6, 7);
        } else {
            arrow.addVertex(0, 0);
            arrow.addVertex(6, 0);
            arrow.addVertex(3, 7);
        }
        // canvas.drawRectangle(x + axis.getColumnWidth(i) - 10, 3, 7, 8, Toolkit.getColor("secondary3"));
        canvas.drawShape(arrow, x, 3, Toolkit.getColor("secondary2"));
    }

    public View identify(final Location location) {
        getViewManager().getSpy().addTrace("Identify over column " + location);
        if (isOverColumnBorder(location)) {
            getViewManager().getSpy().addAction("Identified over column ");
            return getView();
        }
        return super.identify(location);
    }

    private boolean isOverColumnBorder(final Location at) {
        int x = at.getX();
        TableAxis axis = ((TableAxis) getViewAxis());
        return axis.getColumnBorderAt(x) >= 0;
    }

    public void mouseMoved(final Location at) {
        if (isOverColumnBorder(at)) {
            getFeedbackManager().showResizeRightCursor();
        } else {
            super.mouseMoved(at);
            getFeedbackManager().showDefaultCursor();
        }
    }

    public void secondClick(final Click click) {
        if (isOverColumnBorder(click.getLocation())) {
            TableAxis axis = ((TableAxis) getViewAxis());
            int column = axis.getColumnBorderAt(click.getLocation().getX()) - 1;
            if (column == -1) {
                View[] subviews = getSubviews();
                for (int i = 0; i < subviews.length; i++) {
                    View row = subviews[i];
                    axis.ensureOffset(((TableRowBorder) row).requiredTitleWidth());
                }

            } else {
                View[] subviews = getSubviews();
                int max = 0;
                for (int i = 0; i < subviews.length; i++) {
                    View row = subviews[i];
                    View cell = row.getSubviews()[column];
                    max = Math.max(max, cell.getRequiredSize(new Size()).getWidth());
                }
                axis.setWidth(column, max);
            }
            axis.invalidateLayout();
        } else {
            super.secondClick(click);
        }
    }

    public String toString() {
        return "TableHeader";
    }

    public ViewAreaType viewAreaType(final Location at) {
        int x = at.getX();
        TableAxis axis = ((TableAxis) getViewAxis());

        if (axis.getColumnBorderAt(x) >= 0) {
            return ViewAreaType.INTERNAL;
        } else {
            return super.viewAreaType(at);
        }
    }
}
// Copyright (c) Naked Objects Group Ltd.
