package org.nakedobjects.nos.client.dnd.table;

import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Click;
import org.nakedobjects.nos.client.dnd.ContentDrag;
import org.nakedobjects.nos.client.dnd.Drag;
import org.nakedobjects.nos.client.dnd.DragStart;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAreaType;
import org.nakedobjects.nos.client.dnd.ViewDrag;
import org.nakedobjects.nos.client.dnd.border.AbstractBorder;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Offset;
import org.nakedobjects.nos.client.dnd.drawing.Size;
import org.nakedobjects.nos.client.dnd.view.graphic.IconGraphic;
import org.nakedobjects.nos.client.dnd.view.simple.DragViewOutline;
import org.nakedobjects.nos.client.dnd.view.text.ObjectTitleText;
import org.nakedobjects.nos.client.dnd.view.text.TitleText;

// REVIEW can we use ObjectBorder to provide the basic functionality
public class TableRowBorder extends AbstractBorder {
    private static final int BORDER = 13;

    private int baseline;
    private IconGraphic icon;
    private TitleText title;

    public TableRowBorder(final View wrappedRow) {
        super(wrappedRow);

        icon = new IconGraphic(this, Toolkit.getText("normal"));
        title = new ObjectTitleText(this, Toolkit.getText("normal"));
        baseline = icon.getBaseline();

        left = requiredTitleWidth() + BORDER;

        ((TableAxis) wrappedRow.getViewAxis()).ensureOffset(left);
    }

    public void debugDetails(final DebugString debug) {
        debug.append("RowBorder " + left + " pixels");
    }

    public Drag dragStart(final DragStart drag) {
        int x = drag.getLocation().getX();
        TableAxis axis = ((TableAxis) getViewAxis());
        int left = axis.getHeaderOffset();; 
        if (x < left - BORDER) {
            View dragOverlay = Toolkit.getViewFactory().getContentDragSpecification().createView(getContent(), null);
            return new ContentDrag(this, drag.getLocation(), dragOverlay);
        } else if (x < left) {
            View dragOverlay = new DragViewOutline(getView());
            return new ViewDrag(this, new Offset(drag.getLocation()), dragOverlay);
        } else {
            return super.dragStart(drag);
        }
    }

    public void draw(final Canvas canvas) {
        int baseline = getBaseline();

        TableAxis axis = ((TableAxis) getViewAxis());
        int width = axis.getHeaderOffset();
        Size s = getSize();
        Canvas subcanvas = canvas.createSubcanvas(0, 0, width, s.getHeight());
        int offset = HPADDING;
        icon.draw(subcanvas, offset, baseline);
        offset += icon.getSize().getWidth() + HPADDING + 0 + HPADDING;
        title.draw(subcanvas, offset, baseline, getLeft() - offset);

        int columns = axis.getColumnCount();
        int x = -1;
        x += axis.getHeaderOffset();
        canvas.drawLine(x - 1, 0, x - 1, s.getHeight() - 1, Toolkit.getColor("secondary1"));
        canvas.drawLine(x, 0, x, s.getHeight() - 1, Toolkit.getColor("secondary1"));
        for (int i = 0; i < columns; i++) {
            x += axis.getColumnWidth(i);
            canvas.drawLine(x, 0, x, s.getHeight() - 1, Toolkit.getColor("secondary1"));
        }

        int y = s.getHeight() - 1;
        canvas.drawLine(0, y, s.getWidth(), y, Toolkit.getColor("secondary2"));
        
        if (getState().isObjectIdentified()) {
            int xExtent = width - 1;
            canvas.drawLine(xExtent - BORDER, top, xExtent - BORDER, top + s.getHeight() - 1, Toolkit.getColor("secondary2"));
            canvas.drawSolidRectangle(xExtent - BORDER + 1, top, BORDER - 2, s.getHeight() - 2 * top - 1, Toolkit.getColor("secondary3"));
        }
        
        // components
        super.draw(canvas);
    }

    public int getBaseline() {
        return baseline;
    }

    protected int getLeft() {
        return ((TableAxis) wrappedView.getViewAxis()).getHeaderOffset();
    }

    protected int requiredTitleWidth() {
        return HPADDING + icon.getSize().getWidth() + HPADDING + title.getSize().getWidth() + HPADDING;
    }

    public void entered() {
        getState().setContentIdentified();
        getState().setViewIdentified();
        wrappedView.entered();
        markDamaged();
    }

    public void exited() {
        getState().clearObjectIdentified();
        getState().clearViewIdentified();
        wrappedView.exited();
        markDamaged();
    }

    public void secondClick(final Click click) {
        TableAxis axis = ((TableAxis) getViewAxis());
        int left = axis.getHeaderOffset();; 
        int x = click.getLocation().getX();
        if (x <= left) {
            /*
             * REVIEW the same thing is done in TreeNodeBorder, ObjectView and possibly elsewhere - should we
             * pass location and content to workspace add get it to open a new view
             */ 
            Location location = getAbsoluteLocation();
            location.translate(click.getLocation());
            View openWindow = Toolkit.getViewFactory().createWindow(getContent());
            openWindow.setLocation(location);
            getWorkspace().addView(openWindow);
        } else {
            super.secondClick(click);
        }
    }

    public String toString() {
        return wrappedView.toString() + "/RowBorder";
    }

    public ViewAreaType viewAreaType(final Location mouseLocation) {
        if (mouseLocation.getX() <= left) {
            return ViewAreaType.CONTENT;
        } else if (mouseLocation.getX() >= getSize().getWidth() - right) {
            return ViewAreaType.VIEW;
        } else {
            return super.viewAreaType(mouseLocation);
        }
    }
}
// Copyright (c) Naked Objects Group Ltd.
