package org.nakedobjects.nos.client.dnd.tree;

import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedCollection;
import org.nakedobjects.noa.spec.NakedObjectSpecification;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Content;
import org.nakedobjects.nos.client.dnd.FocusManager;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAxis;
import org.nakedobjects.nos.client.dnd.ViewSpecification;
import org.nakedobjects.nos.client.dnd.border.ScrollBorder;
import org.nakedobjects.nos.client.dnd.drawing.Bounds;
import org.nakedobjects.nos.client.dnd.drawing.Color;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Padding;
import org.nakedobjects.nos.client.dnd.drawing.Size;
import org.nakedobjects.nos.client.dnd.view.form.InternalFormSpecification;
import org.nakedobjects.nos.client.dnd.view.simple.AbstractView;


public class TreeBrowserFrame extends AbstractView implements ViewAxis {
    private ViewSpecification mainViewFormSpec;
    private ViewSpecification mainViewListSpec;
    private ViewSpecification mainViewTableSpec;
    private boolean invalidLayout = true;
    private int layoutCount = 0;
    private View left;
    private View right;
    private View selectedNode;
    private FocusManager focusManager;

    public TreeBrowserFrame(final Content content, final ViewSpecification specification) {
        super(content, specification, null);

        mainViewFormSpec = new InternalFormSpecification();
        mainViewTableSpec = new InternalTableSpecification();
        mainViewListSpec = new SimpleListSpecification();
    }

    public void debug(DebugString debug) {
        super.debug(debug);
        debug.append("\nLaid out:  " + (invalidLayout ? "no" : "yes") + ", " + layoutCount + " layouts");
        debug.append("\nBrowser:   ");
        debug.append(this);

        debug.append("\n           left: ");
        debug.append(left.getBounds());
        debug.append(" ");
        debug.append(left);
        debug.append(": ");
        debug.append(left.getContent());

        debug.append("\n           right: ");
        if (right == null) {
            debug.append("nothing");
        } else {
            debug.append(right.getBounds());
            debug.append(" ");
            debug.append(right);
            debug.append(": ");
            debug.append(right.getContent());
        }
        debug.append("\n\n");
    }

    public void dispose() {
        left.dispose();

        if (right != null) {
            right.dispose();
        }

        super.dispose();
    }

    public void invalidateContent() {
        if (left != null) {
            left.invalidateContent();
        }
        if (right != null) {
            right.invalidateContent();
        }
        super.invalidateContent();
    }

    public void draw(final Canvas canvas) {
        Bounds bounds = left.getBounds();
        Canvas subCanvas = canvas.createSubcanvas(bounds);
        Color leftBackground = Toolkit.getColor("background.window." + getSpecification().getName() +  ".left");
        if (leftBackground != Toolkit.getColor("background.window")) {
            subCanvas.clearBackground(left, leftBackground);
        }
        left.draw(subCanvas);
        
        if (right != null) {
            bounds = right.getBounds();
            subCanvas = canvas.createSubcanvas(bounds);
            Color rightBackground = Toolkit.getColor("background.window." + getSpecification().getName() + ".right");
            if (rightBackground != Toolkit.getColor("background.window")) {
                subCanvas.clearBackground(right, rightBackground);
            }
            right.draw(subCanvas);
        }
        canvas.drawLine(0, bounds.getHeight() - 1, getSize().getWidth(), bounds.getHeight() - 1, Toolkit.getColor("secondary1"));
    }

    public FocusManager getFocusManager() {
        return focusManager;
    }

    public Size getMaximumSize() {
        Size leftMaximumSize = left == null ? new Size() : left.getMaximumSize();
        Size rightMaximumSize = right == null ? new Size() : right.getMaximumSize();

        Size total = new Size(leftMaximumSize);
        total.extendWidth(rightMaximumSize.getWidth());
        total.ensureHeight(rightMaximumSize.getHeight());

        return total;
    }

    public Size getRequiredSize(final Size maximumSize) {
        Size leftPanelRequiredSize = left.getRequiredSize(new Size(maximumSize));
        Size rightPanelRequiredSize = right == null ? new Size() : right.getRequiredSize(new Size(maximumSize));

        if (leftPanelRequiredSize.getWidth() + rightPanelRequiredSize.getWidth() > maximumSize.getWidth()) {
            /*
             * If the combined width is greater than the availble then we need to divide the space between the
             * two sides and recalculate
             */
            int availableWidth = maximumSize.getWidth();
            int leftWidth = leftPanelRequiredSize.getWidth();
            int rightWidth = rightPanelRequiredSize.getWidth();
            int totalWidth = leftWidth + rightWidth;

            int bestWidth = (int) (1.0 * leftWidth / totalWidth * availableWidth);
            Size maximumSizeLeft = new Size(bestWidth, maximumSize.getHeight());
            leftPanelRequiredSize = left.getRequiredSize(maximumSizeLeft);

            Size maximumSizeRight = new Size(availableWidth - leftPanelRequiredSize.getWidth(), maximumSize.getHeight());
            rightPanelRequiredSize = right.getRequiredSize(maximumSizeRight);
        }

        // combine the two required sizes
        Size combinedSize = new Size(leftPanelRequiredSize);
        combinedSize.extendWidth(rightPanelRequiredSize.getWidth());
        combinedSize.ensureHeight(rightPanelRequiredSize.getHeight());
        return combinedSize;
    }

    public View getSelectedNode() {
        return selectedNode;
    }

    public View[] getSubviews() {
        return (right == null) ? new View[] { left } : new View[] { left, right };
    }

    public void invalidateLayout() {
        super.invalidateLayout();
        invalidLayout = true;
    }

    public void layout(final Size maximumSize) {
        if (invalidLayout) {
            maximumSize.contract(getView().getPadding());

            Size leftPanelRequiredSize = left.getRequiredSize(new Size(maximumSize));
            Size rightPanelRequiredSize = right == null ? new Size() : right.getRequiredSize(new Size(maximumSize));

            if (leftPanelRequiredSize.getWidth() + rightPanelRequiredSize.getWidth() > maximumSize.getWidth()) {
                /*
                 * If the combined width is greater than the availble then we need to divide the space between
                 * the two sides and recalculate
                 */
                int availableWidth = maximumSize.getWidth();
                int leftWidth = leftPanelRequiredSize.getWidth();
                int rightWidth = rightPanelRequiredSize.getWidth();
                int totalWidth = leftWidth + rightWidth;

                int bestWidth = (int) (1.0 * leftWidth / totalWidth * availableWidth);
                Size maximumSizeLeft = new Size(bestWidth, maximumSize.getHeight());
                leftPanelRequiredSize = left.getRequiredSize(maximumSizeLeft);

                Size maximumSizeRight = new Size(availableWidth - leftPanelRequiredSize.getWidth(), maximumSize.getHeight());
                rightPanelRequiredSize = right.getRequiredSize(maximumSizeRight);
            }

            // combine the two sizes
            Size combinedSize = new Size(leftPanelRequiredSize);
            combinedSize.extendWidth(rightPanelRequiredSize.getWidth());
            combinedSize.ensureHeight(rightPanelRequiredSize.getHeight());
            combinedSize.setHeight(Math.min(combinedSize.getHeight(), maximumSize.getHeight()));

            left.setSize(new Size(leftPanelRequiredSize.getWidth(), combinedSize.getHeight()));
            left.layout(new Size(new Size(leftPanelRequiredSize)));

            if (right != null) {
                right.setLocation(new Location(left.getSize().getWidth(), 0));
                rightPanelRequiredSize.setHeight(combinedSize.getHeight());

                right.setSize(rightPanelRequiredSize);
                right.layout(rightPanelRequiredSize);
            }

            layoutCount++;
            invalidLayout = false;
        }
    }

    public View subviewFor(final Location location) {
        Location l = new Location(location);
        Padding padding = getPadding();
        l.subtract(padding.getLeft(), padding.getTop());
        if (left.getBounds().contains(location)) {
            return left;
        } else if (right != null && right.getBounds().contains(location)) {
            return right;
        } else {
            return null;
        }
    }

    public void replaceView(final View toReplace, final View replacement) {
        if (toReplace == left) {
            initLeftPane(replacement);
            invalidateLayout();
        } else if (toReplace == right) {
            showInRightPane(replacement);
        } else {
            throw new NakedObjectRuntimeException();
        }
    }

    public void removeView(final View view) {
        if (view == left) {
            left = null;
        } else if (view == right) {
            right = null;
        }
    }

    void initLeftPane(View view) {
        left = new TreeBrowserResizeBorder(new ScrollBorder(view));
        left.setParent(getView());
    }

    protected void showInRightPane(final View view) {
        right = view;
        right.setParent(getView());
        invalidateLayout();
    }

    public void setFocusManager(final FocusManager focusManager) {
        this.focusManager = focusManager;
    }

    public void setSelectedNode(final View view) {
        Content content = view.getContent();
        Naked object = content.getNaked();
        NakedObjectSpecification specification = object.getSpecification();

        if (specification.getType() == NakedObjectSpecification.OBJECT) {
            if (object != null && mainViewFormSpec.canDisplay(content)) {
                selectedNode = view;
                showInRightPane(mainViewFormSpec.createView(content, null));
            }
        } else if (specification.getType() == NakedObjectSpecification.COLLECTION && ((NakedCollection) object).size() > 0) {
            if (mainViewTableSpec.canDisplay(content)) {
                selectedNode = view;
                showInRightPane(mainViewTableSpec.createView(content, null));
            } else if (mainViewListSpec.canDisplay(content)) {
                selectedNode = view;
                showInRightPane(mainViewListSpec.createView(content, null));
            }
        }
    }

    public String toString() {
        return "TreeBrowserFrame" + getId();
    }
}
// Copyright (c) Naked Objects Group Ltd.
