package org.nakedobjects.nos.client.dnd.tree;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.adapter.NakedReference;
import org.nakedobjects.noa.adapter.ResolveState;
import org.nakedobjects.noa.reflect.Consent;
import org.nakedobjects.noa.reflect.NakedObjectField;
import org.nakedobjects.noa.reflect.OneToManyAssociation;
import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Click;
import org.nakedobjects.nos.client.dnd.CollectionContent;
import org.nakedobjects.nos.client.dnd.ContentDrag;
import org.nakedobjects.nos.client.dnd.Drag;
import org.nakedobjects.nos.client.dnd.DragStart;
import org.nakedobjects.nos.client.dnd.FieldContent;
import org.nakedobjects.nos.client.dnd.ObjectContent;
import org.nakedobjects.nos.client.dnd.OneToManyField;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.UserActionSet;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAreaType;
import org.nakedobjects.nos.client.dnd.ViewDrag;
import org.nakedobjects.nos.client.dnd.ViewSpecification;
import org.nakedobjects.nos.client.dnd.Workspace;
import org.nakedobjects.nos.client.dnd.action.AbstractUserAction;
import org.nakedobjects.nos.client.dnd.border.AbstractBorder;
import org.nakedobjects.nos.client.dnd.content.CollectionElement;
import org.nakedobjects.nos.client.dnd.drawing.Bounds;
import org.nakedobjects.nos.client.dnd.drawing.Color;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Offset;
import org.nakedobjects.nos.client.dnd.drawing.Size;
import org.nakedobjects.nos.client.dnd.drawing.Text;
import org.nakedobjects.nos.client.dnd.view.graphic.IconGraphic;
import org.nakedobjects.nos.client.dnd.view.simple.DragViewOutline;
import org.nakedobjects.nos.client.dnd.view.text.ObjectTitleText;
import org.nakedobjects.nos.client.dnd.view.text.TitleText;


// TODO use ObjectBorder to provide the basic border functionality
public class TreeNodeBorder extends AbstractBorder {
    private static final int BORDER = 13;
    private static final int BOX_PADDING = 2;
    private static final int BOX_SIZE = 9;
    private static final int BOX_X_OFFSET = 5;
    private final static Text LABEL_STYLE = Toolkit.getText("normal");
    private static final Logger LOG = Logger.getLogger(TreeNodeBorder.class);
    private int baseline;
    private IconGraphic icon;
    private ViewSpecification replaceWithSpecification;
    private TitleText text;

    public TreeNodeBorder(final View wrappedView, final ViewSpecification replaceWith) {
        super(wrappedView);

        replaceWithSpecification = replaceWith;

        icon = new IconGraphic(this, LABEL_STYLE);
        text = new ObjectTitleText(this, LABEL_STYLE);
        int height = icon.getSize().getHeight();

        baseline = icon.getBaseline() + 1;

        left = 22;
        right = 0 + BORDER;
        top = height + 2;
        bottom = 0;
    }

    private int canOpen() {
        return ((NodeSpecification) getSpecification()).canOpen(getContent());
    }

    protected Bounds contentArea() {
        return new Bounds(getLeft(), getTop(), wrappedView.getSize().getWidth(), wrappedView.getSize().getHeight());
    }

    public void debugDetails(final DebugString debug) {
        debug.append("TreeNodeBorder " + left + " pixels\n");
        debug.append("           titlebar " + (top) + " pixels\n");
        debug.append("           replace with  " + replaceWithSpecification);
        debug.append("           text " + text);
        debug.append("           icon " + icon);
        super.debugDetails(debug);

    }

    public Drag dragStart(final DragStart drag) {
        if (drag.getLocation().getX() > getSize().getWidth() - right) {
            View dragOverlay = new DragViewOutline(getView());
            return new ViewDrag(this, new Offset(drag.getLocation()), dragOverlay);
        } else if (overBorder(drag.getLocation())) {
            View dragOverlay = Toolkit.getViewFactory().getContentDragSpecification().createView(getContent(), null);
            return new ContentDrag(this, drag.getLocation(), dragOverlay);
        } else {
            return super.dragStart(drag);
        }
    }

    public void draw(final Canvas canvas) {
        Color color = Toolkit.getColor("secondary2");
        if (((TreeBrowserFrame) getViewAxis()).getSelectedNode() == getView()) {
            canvas.drawSolidRectangle(left, 0, getSize().getWidth() - left, top, Toolkit.getColor("primary2"));
            color = Toolkit.getColor("secondary1");
        }

        if (getState().isObjectIdentified()) {
            canvas.drawRectangle(left, 0, getSize().getWidth() - left, top, color);

            int xExtent = getSize().getWidth();
            canvas.drawSolidRectangle(xExtent - BORDER + 1, 1, BORDER - 2, top - 2, Toolkit.getColor("secondary3"));
            canvas.drawLine(xExtent - BORDER, 0, xExtent - BORDER, top - 2, color);
        }

        // lines
        int x = 0;
        int y = top / 2;
        canvas.drawLine(x, y, x + left, y, Toolkit.getColor("secondary2"));

        boolean isOpen = getSpecification().isOpen();
        int canOpen = canOpen();
        boolean addBox = isOpen || canOpen != NodeSpecification.CANT_OPEN;
        if (addBox) {
            x += BOX_X_OFFSET;
            canvas.drawLine(x, y, x + BOX_SIZE - 1, y, Toolkit.getColor("secondary3"));
            canvas.drawSolidRectangle(x, y - BOX_SIZE / 2, BOX_SIZE, BOX_SIZE, Toolkit.getColor("white"));
            canvas.drawRectangle(x, y - BOX_SIZE / 2, BOX_SIZE, BOX_SIZE, Toolkit.getColor("secondary1"));

            if (canOpen == NodeSpecification.UNKNOWN) {

            } else {
                canvas.drawLine(x + BOX_PADDING, y, x + BOX_SIZE - 1 - BOX_PADDING, y, Toolkit.getColor("black"));
                if (!isOpen) {
                    x += BOX_SIZE / 2;
                    canvas.drawLine(x, y - BOX_SIZE / 2 + BOX_PADDING, x, y + BOX_SIZE / 2 - BOX_PADDING, Toolkit.getColor("black"));
                }
            }
        }

        View[] nodes = getSubviews();
        if (nodes.length > 0) {
            int y1 = top / 2;
            View node = nodes[nodes.length - 1];
            int y2 = top + node.getLocation().getY() + top / 2;
            canvas.drawLine(left - 1, y1, left - 1, y2, Toolkit.getColor("secondary2"));
        }

        // icon & title
        x = left + 1;
        icon.draw(canvas, x, baseline);
        x += icon.getSize().getWidth();
        text.draw(canvas, x, baseline);

        if (Toolkit.debug) {
            canvas.drawRectangleAround(this, Toolkit.getColor("debug.baseline"));
        }

        // draw components
        super.draw(canvas);
    }

    public void entered() {
        getState().setContentIdentified();
        getState().setViewIdentified();
        wrappedView.entered();
        markDamaged();
    }

    public void exited() {
        getState().clearObjectIdentified();
        getState().clearViewIdentified();
        wrappedView.exited();
        markDamaged();
    }

    public void firstClick(final Click click) {
        int x = click.getLocation().getX();
        int y = click.getLocation().getY();

        if (withinBox(x, y)) {
            if (canOpen() == NodeSpecification.UNKNOWN) {
                resolveContent();
                markDamaged();
            }
            LOG.debug((getSpecification().isOpen() ? "close" : "open") + " node " + getContent().getNaked());
            if (canOpen() == NodeSpecification.CAN_OPEN) {
                View newView = replaceWithSpecification.createView(getContent(), getViewAxis());
                getParent().replaceView(getView(), newView);
            }
        } else if (y < top && x > left) {
            if (canOpen() == NodeSpecification.UNKNOWN) {
                resolveContent();
                markDamaged();
            }
            selectNode();
        } else {
            super.firstClick(click);
        }
    }

    public int getBaseline() {
        return wrappedView.getBaseline() + baseline;
    }

    public Size getRequiredSize(final Size maximumSize) {
        Size size = super.getRequiredSize(maximumSize);
        // size.extendHeight(2 * VPADDING);
        size.ensureWidth(left + HPADDING + icon.getSize().getWidth() + text.getSize().getWidth() + HPADDING + right);
        return size;
    }

    public void objectActionResult(final Naked result, final Location at) {
        if (getContent() instanceof OneToManyField && result instanceof NakedObject) {
            // same as InternalCollectionBorder
            OneToManyField internalCollectionContent = (OneToManyField) getContent();
            OneToManyAssociation field = internalCollectionContent.getOneToManyAssociation();
            NakedObject target = ((ObjectContent) getParent().getContent()).getObject();

            Consent about = field.isValidToAdd(target, (NakedObject) result);
            if (about.isAllowed()) {
                field.addElement(target, (NakedObject) result);
            }
        }
        super.objectActionResult(result, at);
    }

    private void resolveContent() {
        Naked parent = getParent().getContent().getNaked();
        if (!(parent instanceof NakedObject)) {
            parent = getParent().getParent().getContent().getNaked();
        }

        if (getContent() instanceof FieldContent) {
            NakedObjectField field = ((FieldContent) getContent()).getField();
            NakedObjectsContext.getObjectPersistor().resolveField((NakedObject) parent, field);
        } else if (getContent() instanceof CollectionContent) {
            NakedObjectsContext.getObjectPersistor().resolveImmediately((NakedObject) parent);
        } else if (getContent() instanceof CollectionElement) {
            NakedObjectsContext.getObjectPersistor().resolveImmediately((NakedObject) getContent().getNaked());
        }
    }

    public void secondClick(final Click click) {
    	int x = click.getLocation().getX();
        int y = click.getLocation().getY();
        if (y < top && x > left) {
            if (canOpen() == NodeSpecification.UNKNOWN) {
                resolveContent();
                markDamaged();
            }
            Location location = getAbsoluteLocation();
            location.translate(click.getLocation());
            View openWindow = Toolkit.getViewFactory().createWindow(getContent());
            openWindow.setLocation(location);
            getWorkspace().addView(openWindow);
        } else {
            super.secondClick(click);
        }
    }

    private void selectNode() {
        TreeBrowserFrame axis = ((TreeBrowserFrame) getViewAxis());
        if (axis.getSelectedNode() != getView()) {
            LOG.debug("node selected " + getContent().getNaked());
            axis.setSelectedNode(getView());
        }
    }

    public String toString() {
        return wrappedView.toString() + "/TreeNodeBorder";
    }

    public ViewAreaType viewAreaType(final Location mouseLocation) {
        Bounds bounds = new Bounds(left + 1, 0, getSize().getWidth() - left - BORDER, top);
        if (bounds.contains(mouseLocation)) {
            return ViewAreaType.CONTENT;
        } else {
            return super.viewAreaType(mouseLocation);
        }
    }

    public void viewMenuOptions(final UserActionSet options) {
        super.viewMenuOptions(options);
        TreeDisplayRules.menuOptions(options);

        options.add(new AbstractUserAction("Select node") {
            public void execute(final Workspace workspace, final View view, final Location at) {
                selectNode();
            }

            public String getDescription(final View view) {
                return "Show this node in the right-hand pane";
            }
        });

        Naked object = getView().getContent().getNaked();
        ResolveState resolveState = ((NakedReference) object).getResolveState();
        if (object instanceof NakedReference && (resolveState.isGhost() || resolveState.isPartlyResolved())) {
            options.add(new AbstractUserAction("Load object") {
                public void execute(final Workspace workspace, final View view, final Location at) {
                    resolveContent();
                }
            });
        }
    }

    private boolean withinBox(final int x, final int y) {
        return x >= BOX_X_OFFSET && x <= BOX_X_OFFSET + BOX_SIZE && y >= (top - BOX_SIZE) / 2 && y <= (top + BOX_SIZE) / 2;
    }
}
// Copyright (c) Naked Objects Group Ltd.
