package org.nakedobjects.nos.client.dnd.view.field;

import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.adapter.NakedObject;
import org.nakedobjects.noa.reflect.Consent;
import org.nakedobjects.nof.core.util.NotImplementedException;
import org.nakedobjects.nos.client.dnd.BackgroundTask;
import org.nakedobjects.nos.client.dnd.BackgroundThread;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Content;
import org.nakedobjects.nos.client.dnd.InternalDrag;
import org.nakedobjects.nos.client.dnd.KeyboardAction;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.UserActionSet;
import org.nakedobjects.nos.client.dnd.ValueContent;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAreaType;
import org.nakedobjects.nos.client.dnd.ViewAxis;
import org.nakedobjects.nos.client.dnd.ViewSpecification;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Padding;
import org.nakedobjects.nos.client.dnd.view.simple.AbstractView;


public abstract class AbstractField extends AbstractView {
    protected static final int TEXT_WIDTH = 20;
    private boolean identified;

    protected AbstractField(final Content content, final ViewSpecification design, final ViewAxis axis) {
        super(content, design, axis);
    }

    public boolean canFocus() {
        return canChangeValue().isAllowed();
    }
    
    void clear() {}

    void copyToClipboard() {
    }
    
    public Consent canChangeValue() {
        ValueContent cont = (ValueContent) getContent();
        return cont.isEditable();
    }

    /**
     * Indicates the drag started within this view's bounds is continuing. By default does nothing.
     */
    public void drag(final InternalDrag drag) {}

    /**
     * Default implementation - does nothing
     */
    public void dragCancel(final InternalDrag drag) {}

    /**
     * Indicates the start of a drag within this view's bounds. By default does nothing.
     */
    public View dragFrom(final Location location) {
        return null;
    }

    /**
     * Indicates the drag started within this view's bounds has been finished (although the location may now
     * be outside of its bounds). By default does nothing.
     */
    public void dragTo(final InternalDrag drag) {}

    public void draw(final Canvas canvas) {
        if (getState().isActive()) {
            canvas.clearBackground(this, Toolkit.getColor("identified"));
        }

        if (getState().isOutOfSynch()) {
            canvas.clearBackground(this,Toolkit.getColor("out-of-sync"));
        }

        if (getState().isInvalid()) {
            canvas.clearBackground(this, Toolkit.getColor("invalid"));
        }

        super.draw(canvas);
    }

    /**
     * Indicates that editing has been completed and the entry should be saved. Will be called by the view
     * manager when other action place within the parent.
     * 
     */
    public void editComplete() {}

    public void entered() {
        super.entered();
        identified = true;
        Consent changable = canChangeValue();
        if (changable.isVetoed()) {
            getFeedbackManager().setViewDetail(changable.getReason());
        }
        markDamaged();
    }

    public void exited() {
        super.exited();
        identified = false;
        markDamaged();
    }

    public boolean getIdentified() {
        return identified;
    }

    public Padding getPadding() {
        return new Padding(0, 0, 0, 0);
    }

    public View getRoot() {
        throw new NotImplementedException();
    }

    String getSelectedText() {
        return "";
    }
    
    public boolean hasFocus() {
        return getViewManager().hasFocus(getView());
    }

    public boolean indicatesForView(final Location mouseLocation) {
        return false;
    }

    /**
     * Called when the user presses any key on the keyboard while this view has the focus.
     */
    public void keyPressed(final KeyboardAction key) {}

    /**
     * Called when the user releases any key on the keyboard while this view has the focus.
     * 
     * @param keyCode
     * @param modifiers
     */
    public void keyReleased(final int keyCode, final int modifiers) {}

    /**
     * Called when the user presses a non-control key (i.e. data entry keys and not shift, up-arrow etc). Such
     * a key press will result in a prior call to <code>keyPressed</code> and a subsequent call to
     * <code>keyReleased</code>.
     */
    public void keyTyped(final char keyCode) {}

    public void contentMenuOptions(final UserActionSet options) {
        options.add(new CopyValueOption(this));
        options.add(new PasteValueOption(this));
        options.add(new ClearValueOption(this));
        
        if (getView().getSpecification().isReplaceable()) {
            replaceOptions(Toolkit.getViewFactory().valueViews(getContent(), this), options);
        }

        super.contentMenuOptions((options));
        options.setColor(Toolkit.getColor("background.value-menu"));
    }

    protected final void initiateSave() {
        BackgroundThread.run(this, new BackgroundTask() {
            public void execute() {
                save();
                getParent().updateView();
                invalidateLayout();
            }

            public String getName() {
                return "Save field";
            }

            public String getDescription() {
                return "Saving " + getContent().windowTitle();
            }

        });
    }

    protected abstract void save();

    protected void saveValue(final NakedObject value) {
        parseEntry(value.titleString());
    }

    protected void parseEntry(final String entryText) {
        ValueContent content = (ValueContent) getContent();
        content.parseTextEntry(entryText);
        content.entryComplete();
    }

    void pasteFromClipboard() {} 
    
    public String toString() {
        String cls = getClass().getName();
        Naked naked = getContent().getNaked();
        return cls.substring(cls.lastIndexOf('.') + 1) + getId() + " [location=" + getLocation() + ",object="
                + (naked == null ? "" : naked.getObject()) + "]";
    }

    public ViewAreaType viewAreaType(final Location mouseLocation) {
        return ViewAreaType.INTERNAL;
    }

    public int getBaseline() {
        return Toolkit.defaultBaseline();
    }
}
// Copyright (c) Naked Objects Group Ltd.
