package org.nakedobjects.nos.client.dnd.view.field;

import org.nakedobjects.nof.core.context.NakedObjectsContext;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Content;
import org.nakedobjects.nos.client.dnd.KeyboardAction;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.UserActionSet;
import org.nakedobjects.nos.client.dnd.ValueContent;
import org.nakedobjects.nos.client.dnd.ViewAxis;
import org.nakedobjects.nos.client.dnd.ViewSpecification;
import org.nakedobjects.nos.client.dnd.drawing.Color;
import org.nakedobjects.nos.client.dnd.drawing.Size;
import org.nakedobjects.nos.client.dnd.drawing.Text;
import org.nakedobjects.nos.client.dnd.util.Properties;

import java.awt.event.KeyEvent;


public class PasswordField extends AbstractField {
    protected static final Text style = Toolkit.getText("normal");
    private boolean isSaved;
    private int maxTextWidth;
    private String password;
    private boolean identified;
    private char echoCharacter;

    public PasswordField(final Content content, final ViewSpecification design, final ViewAxis axis) {
        super(content, design, axis);
        setMaxTextWidth(TEXT_WIDTH);
        String echoCharacterSetting = NakedObjectsContext.getConfiguration().getString(Properties.PROPERTY_BASE + "echo");
        if (echoCharacterSetting == null || echoCharacterSetting.equals(" ")) {
            echoCharacter = '*';
        } else {
            echoCharacter = echoCharacterSetting.charAt(0);
        }
        password = text();
    }

    public void contentMenuOptions(final UserActionSet options) {
        options.add(new ClearValueOption(this));
        options.setColor(Toolkit.getColor("background.value-menu"));
    }
    
    void clear() {
        password = "";
        editComplete();
    }

    private void delete() {
        isSaved = false;
        int lastChar = password.length() - 1;
        password = password.substring(0, Math.max(0, lastChar));
        markDamaged();
    }

    public void draw(final Canvas canvas) {
        super.draw(canvas);

        Color color = identified ? Toolkit.getColor("identified") : Toolkit.getColor("secondary2");
        color = hasFocus() ? Toolkit.getColor("primary1") : color;
        int baseline = getBaseline();
        canvas.drawLine(HPADDING, baseline, HPADDING + getSize().getWidth(), baseline, color);

        String echoedPassword = echoPassword();
        int x = 3;
        Color textColor = hasFocus() ? Toolkit.getColor("text.edit") : Toolkit.getColor("text.saved");
        canvas.drawText(echoedPassword, x, getBaseline(), textColor, style);
        x += style.stringWidth(echoedPassword);
        if (hasFocus() && canChangeValue().isAllowed()) {
            canvas.drawLine(x, (baseline + style.getDescent()), x, 0, Toolkit.getColor("primary1"));
        }
    }

    private String echoPassword() {
        int length = password.length();
        String echoedPassword = "";
        for (int i = 0; i < length; i++) {
            echoedPassword += echoCharacter;
        }
        return echoedPassword;
    }

    public void editComplete() {
        if (canChangeValue().isAllowed() && !isSaved) {
            isSaved = true;
            initiateSave();
        }
    }

    public void escape() {
        password = "";
        isSaved = false;
        markDamaged();
    }

    public void entered() {
        if (canChangeValue().isAllowed()) {
            getFeedbackManager().showTextCursor();
            identified = true;
            markDamaged();
        }
    }

    public void exited() {
        if (canChangeValue().isAllowed()) {
            getFeedbackManager().showDefaultCursor();
            identified = false;
            markDamaged();
        }
    }

    public void focusLost() {
        editComplete();
    }

    public Size getMaximumSize() {
        int width = HPADDING + maxTextWidth + HPADDING;
        int height = style.getTextHeight() + VPADDING;
        height = Math.max(height, Toolkit.defaultFieldHeight());

        return new Size(width, height);
    }

    public void keyPressed(final KeyboardAction key) {
        if (!canChangeValue().isAllowed()) {
            return;
        }

        int keyCode = key.getKeyCode();
        if (keyCode == KeyEvent.VK_CONTROL || keyCode == KeyEvent.VK_SHIFT || keyCode == KeyEvent.VK_ALT) {
            return;
        }

        switch (keyCode) {
        case KeyEvent.VK_LEFT:
            key.consume();
            delete();
            break;
        case KeyEvent.VK_DELETE:
            key.consume();
            delete();
            break;
        case KeyEvent.VK_BACK_SPACE:
            key.consume();
            delete();
            break;
        case KeyEvent.VK_TAB:
            editComplete();
            break;
        case KeyEvent.VK_ENTER:
            key.consume();
            editComplete();
            getParent().keyPressed(key);
            break;
        case KeyEvent.VK_ESCAPE:
            key.consume();
            escape();
            break;

        default:
            break;
        }
    }

    public void keyTyped(final char keyCode) {
        password += keyCode;
        isSaved = false;
        markDamaged();
    }

    protected void save() {
        ValueContent content = (ValueContent) getContent();
        content.parseTextEntry(password);
        content.entryComplete();
    }

    /**
     * Set the maximum width of the field, as a number of characters
     */
    public void setMaxTextWidth(final int noCharacters) {
        maxTextWidth = style.charWidth('o') * noCharacters;
    }

    /**
     * Set the maximum width of the field, as a number of pixels
     */
    public void setMaxWidth(final int width) {
        maxTextWidth = width;
    }

    private String text() {
        String text;
        ValueContent content = (ValueContent) getContent();
        text = content.getNaked().titleString();
        return text;
    }

}
// Copyright (c) Naked Objects Group Ltd.
