package org.nakedobjects.nos.client.dnd.view.simple;

import java.util.Enumeration;

import org.apache.log4j.Logger;
import org.nakedobjects.noa.NakedObjectRuntimeException;
import org.nakedobjects.noa.adapter.Naked;
import org.nakedobjects.noa.reflect.Consent;
import org.nakedobjects.nof.core.reflect.Allow;
import org.nakedobjects.nof.core.reflect.Veto;
import org.nakedobjects.nof.core.undo.UndoStack;
import org.nakedobjects.nof.core.util.Assert;
import org.nakedobjects.nof.core.util.DebugString;
import org.nakedobjects.nof.core.util.UnexpectedCallException;
import org.nakedobjects.nos.client.dnd.Canvas;
import org.nakedobjects.nos.client.dnd.Click;
import org.nakedobjects.nos.client.dnd.Content;
import org.nakedobjects.nos.client.dnd.ContentDrag;
import org.nakedobjects.nos.client.dnd.Drag;
import org.nakedobjects.nos.client.dnd.DragStart;
import org.nakedobjects.nos.client.dnd.Feedback;
import org.nakedobjects.nos.client.dnd.FocusManager;
import org.nakedobjects.nos.client.dnd.InternalDrag;
import org.nakedobjects.nos.client.dnd.KeyboardAction;
import org.nakedobjects.nos.client.dnd.Toolkit;
import org.nakedobjects.nos.client.dnd.UserAction;
import org.nakedobjects.nos.client.dnd.UserActionSet;
import org.nakedobjects.nos.client.dnd.View;
import org.nakedobjects.nos.client.dnd.ViewAreaType;
import org.nakedobjects.nos.client.dnd.ViewAxis;
import org.nakedobjects.nos.client.dnd.ViewDrag;
import org.nakedobjects.nos.client.dnd.ViewSpecification;
import org.nakedobjects.nos.client.dnd.ViewState;
import org.nakedobjects.nos.client.dnd.Viewer;
import org.nakedobjects.nos.client.dnd.Workspace;
import org.nakedobjects.nos.client.dnd.action.AbstractUserAction;
import org.nakedobjects.nos.client.dnd.action.CloseAllViewsForObjectOption;
import org.nakedobjects.nos.client.dnd.action.CloseAllViewsOption;
import org.nakedobjects.nos.client.dnd.action.CloseViewOption;
import org.nakedobjects.nos.client.dnd.action.OpenViewOption;
import org.nakedobjects.nos.client.dnd.action.ReplaceViewOption;
import org.nakedobjects.nos.client.dnd.drawing.Bounds;
import org.nakedobjects.nos.client.dnd.drawing.Location;
import org.nakedobjects.nos.client.dnd.drawing.Padding;
import org.nakedobjects.nos.client.dnd.drawing.Size;


public abstract class AbstractView implements View {
    private static final UserAction CLOSE_ALL_OPTION = new CloseAllViewsOption();
    private static final UserAction CLOSE_OPTION = new CloseViewOption();
    private static final UserAction CLOSE_VIEWS_FOR_OBJECT = new CloseAllViewsForObjectOption();
    private static final Logger LOG = Logger.getLogger(AbstractView.class);

    private static int nextId = 0;
    private Content content;
    private int height;
    private int id = 0;
    private View parent;
    private ViewSpecification specification;
    private ViewState state;
    private View view;
    private ViewAxis viewAxis;
    private int width;
    private int x;
    private int y;

    protected AbstractView() {
        this(null, null, null);
    }

    protected AbstractView(final Content content, final ViewSpecification specification, final ViewAxis axis) {
        assignId();
        this.content = content;
        this.specification = specification;
        this.viewAxis = axis;
        state = new ViewState();
        view = this;
    }

    public void addView(final View view) {
        throw new NakedObjectRuntimeException("Can't add views to " + this);
    }

    protected void assignId() {
        id = nextId++;
    }

    public Consent canChangeValue() {
        return Veto.DEFAULT;
    }

    public boolean canFocus() {
        return true;
    }

    public boolean contains(final View view) {
        View[] subviews = getSubviews();
        for (int i = 0; i < subviews.length; i++) {
            if (subviews[i] == view || (subviews[i] != null && subviews[i].contains(view))) {
                return true;
            }
        }
        return false;
    }

    public boolean containsFocus() {
        if (hasFocus()) {
            return true;
        }

        View[] subviews = getSubviews();
        for (int i = 0; i < subviews.length; i++) {
            if (subviews[i] != null && subviews[i].containsFocus()) {
                return true;
            }
        }
        return false;
    }

    public void contentMenuOptions(final UserActionSet options) {
        options.setColor(Toolkit.getColor("background.content-menu"));

        Content content = getContent();
        if (content != null) {
            content.contentMenuOptions(options);
        }
    }

    /**
     * Returns debug details about this view.
     */
    public void debug(DebugString debug) {
        String name = getClass().getName();
        debug.appendln("Root", name.substring(name.lastIndexOf('.') + 1) + getId());
        debug.indent();
        debug.appendln("set size", getSize());
        debug.appendln("maximum", getMaximumSize());
        debug.appendln("required", getRequiredSize(new Size()));
        debug.appendln("w/in parent", getRequiredSize(getParent() == null ? new Size() : getParent().getSize()));
        debug.appendln("parent's", (getParent() == null ? new Size() : getParent().getSize()) + ")");
        debug.appendln("padding", getPadding());
        debug.appendln("base line", getBaseline() + "px");
        debug.unindent();
        debug.appendln();

        debug.appendTitle("Specification");
        if (specification == null) {
            debug.append("\none");
        } else {
            debug.appendln(specification.getName());
            debug.appendln("  " + specification.getClass().getName());
            debug.appendln("  " + (specification.isOpen() ? "open" : "closed"));
            debug.appendln("  " + (specification.isReplaceable() ? "replaceable" : "non-replaceable"));
            debug.appendln("  " + (specification.isSubView() ? "subview" : "main view"));
        }

        debug.appendln();
        debug.appendTitle("View");

        debug.appendln("Changable", canChangeValue());

        debug.appendln();
        debug.appendln("Focus", (canFocus() ? "focusable" : "non-focusable"));
        debug.appendln("Has", hasFocus());
        debug.appendln("Contains focus", containsFocus());
        debug.appendln("Focus manager", getFocusManager());

        debug.appendln();
        debug.appendln("Self", getView());
        debug.appendln("Axis", getViewAxis());
        debug.appendln("State", getState());
        debug.appendln("Location", getLocation());


        View p = getParent();
        String parent = p == null ? "none" : "" + p;
        debug.appendln("Parent", parent);

        debug.indent();
        while (p != null) {
            debug.appendln(p.toString());
            p = p.getParent();
        }
        debug.unindent();

        debug.appendln("Workspace", getWorkspace());

        debug.appendln();
        debug.appendln();
        debug.appendln();

        debug.appendTitle("View structure");
   //     b.appendln("Built", (buildInvalid ? "no" : "yes") + ", " + buildCount + " builds");
   //     b.appendln("Laid out", (layoutInvalid ? "no" : "yes") + ", " + layoutCount + " layouts");

        debugStructure(debug);
    }

    public void debugStructure(final DebugString b) {
        b.appendln(getSpecification().getName());
        b.indent();
        b.appendln("Content", getContent() == null ? null : getContent().getId());
        b.appendln("Required size ", getRequiredSize(new Size()));
        b.appendln("Bounds", getBounds());
        b.appendln("Baseline", getBaseline());
        View views[] = getSubviews();
        b.indent();
        for (int i = 0; i < views.length; i++) {
            View subview = views[i];
            subview.debugStructure(b);            
            b.unindent();
        }
        b.unindent();
    }

    public void dispose() {
        if (parent != null) {
            parent.removeView(getView());
        }
    }

    public void drag(final InternalDrag drag) {}

    public void drag(ContentDrag contentDrag) {}
    
    public void dragCancel(final InternalDrag drag) {
        getFeedbackManager().showDefaultCursor();
    }

    public View dragFrom(final Location location) {
        View subview = subviewFor(location);
        if (subview != null) {
            location.subtract(subview.getLocation());
            return subview.dragFrom(location);
        } else {
            return null;
        }
    }

    public void dragIn(final ContentDrag drag) {}

    public void dragOut(final ContentDrag drag) {}

    public Drag dragStart(final DragStart drag) {
        View subview = subviewFor(drag.getLocation());
        if (subview != null) {
            drag.subtract(subview.getLocation());
            return subview.dragStart(drag);
        } else {
            return null;
        }
    }

    public void dragTo(final InternalDrag drag) {}

    public void draw(final Canvas canvas) {
        if (Toolkit.debug) {
            canvas.drawDebugOutline(new Bounds(getSize()), getBaseline(), Toolkit.getColor("debug.bounds.view"));
        }
    }

    public void drop(final ContentDrag drag) {}

    /**
     * No default behaviour, views can only be dropped on workspace
     */
    public void drop(final ViewDrag drag) {}

    public void editComplete() {}

    public void entered() {
        Content cont = getContent();
        if (cont != null) {
            String description = cont.getDescription();
            if (description != null && !"".equals(description)) {
                getFeedbackManager().setViewDetail(description);
            }
        }
    }

    public void exited() {}

    public void firstClick(final Click click) {
        View subview = subviewFor(click.getLocation());
        if (subview != null) {
            click.subtract(subview.getLocation());
            subview.firstClick(click);
        }
    }

    public void focusLost() {
    }

    public void focusReceived() {
    }

    public Location getAbsoluteLocation() {
        if (parent == null) {
            return getLocation();
        } else {
            Location location = parent.getAbsoluteLocation();
            getViewManager().getSpy().addTrace(this, "parent location", location);
            location.add(x, y);
            getViewManager().getSpy().addTrace(this, "plus view's location", location);
            Padding pad = parent.getPadding();
            location.add(pad.getLeft(), pad.getTop());
            getViewManager().getSpy().addTrace(this, "plus view's padding", location);
            return location;
        }
    }

    public int getBaseline() {
        return 0;
    }

    public Bounds getBounds() {
        return new Bounds(x, y, width, height);
    }

    public Content getContent() {
        return content;
    }

    public FocusManager getFocusManager() {
        return getParent() == null ? null : getParent().getFocusManager();
    }

    public int getId() {
        return id;
    }

    public Location getLocation() {
        return new Location(x, y);
    }

    public Padding getPadding() {
        return new Padding(0, 0, 0, 0);
    }

    public final View getParent() {
        Assert.assertEquals(parent == null ? null : parent.getView(), parent);
        return parent;
    }

    public Size getRequiredSize(final Size maximumSize) {
        return getMaximumSize();
    }

    public Size getMaximumSize() {
        return new Size();
    }

    public Size getSize() {
        return new Size(width, height);
    }

    public ViewSpecification getSpecification() {
        if (specification == null) {
            specification = new NonBuildingSpecification(this);
        }
        return specification;
    }

    public ViewState getState() {
        return state;
    }

    public View[] getSubviews() {
        return new View[0];
    }

    public final View getView() {
        return view;
    }

    public final ViewAxis getViewAxis() {
        return viewAxis;
    }

    public Viewer getViewManager() {
        return Toolkit.getViewer();
    }

    public Feedback getFeedbackManager() {
        return Toolkit.getFeedbackManager();
    }

    public Workspace getWorkspace() {
        return getParent() == null ? null : getParent().getWorkspace();
    }

    public boolean hasFocus() {
        return getViewManager().hasFocus(getView());
    }

    public View identify(final Location location) {
        View subview = subviewFor(location);
        if (subview == null) {
            getViewManager().getSpy().addTrace(this, "mouse location within node view", location);
            getViewManager().getSpy().addTrace("----");
            return getView();
        } else {
            location.subtract(subview.getLocation());
            return subview.identify(location);
        }
    }

    public void invalidateContent() {}

    public void invalidateLayout() {
        if (parent != null) {
            parent.invalidateLayout();
        }
    }

    public void keyPressed(final KeyboardAction key) {}

    public void keyReleased(final int keyCode, final int modifiers) {}

    public void keyTyped(final char keyCode) {}

    public void layout(final Size maximumSize) {}

    /**
     * Limits the bounds of the this view (when being moved or dropped) so it never extends outside the
     * specified bounds e.g. outside of a parent view
     */
    public void limitBoundsWithin(final Bounds containerBounds) {
        Bounds contentBounds = getView().getBounds();
        if (containerBounds.limitBounds(contentBounds)) {
            getView().setBounds(contentBounds);
        }
    }

    public void limitBoundsWithin(final Size size) {
        int w = getView().getSize().getWidth();
        int h = getView().getSize().getHeight();

        int x = getView().getLocation().getX();
        int y = getView().getLocation().getY();

        if (x + w > size.getWidth()) {
            x = size.getWidth() - w;
        }
        if (x < 0) {
            x = 0;
        }

        if (y + h > size.getHeight()) {
            y = size.getHeight() - h;
        }
        if (y < 0) {
            y = 0;
        }

        getView().setLocation(new Location(x, y));
    }

    public void markDamaged() {
        markDamaged(getView().getBounds());
    }

    public void markDamaged(final Bounds bounds) {
        if (parent == null) {
            getViewManager().markDamaged(bounds);
        } else {
            Location pos = parent.getLocation();
            bounds.translate(pos.getX(), pos.getY());
            Padding pad = parent.getPadding();
            bounds.translate(pad.getLeft(), pad.getTop());
            parent.markDamaged(bounds);
        }
    }

    public void mouseDown(final Click click) {
        View subview = subviewFor(click.getLocation());
        if (subview != null) {
            click.subtract(subview.getLocation());
            subview.mouseDown(click);
        }
    }

    public void mouseMoved(final Location location) {
        View subview = subviewFor(location);
        if (subview != null) {
            location.subtract(subview.getLocation());
            subview.mouseMoved(location);
        }
    }

    public void mouseUp(final Click click) {
        View subview = subviewFor(click.getLocation());
        if (subview != null) {
            click.subtract(subview.getLocation());
            subview.mouseUp(click);
        }
    }

    public void objectActionResult(final Naked result, final Location at) {
        getWorkspace().addOpenViewFor(result, at);
    }

    public View pickupContent(final Location location) {
        View subview = subviewFor(location);
        if (subview != null) {
            location.subtract(subview.getLocation());
            return subview.pickupView(location);
        } else {
            return Toolkit.getViewFactory().createDragViewOutline(getView());
        }
    }

    public View pickupView(final Location location) {
        View subview = subviewFor(location);
        if (subview != null) {
            location.subtract(subview.getLocation());
            return subview.pickupView(location);
        } else {
            return null;
        }
    }

    /**
     * Delegates all printing the the draw method.
     * 
     * @see #draw(Canvas)
     */
    public void print(final Canvas canvas) {
        draw(canvas);
    }

    public void refresh() {}

    public void removeView(final View view) {
        throw new NakedObjectRuntimeException();
    }

    protected void replaceOptions(final Enumeration possibleViews, final UserActionSet options) {
        while (possibleViews.hasMoreElements()) {
            ViewSpecification specification = (ViewSpecification) possibleViews.nextElement();

            if (specification != getSpecification() && view.getParent() == view.getWorkspace() && view.getClass() != getClass()) {
                UserAction viewAs = new ReplaceViewOption(specification);
                options.add(viewAs);
            }
        }
    }

    public void replaceView(final View toReplace, final View replacement) {
        throw new NakedObjectRuntimeException();
    }

    public void secondClick(final Click click) {
        View subview = subviewFor(click.getLocation());
        if (subview != null) {
            click.subtract(subview.getLocation());
            subview.secondClick(click);
        }
    }

    public void setBounds(final Bounds bounds) {
        x = bounds.getX();
        y = bounds.getY();
        width = bounds.getWidth();
        height = bounds.getHeight();
    }

    public void setFocusManager(final FocusManager focusManager) {
        throw new UnexpectedCallException();
    }

    protected void setContent(final Content content) {
        this.content = content;
    }

    public void setLocation(final Location location) {
        x = location.getX();
        y = location.getY();
    }

    public final void setParent(final View parentView) {
        LOG.debug("set parent " + parent + " for " + this);
        parent = parentView.getView();
    }

    public void setMaximumSize(final Size size) {}

    public void setSize(final Size size) {
        width = size.getWidth();
        height = size.getHeight();
    }

    protected void setSpecification(final ViewSpecification specification) {
        this.specification = specification;
    }

    public final void setView(View view) {
        this.view = view;
    }

    protected void setViewAxis(final ViewAxis viewAxis) {
        this.viewAxis = viewAxis;
    }

    public View subviewFor(final Location location) {
        return null;
    }

    public void thirdClick(final Click click) {
        View subview = subviewFor(click.getLocation());
        if (subview != null) {
            click.subtract(subview.getLocation());
            subview.thirdClick(click);
        }
    }

    public String toString() {
        String name = getClass().getName();
        return name.substring(name.lastIndexOf('.') + 1) + getId() + ":" + getState() + ":" + getContent();
    }

    public void update(final Naked object) {}

    public void updateView() {}

    public ViewAreaType viewAreaType(final Location location) {
        View subview = subviewFor(location);
        if (subview != null) {
            location.subtract(subview.getLocation());
            return subview.viewAreaType(location);
        } else {
            return ViewAreaType.CONTENT;
        }
    }

    public void viewMenuOptions(final UserActionSet options) {
        options.setColor(Toolkit.getColor("background.view-menu"));

        Content content = getContent();
        if (content != null) {
            content.viewMenuOptions(options);
        }

        if (getParent() != null) {
            Enumeration possibleViews = Toolkit.getViewFactory().openRootViews(content, null);
            while (possibleViews.hasMoreElements()) {
                ViewSpecification specification = (ViewSpecification) possibleViews.nextElement();
                AbstractUserAction viewAs = new OpenViewOption(specification);
                options.add(viewAs);
            }
        }

        if (view.getSpecification() != null && view.getSpecification().isSubView()) {
            if (view.getSpecification().isReplaceable()) {
                replaceOptions(Toolkit.getViewFactory().openSubviews(content, this), options);
                replaceOptions(Toolkit.getViewFactory().closedSubviews(content, this), options);
            }
        } else {
            if (view.getSpecification() != null && view.getSpecification().isReplaceable()) {
                // offer other/alternative views
                replaceOptions(Toolkit.getViewFactory().openRootViews(content, this), options);
            }
            // TODO ask the viewer for the print option - provided by the underlying system
            // options.add(new PrintOption());
            if (getParent() != null) {
                options.add(CLOSE_OPTION);
                options.add(CLOSE_ALL_OPTION);
                options.add(CLOSE_VIEWS_FOR_OBJECT);
            }
        }

        options.add(new AbstractUserAction("Refresh view", UserAction.DEBUG) {
            public void execute(final Workspace workspace, final View view, final Location at) {
                refresh();
            }
        });

        options.add(new AbstractUserAction("Invalidate content", UserAction.DEBUG) {
            public void execute(final Workspace workspace, final View view, final Location at) {
                invalidateContent();
            }
        });

        options.add(new AbstractUserAction("Invalidate layout", UserAction.DEBUG) {
            public void execute(final Workspace workspace, final View view, final Location at) {
                invalidateLayout();
            }
        });

        final UndoStack undoStack = getViewManager().getUndoStack();
        if (!undoStack.isEmpty()) {
            options.add(new AbstractUserAction("Undo " + undoStack.getNameOfUndo()) {

                public Consent disabled(final View component) {
                    return new Allow(undoStack.descriptionOfUndo());
                }

                public void execute(final Workspace workspace, final View view, final Location at) {
                    undoStack.undoLastCommand();
                }
            });
        }
    }
}
// Copyright (c) Naked Objects Group Ltd.
