package org.nakedobjects.nos.client.dnd.view.text;

import junit.framework.TestCase;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.nakedobjects.nof.core.conf.PropertiesConfiguration;
import org.nakedobjects.nof.core.context.NakedObjectsContext;


public class MultilineTextFieldContentTest extends TestCase {

    public static void main(final String[] args) {
        junit.textui.TestRunner.run(MultilineTextFieldContentTest.class);
    }

    private TextContent content;

    protected void setUp() throws Exception {
        Logger.getRootLogger().setLevel(Level.OFF);

        NakedObjectsContext.setConfiguration(new PropertiesConfiguration());

        TextBlockTarget target = new TextBlockTargetExample();

        content = new TextContent(target, 4, TextContent.WRAPPING);
        content.setText("Line one\nLine two\nLine three\nLine four that is long enough that it wraps");
    }

    public void testDeleteOnSingleLine() {
        TextSelection selection = new TextSelection(content);
        selection.resetTo(new CursorPosition(content, 1, 3));
        selection.extendTo(new CursorPosition(content, 1, 7));
        content.delete(selection);
        assertEquals("Line one\nLino\nLine three\nLine four that is long enough that it wraps", content.getText());
    }

    public void testDeleteOnSingleLineWithStartAndEndOutOfOrder() {
        TextSelection selection = new TextSelection(content);
        selection.resetTo(new CursorPosition(content, 1, 7));
        selection.extendTo(new CursorPosition(content, 1, 3));
        content.delete(selection);
        assertEquals("Line one\nLino\nLine three\nLine four that is long enough that it wraps", content.getText());
    }
    
    public void testDeleteAcrossTwoLines() {
        TextSelection selection = new TextSelection(content);
        selection.resetTo(new CursorPosition(content, 0, 6));
        selection.extendTo(new CursorPosition(content, 1, 6));
        content.delete(selection);
        assertEquals("Line owo\nLine three\nLine four that is long enough that it wraps", content.getText());
    }

    public void testDeleteAcrossThreeLines() {
        TextSelection selection = new TextSelection(content);
        selection.resetTo(new CursorPosition(content, 0, 6));
        selection.extendTo(new CursorPosition(content, 2, 6));
        content.delete(selection);
        assertEquals("Line ohree\nLine four that is long enough that it wraps", content.getText());
    }

    public void testDeleteAcrossThreeLinesIncludingWrappedBlock() {
        TextSelection selection = new TextSelection(content);
        selection.resetTo(new CursorPosition(content, 2, 5));
        selection.extendTo(new CursorPosition(content, 4, 5));
        content.delete(selection);
        assertEquals("Line one\nLine two\nLine enough that it wraps", content.getText());
    }

    public void testDeleteWithinWrappedBlock() {
        TextSelection selection = new TextSelection(content);
        selection.resetTo(new CursorPosition(content, 5, 0));
        selection.extendTo(new CursorPosition(content, 5, 3));
        content.delete(selection);
        assertEquals("Line one\nLine two\nLine three\nLine four that is long enough that wraps", content.getText());
    }

    public void testDeleteMultipleLinesWithinWrappedBlock() {
        TextSelection selection = new TextSelection(content);
        selection.resetTo(new CursorPosition(content, 3, 5));
        selection.extendTo(new CursorPosition(content, 5, 3));
        content.delete(selection);
        assertEquals("Line one\nLine two\nLine three\nLine wraps", content.getText());
    }

    public void testLineBreaks() {
        assertEquals(6, content.getNoLinesOfContent());

        content.setNoDisplayLines(8);
        String[] lines = content.getDisplayLines();

        assertEquals(8, lines.length);
        assertEquals("Line one", lines[0]);
        assertEquals("Line two", lines[1]);
        assertEquals("Line three", lines[2]);
        assertEquals("Line four that is ", lines[3]);
        assertEquals("long enough that ", lines[4]);
        assertEquals("it wraps", lines[5]);
        assertEquals("", lines[6]);
        assertEquals("", lines[7]);

    }

}
// Copyright (c) Naked Objects Group Ltd.
