package org.nakedobjects.plugins.dndviewer.viewer.border;

import org.nakedobjects.metamodel.adapter.NakedObject;
import org.nakedobjects.metamodel.consent.Consent;
import org.nakedobjects.metamodel.consent.Veto;
import org.nakedobjects.plugins.dndviewer.Content;
import org.nakedobjects.plugins.dndviewer.ContentDrag;
import org.nakedobjects.plugins.dndviewer.FieldContent;
import org.nakedobjects.plugins.dndviewer.InternalDrag;
import org.nakedobjects.plugins.dndviewer.ObjectContent;
import org.nakedobjects.plugins.dndviewer.ParameterContent;
import org.nakedobjects.plugins.dndviewer.Toolkit;
import org.nakedobjects.plugins.dndviewer.View;
import org.nakedobjects.plugins.dndviewer.ViewAreaType;
import org.nakedobjects.plugins.dndviewer.ViewState;
import org.nakedobjects.plugins.dndviewer.viewer.drawing.Color;
import org.nakedobjects.plugins.dndviewer.viewer.drawing.Location;
import org.nakedobjects.plugins.dndviewer.viewer.drawing.Text;


public class DroppableLabelBorder extends LabelBorder {

    public static View createObjectFieldLabelBorder(final View view) {
        final FieldContent fieldContent = (FieldContent) view.getContent();
        return createDroppableLabelBorder(fieldContent.getFieldName(), fieldContent.isMandatory(), view);
    }

    public static View createDroppableLabelBorder(final String name, final boolean mandatory, final View view) {
        final Text style = mandatory ? Toolkit.getText("mandatory-label") : Toolkit.getText("label");
        return new DroppableLabelBorder(name, style, view);
    }

    public static View createObjectParameterLabelBorder(final View wrappedView) {
        final ParameterContent parameterContent = (ParameterContent) wrappedView.getContent();
        return createDroppableLabelBorder(parameterContent.getParameterName(), parameterContent.isRequired(), wrappedView);
    }

    private final ViewState labelState = new ViewState();
    private boolean overContent;

    protected DroppableLabelBorder(final String label, final Text style, final View view) {
        super(label, style, view);
    }

    @Override
    public ViewAreaType viewAreaType(final Location location) {
        if (overBorder(location)) {
            return ViewAreaType.CONTENT; // used to ensure menu options for contained object are shown
        } else {
            return super.viewAreaType(location);
        }
    }

    @Override
    public void dragCancel(final InternalDrag drag) {
        super.dragCancel(drag);
        labelState.clearViewIdentified();
    }

    @Override
    public void drag(final ContentDrag drag) {
        final Location targetLocation = drag.getTargetLocation();
        if (overContent(targetLocation) && overContent == false) {
            overContent = true;
            super.dragIn(drag);
            dragOutOfLabel();
        } else if (overBorder(targetLocation) && overContent == true) {
            overContent = false;
            super.dragOut(drag);
            dragInToLabel(drag.getSourceContent());
        }

        super.drag(drag);
    }

    @Override
    public void dragIn(final ContentDrag drag) {
        if (overContent(drag.getTargetLocation())) {
            super.dragIn(drag);
        } else {
            final Content sourceContent = drag.getSourceContent();
            dragInToLabel(sourceContent);
            markDamaged();
        }
    }

    private void dragInToLabel(final Content sourceContent) {
        overContent = false;
        final Consent canDrop = canDrop(sourceContent);
        if (canDrop.isAllowed()) {
            labelState.setCanDrop();
        } else {
            labelState.setCantDrop();
        }
        String actionText = canDrop.isVetoed() ? 
                            canDrop.getReason() : "Set to " + sourceContent.title();
        getFeedbackManager().setAction(actionText);
    }

    @Override
    public void dragOut(final ContentDrag drag) {
        super.dragOut(drag);
        dragOutOfLabel();
    }

    private void dragOutOfLabel() {
        labelState.clearObjectIdentified();
        markDamaged();
    }

    @Override
    public void drop(final ContentDrag drag) {
        if (overContent(drag.getTargetLocation())) {
            super.drop(drag);
        } else {
            dragOutOfLabel();
            final Content sourceContent = drag.getSourceContent();
            if (canDrop(sourceContent).isAllowed()) {
                drop(sourceContent);
            }
        }
    }

    protected Consent canDrop(final Content dropContent) {
        if (dropContent instanceof ObjectContent) {
            final NakedObject source = ((ObjectContent) dropContent).getObject();
            final ObjectContent content = (ObjectContent) getContent();
            return content.canSet(source);
        } else {
            return Veto.DEFAULT;
        }
    }

    protected void drop(final Content dropContent) {
        if (dropContent instanceof ObjectContent) {
            final NakedObject object = ((ObjectContent) dropContent).getObject();
            ((ObjectContent) getContent()).setObject(object);
            getParent().invalidateContent();
        }
    }

    @Override
    protected Color textColor() {
        Color color;
        if (labelState.canDrop()) {
            color = Toolkit.getColor("valid");
        } else if (labelState.cantDrop()) {
            color = Toolkit.getColor("invalid");
        } else {
            color = DEFAULT_COLOR;
        }
        return color;
    }
}

// Copyright (c) Naked Objects Group Ltd.
