package org.nakedobjects.plugins.dndviewer.viewer.view.field;

import org.nakedobjects.metamodel.adapter.NakedObject;
import org.nakedobjects.metamodel.commons.exceptions.NotYetImplementedException;
import org.nakedobjects.metamodel.consent.Consent;
import org.nakedobjects.plugins.dndviewer.BackgroundTask;
import org.nakedobjects.plugins.dndviewer.BackgroundThread;
import org.nakedobjects.plugins.dndviewer.Canvas;
import org.nakedobjects.plugins.dndviewer.Content;
import org.nakedobjects.plugins.dndviewer.InternalDrag;
import org.nakedobjects.plugins.dndviewer.KeyboardAction;
import org.nakedobjects.plugins.dndviewer.TextParseableContent;
import org.nakedobjects.plugins.dndviewer.Toolkit;
import org.nakedobjects.plugins.dndviewer.UserActionSet;
import org.nakedobjects.plugins.dndviewer.View;
import org.nakedobjects.plugins.dndviewer.ViewAreaType;
import org.nakedobjects.plugins.dndviewer.ViewAxis;
import org.nakedobjects.plugins.dndviewer.ViewSpecification;
import org.nakedobjects.plugins.dndviewer.viewer.drawing.Location;
import org.nakedobjects.plugins.dndviewer.viewer.drawing.Padding;
import org.nakedobjects.plugins.dndviewer.viewer.view.simple.AbstractView;


public abstract class AbstractField extends AbstractView {
    protected static final int TEXT_WIDTH = 20;
    private boolean identified;

    protected AbstractField(final Content content, final ViewSpecification design, final ViewAxis axis) {
        super(content, design, axis);
    }

    @Override
    public boolean canFocus() {
        return canChangeValue().isAllowed();
    }

    void clear() {}

    void copyToClipboard() {}

    @Override
    public Consent canChangeValue() {
        final TextParseableContent cont = (TextParseableContent) getContent();
        return cont.isEditable();
    }

    /**
     * Indicates the drag started within this view's bounds is continuing. By default does nothing.
     */
    @Override
    public void drag(final InternalDrag drag) {}

    /**
     * Default implementation - does nothing
     */
    @Override
    public void dragCancel(final InternalDrag drag) {}

    /**
     * Indicates the start of a drag within this view's bounds. By default does nothing.
     */
    @Override
    public View dragFrom(final Location location) {
        return null;
    }

    /**
     * Indicates the drag started within this view's bounds has been finished (although the location may now
     * be outside of its bounds). By default does nothing.
     */
    @Override
    public void dragTo(final InternalDrag drag) {}

    @Override
    public void draw(final Canvas canvas) {
        if (getState().isActive()) {
            canvas.clearBackground(this, Toolkit.getColor("identified"));
        }

        if (getState().isOutOfSynch()) {
            canvas.clearBackground(this, Toolkit.getColor("out-of-sync"));
        }

        if (getState().isInvalid()) {
            canvas.clearBackground(this, Toolkit.getColor("invalid"));
        }

        super.draw(canvas);
    }

    /**
     * Indicates that editing has been completed and the entry should be saved. Will be called by the view
     * manager when other action place within the parent.
     * 
     */
    @Override
    public void editComplete(boolean moveToNextField) {}

    @Override
    public void entered() {
        super.entered();
        identified = true;
        final Consent changable = canChangeValue();
        if (changable.isVetoed()) {
            getFeedbackManager().setViewDetail(changable.getReason());
        }
        markDamaged();
    }

    @Override
    public void exited() {
        super.exited();
        identified = false;
        markDamaged();
    }

    public boolean getIdentified() {
        return identified;
    }

    @Override
    public Padding getPadding() {
        return new Padding(0, 0, 0, 0);
    }

    public View getRoot() {
        throw new NotYetImplementedException();
    }

    String getSelectedText() {
        return "";
    }

    @Override
    public boolean hasFocus() {
        return getViewManager().hasFocus(getView());
    }

    public boolean indicatesForView(final Location mouseLocation) {
        return false;
    }

    /**
     * Called when the user presses any key on the keyboard while this view has the focus.
     */
    @Override
    public void keyPressed(final KeyboardAction key) {}

    /**
     * Called when the user releases any key on the keyboard while this view has the focus.
     * 
     * @param keyCode
     * @param modifiers
     */
    @Override
    public void keyReleased(final int keyCode, final int modifiers) {}

    /**
     * Called when the user presses a non-control key (i.e. data entry keys and not shift, up-arrow etc). Such
     * a key press will result in a prior call to <code>keyPressed</code> and a subsequent call to
     * <code>keyReleased</code>.
     */
    @Override
    public void keyTyped(final char keyCode) {}

    @Override
    public void contentMenuOptions(final UserActionSet options) {
        options.add(new CopyValueOption(this));
        options.add(new PasteValueOption(this));
        options.add(new ClearValueOption(this));

        if (getView().getSpecification().isReplaceable()) {
            replaceOptions(Toolkit.getViewFactory().valueViews(getContent(), this), options);
        }

        super.contentMenuOptions((options));
        options.setColor(Toolkit.getColor("background.value-menu"));
    }

    protected final void initiateSave(final boolean moveToNextField) {
        BackgroundThread.run(this, new BackgroundTask() {
            public void execute() {
                save();
                getParent().updateView();
                invalidateLayout();
                if (moveToNextField) {
                    getFocusManager().focusNextView();
                }
            }

            public String getName() {
                return "Save field";
            }

            public String getDescription() {
                return "Saving " + getContent().windowTitle();
            }

        });
    }

    protected abstract void save();

    protected void saveValue(final NakedObject value) {
        parseEntry(value.titleString());
    }

    protected void parseEntry(final String entryText) {
        final TextParseableContent content = (TextParseableContent) getContent();
        content.parseTextEntry(entryText);
        content.entryComplete();
    }

    void pasteFromClipboard() {}

    @Override
    public String toString() {
        final String cls = getClass().getName();
        final NakedObject nakedObject = getContent().getNaked();
        return cls.substring(cls.lastIndexOf('.') + 1) + getId() + " [location=" + getLocation() + ",object="
                + (nakedObject == null ? "" : nakedObject.getObject()) + "]";
    }

    @Override
    public ViewAreaType viewAreaType(final Location mouseLocation) {
        return ViewAreaType.INTERNAL;
    }

    @Override
    public int getBaseline() {
        return Toolkit.defaultBaseline();
    }
}
// Copyright (c) Naked Objects Group Ltd.
