package org.nakedobjects.plugins.dndviewer.viewer.view.field;

import java.awt.event.KeyEvent;

import org.nakedobjects.plugins.dndviewer.Canvas;
import org.nakedobjects.plugins.dndviewer.Content;
import org.nakedobjects.plugins.dndviewer.KeyboardAction;
import org.nakedobjects.plugins.dndviewer.TextParseableContent;
import org.nakedobjects.plugins.dndviewer.Toolkit;
import org.nakedobjects.plugins.dndviewer.UserActionSet;
import org.nakedobjects.plugins.dndviewer.ViewAxis;
import org.nakedobjects.plugins.dndviewer.ViewSpecification;
import org.nakedobjects.plugins.dndviewer.viewer.drawing.Color;
import org.nakedobjects.plugins.dndviewer.viewer.drawing.Size;
import org.nakedobjects.plugins.dndviewer.viewer.drawing.Text;
import org.nakedobjects.plugins.dndviewer.viewer.util.Properties;
import org.nakedobjects.runtime.context.NakedObjectsContext;


public class PasswordField extends AbstractField {
    protected static final Text style = Toolkit.getText("normal");
    private boolean isSaved;
    private int maxTextWidth;
    private String password;
    private boolean identified;
    private char echoCharacter;

    public PasswordField(final Content content, final ViewSpecification design, final ViewAxis axis) {
        super(content, design, axis);
        setMaxTextWidth(TEXT_WIDTH);
        final String echoCharacterSetting = NakedObjectsContext.getConfiguration().getString(Properties.PROPERTY_BASE + "echo");
        if (echoCharacterSetting == null || echoCharacterSetting.equals(" ")) {
            echoCharacter = '*';
        } else {
            echoCharacter = echoCharacterSetting.charAt(0);
        }
        password = text();
    }

    @Override
    public void contentMenuOptions(final UserActionSet options) {
        options.add(new ClearValueOption(this));
        options.setColor(Toolkit.getColor("background.value-menu"));
    }

    @Override
    void clear() {
        password = "";
        editComplete(false);
    }

    private void delete() {
        isSaved = false;
        final int lastChar = password.length() - 1;
        password = password.substring(0, Math.max(0, lastChar));
        markDamaged();
    }

    @Override
    public void draw(final Canvas canvas) {
        super.draw(canvas);

        Color color = identified ? Toolkit.getColor("identified") : Toolkit.getColor("secondary2");
        color = hasFocus() ? Toolkit.getColor("primary1") : color;
        final int baseline = getBaseline();
        canvas.drawLine(HPADDING, baseline, HPADDING + getSize().getWidth(), baseline, color);

        final String echoedPassword = echoPassword();
        int x = 3;
        final Color textColor = hasFocus() ? Toolkit.getColor("text.edit") : Toolkit.getColor("text.saved");
        canvas.drawText(echoedPassword, x, getBaseline(), textColor, style);
        x += style.stringWidth(echoedPassword);
        if (hasFocus() && canChangeValue().isAllowed()) {
            canvas.drawLine(x, (baseline + style.getDescent()), x, 0, Toolkit.getColor("primary1"));
        }
    }

    private String echoPassword() {
        final int length = password.length();
        String echoedPassword = "";
        for (int i = 0; i < length; i++) {
            echoedPassword += echoCharacter;
        }
        return echoedPassword;
    }

    @Override
    public void editComplete(boolean moveToNextField) {
        if (canChangeValue().isAllowed() && !isSaved) {
            isSaved = true;
            initiateSave(moveToNextField);
        }
    }

    public void escape() {
        password = "";
        isSaved = false;
        markDamaged();
    }

    @Override
    public void entered() {
        if (canChangeValue().isAllowed()) {
            getFeedbackManager().showTextCursor();
            identified = true;
            markDamaged();
        }
    }

    @Override
    public void exited() {
        if (canChangeValue().isAllowed()) {
            getFeedbackManager().showDefaultCursor();
            identified = false;
            markDamaged();
        }
    }

    @Override
    public void focusLost() {
        editComplete(false);
    }

    @Override
    public Size getMaximumSize() {
        final int width = HPADDING + maxTextWidth + HPADDING;
        int height = style.getTextHeight() + VPADDING;
        height = Math.max(height, Toolkit.defaultFieldHeight());

        return new Size(width, height);
    }

    @Override
    public void keyPressed(final KeyboardAction key) {
        if (!canChangeValue().isAllowed()) {
            return;
        }

        final int keyCode = key.getKeyCode();
        if (keyCode == KeyEvent.VK_CONTROL || keyCode == KeyEvent.VK_SHIFT || keyCode == KeyEvent.VK_ALT) {
            return;
        }

        switch (keyCode) {
        case KeyEvent.VK_LEFT:
            key.consume();
            delete();
            break;
        case KeyEvent.VK_DELETE:
            key.consume();
            delete();
            break;
        case KeyEvent.VK_BACK_SPACE:
            key.consume();
            delete();
            break;
        case KeyEvent.VK_TAB:
            editComplete(false);
            break;
        case KeyEvent.VK_ENTER:
            key.consume();
            editComplete(false);
            getParent().keyPressed(key);
            break;
        case KeyEvent.VK_ESCAPE:
            key.consume();
            escape();
            break;

        default:
            break;
        }
    }

    @Override
    public void keyTyped(final char keyCode) {
        password += keyCode;
        isSaved = false;
        markDamaged();
    }

    @Override
    protected void save() {
        final TextParseableContent content = (TextParseableContent) getContent();
        content.parseTextEntry(password);
        content.entryComplete();
    }

    /**
     * Set the maximum width of the field, as a number of characters
     */
    public void setMaxTextWidth(final int noCharacters) {
        maxTextWidth = style.charWidth('o') * noCharacters;
    }

    /**
     * Set the maximum width of the field, as a number of pixels
     */
    public void setMaxWidth(final int width) {
        maxTextWidth = width;
    }

    private String text() {
        String text;
        final TextParseableContent content = (TextParseableContent) getContent();
        text = content.getNaked().titleString();
        return text;
    }

}
// Copyright (c) Naked Objects Group Ltd.
