package org.nakedobjects.plugins.hibernate.authorization;

import org.apache.log4j.Logger;
import org.hibernate.SQLQuery;
import org.nakedobjects.applib.Identifier;
import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.metamodel.config.ConfigurationConstants;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.plugins.hibernate.objectstore.util.HibernateUtil;
import org.nakedobjects.runtime.authorization.standard.Authorizor;


public class DatabaseAuthorizor implements Authorizor {
    @SuppressWarnings("unused")
    private static final Logger LOG = Logger.getLogger(DatabaseAuthorizor.class);
    private final boolean learn;
    private static final String AUTH_LEARN = ConfigurationConstants.ROOT + "security.learn";

    private static final int READ_WRITE = 0;
    @SuppressWarnings("unused")
    private static final int READ_ONLY = 1;

    private static final String NO_FLAG_MATCH = "select count(p.permission) from permissions p, role r where p.role = r.id  and r.rolename = ? and p.permission = ?";
    private static final String FLAG_MATCH = NO_FLAG_MATCH + " and p.flags is NULL or p.flags = ?";

    private static final String NO_FLAG_INSERT = "insert into permissions values (?, ?, ?)";
    private static final String FLAG_INSERT = "insert into permissions values (?, ?, ?, ?)";

    public DatabaseAuthorizor(NakedObjectConfiguration configuration) {
        learn = configuration.getBoolean(AUTH_LEARN, false);
    }

    public void init() {
    // do nothing
    }

    private int getNextId() {
        try {
            HibernateUtil.startTransaction();
            final SQLQuery sq = HibernateUtil.getCurrentSession().createSQLQuery(
                    "select id from permissions order by id desc limit 1");
            final Integer id = (Integer) sq.uniqueResult();
            HibernateUtil.commitTransaction();
            return id != null ? id.intValue() + 1 : 0;
        } catch (final Exception e) {
            HibernateUtil.rollbackTransaction();
            throw new NakedObjectException(e);
        }
    }

    private int getRoleId(final String role) {
        try {
            HibernateUtil.startTransaction();
            final SQLQuery sq = HibernateUtil.getCurrentSession().createSQLQuery("select id from role  where rolename = ?");
            sq.setString(0, role);
            final Integer id = (Integer) sq.uniqueResult();
            HibernateUtil.commitTransaction();
            return id.intValue();
        } catch (final Exception e) {
            HibernateUtil.rollbackTransaction();
            throw new NakedObjectException(e);
        }
    }

    private boolean learn(final String role, final String key, final Integer flag) {
        if (isMatch(role, key, flag)) {
            return true;
        }
        try {
            final int permissionId = getNextId();
            final int roleId = getRoleId(role);

            HibernateUtil.startTransaction();
            final SQLQuery sq = HibernateUtil.getCurrentSession().createSQLQuery(flag == null ? NO_FLAG_INSERT : FLAG_INSERT);
            sq.setInteger(0, permissionId);
            sq.setInteger(1, roleId);
            sq.setString(2, key);
            if (flag != null) {
                sq.setInteger(3, flag.intValue());
            }
            sq.executeUpdate();
            HibernateUtil.commitTransaction();
        } catch (final Exception e) {
            HibernateUtil.rollbackTransaction();
            throw new NakedObjectException(e);
        }
        return true;
    }

    private boolean isMatch(final String role, final String key, final Integer flag) {
        try {
            HibernateUtil.startTransaction();
            final SQLQuery sq = HibernateUtil.getCurrentSession().createSQLQuery(flag == null ? NO_FLAG_MATCH : FLAG_MATCH);
            sq.setString(0, role);
            sq.setString(1, key);
            if (flag != null) {
                sq.setInteger(2, flag.intValue());
            }
            final Number count = (Number) sq.uniqueResult();
            HibernateUtil.commitTransaction();
            return count != null && count.intValue() > 0;
        } catch (final Exception e) {
            HibernateUtil.rollbackTransaction();
            throw new NakedObjectException(e);
        }
    }

    public void shutdown() {
    // do nothing
    }

    public boolean isAuthorised(final String role, final Identifier member, final Integer flag) {
        if (learn) {
            return learn(role, member.toIdentityString(Identifier.CLASS_NAME_PARMS), flag);
        }

        for (int depth = Identifier.CLASS; depth <= Identifier.CLASS_NAME_PARMS; depth++) {
            if (isMatch(role, member.toIdentityString(depth), flag)) {
                return true;
            }
        }
        return false;
    }

    public boolean isUsable(final String role, final Identifier member) {
        // only match if flag is either null or READ_WRITE
        return isAuthorised(role, member, Integer.valueOf(READ_WRITE));
    }

    public boolean isVisible(final String role, final Identifier member) {
        // match ignoring flag
        return isAuthorised(role, member, null);
    }
}
