package org.nakedobjects.plugins.htmlviewer.webapp;

import java.io.File;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.config.loader.ConfigurationLoaderDefault;
import org.nakedobjects.metamodel.config.loader.NotFoundPolicy;
import org.nakedobjects.plugins.htmlviewer.component.ImageLookup;
import org.nakedobjects.plugins.htmlviewer.servlet.SessionAccess;
import org.nakedobjects.runtime.context.NakedObjectsContext;
import org.nakedobjects.runtime.installers.InstallerLookupImpl;
import org.nakedobjects.runtime.system.DeploymentType;
import org.nakedobjects.runtime.system.NakedObjectsSystem;
import org.nakedobjects.runtime.system.installers.NakedObjectsSystemUsingInstallers;


/**
 * Initialize the {@link NakedObjectsSystem} when the web application starts, 
 * and destroys it when it ends.
 * 
 * <p>
 * Implementation note: we use a number of helper builders to keep this class as small and
 * focused as possible.  The builders are available for reuse by other bootstrappers.
 */
public class ServletInitializer implements ServletContextListener {

    private static final String SYSTEM_SERVLET_CONTEXT_KEY = "System";

    private static final Logger LOG = Logger.getLogger(ServletInitializer.class);

    public void contextInitialized(final ServletContextEvent servletContextEvent) {
        final ServletInitializer instance = new ServletInitializer();
        instance.init(servletContextEvent.getServletContext());
    }

    private void init(final ServletContext servletContext) {
        
        ImageLookup.setImageDirectory(servletContext.getRealPath("/images"));
        final DeploymentType deploymentType = DeploymentType.SERVER;
        
        final InstallerLookupImpl installerLookup = new InstallerLookupImpl(getClass());
        
        final String dir = servletContext.getRealPath("/WEB-INF");
        if (LOG.isDebugEnabled()) {
            LOG.debug("current dir: " + new File(".").getAbsolutePath());
            LOG.debug("real path: " + dir);
        }
        
        final ConfigurationLoaderDefault configurationLoader = new ConfigurationLoaderDefault(dir);
        addConfigurationFiles(deploymentType, configurationLoader);
        
        installerLookup.setConfigurationLoader(configurationLoader);


        // TODO may still need to work out how to set up password authenticator with path to config files
        // system.setAuthenticatorInstaller(new PasswordFileAuthenticatorInstaller(new File(dir,"passwords")));


        final NakedObjectsSystemUsingInstallers system =  
            new NakedObjectsSystemUsingInstallersWithinWebApp(deploymentType, installerLookup, servletContext);
        system.setHideSplash(!false);
        
        system.lookupAndSetAuthenticatorInstaller();
        system.lookupAndSetUserProfileFactoryInstaller();
        system.lookupAndSetServerListenerInstaller();
        system.lookupAndSetFixturesInstaller();
        system.init();

        // specific to this bootstrap mechanism
        completeServletInitialization(servletContext, system);
    }

    private void completeServletInitialization(final ServletContext servletContext, final NakedObjectsSystemUsingInstallers system) {
        SessionAccess.createInstance(NakedObjectsContext.getAuthenticationManager());

        LOG.info("server started");

        servletContext.setAttribute(SYSTEM_SERVLET_CONTEXT_KEY, system);
    }

    private void addConfigurationFiles(final DeploymentType deploymentType, final ConfigurationLoaderDefault configurationLoader) {
        for(String config: (new String[] {"web.properties", "war.properties"})) {
            if (config != null) {
                configurationLoader.addConfigurationFile(config, NotFoundPolicy.FAIL_FAST);
            }
        }
        String type = deploymentType.name().toLowerCase();
        configurationLoader.addConfigurationFile(type + ".properties", NotFoundPolicy.CONTINUE);
    }


    public void contextDestroyed(final ServletContextEvent ev) {
        LOG.info("server shutting down");
        final NakedObjectsSystemUsingInstallers system = (NakedObjectsSystemUsingInstallers) ev.getServletContext().getAttribute(SYSTEM_SERVLET_CONTEXT_KEY);
        if (system != null) {
            LOG.info("calling system shutdown");
            system.shutdown();
        }
        ev.getServletContext().removeAttribute(SYSTEM_SERVLET_CONTEXT_KEY);
    }


    /*
     * private void ejb_init(ServletContext servletContext) { LOG.info("creating naked objects server");
     * NakedObjectsSystem system; system = new NakedObjectsSystem(); system.addAuthenticator(new
     * PasswordFileAuthenticator(new File("passwords"))); system.loadConfiguration(new
     * ResourceConfigurationLoader()); system.setContext(ThreadContext.createInstance()); // TODO: get this in
     * line with NakedObjects String clientConnection =
     * NakedObjectsContext.getConfiguration().getString("nakedobjects.connection"); if (clientConnection !=
     * null) { ClientConnection connection; Class clazz = null; try { // try 1. Get from the application
     * classpath ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader(); clazz =
     * contextClassLoader.loadClass(clientConnection); } catch (Exception e) { throw new
     * NakedObjectRuntimeException(e); } if (clazz == null) { // try 2. if that fails try this classes
     * classloader try { clazz = Class.forName(clientConnection); } catch (ClassNotFoundException e) { throw
     * new NakedObjectRuntimeException(e); } } try { connection = (ClientConnection) clazz.newInstance(); }
     * catch (Exception e) { throw new NakedObjectRuntimeException(e); }
     * system.setObjectPersistorFactory(connection); system.addReflectivePeer(new TransactionPeerFactory()); }
     * 
     * system.disableSplash(true); // system.setPersistorPerContext(true); system.init();
     * 
     * system.showConsole();
     * 
     * ImageLookup.setImageDirectory(servletContext.getRealPath("/images"));
     * 
     * new SystemAccess(system);
     * 
     * LOG.info("server started"); }
     */



}

// Copyright (c) Naked Objects Group Ltd.
