package org.nakedobjects.plugins.htmlviewer.servlet;

import java.io.IOException;
import java.io.PrintWriter;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.config.ConfigurationConstants;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.plugins.htmlviewer.WebController;
import org.nakedobjects.plugins.htmlviewer.component.Page;
import org.nakedobjects.plugins.htmlviewer.html.HtmlComponentFactory;
import org.nakedobjects.plugins.htmlviewer.request.Context;
import org.nakedobjects.runtime.context.NakedObjectsContext;
import org.nakedobjects.plugins.htmlviewer.request.Request;
import org.nakedobjects.plugins.htmlviewer.request.Task;



public class Controller extends HttpServlet {
    private static final long serialVersionUID = 1L;
    private final Logger LOG = Logger.getLogger(Controller.class);
    private static final String PROPERTY_BASE = ConfigurationConstants.ROOT + "viewer.html.";
    private WebController controller;
    private static String encoding = "ISO-8859-1";

    @Override
    protected void doPost(final HttpServletRequest request, final HttpServletResponse response) throws ServletException,
            IOException {
        request.setCharacterEncoding(encoding);
        processRequest(request, response);
    }

    @Override
    protected void doGet(final HttpServletRequest request, final HttpServletResponse response) throws ServletException,
            IOException {
        processRequest(request, response);
    }

    private void processRequest(final HttpServletRequest request, final HttpServletResponse response) throws ServletException,
            IOException {
        LOG.info("request: " + request.getServletPath() + "?" + request.getQueryString());

        final Request req = new ServletRequest(request);

        if (req.getRequestType() == null) {
            throw new ServletException("No action specified");
        } else if (!controller.actionExists(req)) {
            throw new ServletException("No such action " + req.getRequestType());
        } else {
            try {
                final Context context = getContextForRequest(request);
                processRequest(request, response, req, context);
            } catch (final Exception e) {
                LOG.error("exception during request handling", e);
                throw new ServletException("Internal exception", e);
            }
        }
    }

    private void processRequest(
            final HttpServletRequest request,
            final HttpServletResponse response,
            final Request req,
            final Context context) throws IOException, ServletException {
        if (context.isLoggedIn()) {
            response.setContentType("text/html");

            try {
                SessionAccess.startRequest(context.getSession());
                final Page page = controller.generatePage(context, req);
                if (context.isValid()) {
                    if (controller.isDebug()) {
                        controller.addDebug(page, req);
                        addDebug(request, page);
                    }
                    PrintWriter writer;
                    writer = response.getWriter();
                    page.write(writer);
                } else {
                    response.sendRedirect("logon.app");
                }
            } finally {
                SessionAccess.endRequest(context.getSession());
                if (!context.isLoggedIn()) {
                    final HttpSession httpSession = request.getSession(false);
                    LOG.info("dropping session: " + httpSession);
                }
            }
        } else {
            response.sendRedirect("logon.app");
        }
    }

    private Context getContextForRequest(final HttpServletRequest request) {
        Context context = (Context) request.getSession(true).getAttribute("nof-context");
        if (context == null || !context.isValid()) {
            // TODO reuse the component factory
            context = new Context(new HtmlComponentFactory());
            request.getSession(true).setAttribute("nof-context", context);
        }
        return context;
    }

    private void addDebug(final HttpServletRequest request, final Page page) {
        page.addDebug("Servlet path", request.getServletPath());
        page.addDebug("Query string", request.getQueryString());
        page.addDebug("Context path", request.getContextPath());
        page.addDebug("Path info", request.getPathInfo());
    }

    @Override
    public void init(final ServletConfig arg0) throws ServletException {
        super.init(arg0);
        controller = new WebController();
        final NakedObjectConfiguration configuration = NakedObjectsContext.getConfiguration();
        controller.setDebug(configuration.getBoolean(PROPERTY_BASE + "debug"));
        controller.init();
        encoding = configuration.getString(PROPERTY_BASE + "encoding", encoding);
    }
}

// Copyright (c) Naked Objects Group Ltd.
