package org.nakedobjects.plugins.remoting.command.client;

import java.util.Properties;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.adapter.oid.Oid;
import org.nakedobjects.metamodel.authentication.AuthenticationSession;
import org.nakedobjects.plugins.remoting.command.shared.requests.Authenticate;
import org.nakedobjects.plugins.remoting.command.shared.requests.AuthoriseUsability;
import org.nakedobjects.plugins.remoting.command.shared.requests.AuthoriseVisibility;
import org.nakedobjects.plugins.remoting.command.shared.requests.ClearAssociation;
import org.nakedobjects.plugins.remoting.command.shared.requests.ClearValue;
import org.nakedobjects.plugins.remoting.command.shared.requests.CloseSession;
import org.nakedobjects.plugins.remoting.command.shared.requests.ExecuteClientAction;
import org.nakedobjects.plugins.remoting.command.shared.requests.ExecuteServerAction;
import org.nakedobjects.plugins.remoting.command.shared.requests.FindInstances;
import org.nakedobjects.plugins.remoting.command.shared.requests.GetObject;
import org.nakedobjects.plugins.remoting.command.shared.requests.GetProperties;
import org.nakedobjects.plugins.remoting.command.shared.requests.HasInstances;
import org.nakedobjects.plugins.remoting.command.shared.requests.OidForService;
import org.nakedobjects.plugins.remoting.command.shared.requests.Request;
import org.nakedobjects.plugins.remoting.command.shared.requests.Resolve;
import org.nakedobjects.plugins.remoting.command.shared.requests.ResolveField;
import org.nakedobjects.plugins.remoting.command.shared.requests.SetAssociation;
import org.nakedobjects.plugins.remoting.command.shared.requests.SetValue;
import org.nakedobjects.plugins.remoting.shared.NakedObjectsRemoteException;
import org.nakedobjects.plugins.remoting.shared.ServerFacade;
import org.nakedobjects.plugins.remoting.shared.data.ClientActionResultData;
import org.nakedobjects.plugins.remoting.shared.data.CriteriaData;
import org.nakedobjects.plugins.remoting.shared.data.Data;
import org.nakedobjects.plugins.remoting.shared.data.EncodeableObjectData;
import org.nakedobjects.plugins.remoting.shared.data.IdentityData;
import org.nakedobjects.plugins.remoting.shared.data.ObjectData;
import org.nakedobjects.plugins.remoting.shared.data.ReferenceData;
import org.nakedobjects.plugins.remoting.shared.data.ServerActionResultData;
import org.nakedobjects.plugins.remoting.command.shared.requests.Response;

/**
 * previously called <tt>ClientConnection</tt>.
 */
public class ServerFacadeProxy implements ServerFacade {
    
    private static final Logger LOG = Logger.getLogger(ServerFacadeProxy.class);
    
    /**
     * Injected, see {@link #setConnection(CommandClientConnection).
     */
    private CommandClientConnection connection;

    
    ///////////////////////////////////////////////////////////////////
    // init, shutdown
    ///////////////////////////////////////////////////////////////////

    public void init() {
        connection.init();
    }


    public void shutdown() {
        connection.shutdown();
    }

    ///////////////////////////////////////////////////////////////////
    // session
    ///////////////////////////////////////////////////////////////////

    public void closeSession(final AuthenticationSession session) {
        final CloseSession request = new CloseSession(session);
        execute(request);
    }

    

    ///////////////////////////////////////////////////////////////////
    // authenticate, authorize
    ///////////////////////////////////////////////////////////////////

    public AuthenticationSession authenticate(final String data) {
        final Authenticate request = new Authenticate(data);
        execute(request);
        return request.getSession();
    }

    public boolean authoriseUsability(final AuthenticationSession session, final String data) {
        final AuthoriseUsability request = new AuthoriseUsability(session, data);
        execute(request);
        return request.getAuthorised();

    }

    public boolean authoriseVisibility(final AuthenticationSession session, final String data) {
        final AuthoriseVisibility request = new AuthoriseVisibility(session, data);
        execute(request);
        return request.getAuthorised();
    }

    
    ///////////////////////////////////////////////////////////////////
    // setAssociation, setValue, clearAssociation, clearValue
    ///////////////////////////////////////////////////////////////////

    public ObjectData[] setAssociation(
            final AuthenticationSession session,
            final String fieldIdentifier,
            final IdentityData target,
            final IdentityData associate) {
        final SetAssociation request = new SetAssociation(session, fieldIdentifier, target, associate);
        execute(request);
        return request.getChanges();
    }

    public ObjectData[] setValue(
            final AuthenticationSession session,
            final String fieldIdentifier,
            final IdentityData target,
            final EncodeableObjectData value) {
        final SetValue request = new SetValue(session, fieldIdentifier, target, value);
        execute(request);
        return request.getChanges();
    }

    public ObjectData[] clearAssociation(
            final AuthenticationSession session,
            final String fieldIdentifier,
            final IdentityData target,
            final IdentityData associate) {
        final ClearAssociation request = new ClearAssociation(session, fieldIdentifier, target, associate);
        execute(request);
        return request.getChanges();
    }

    public ObjectData[] clearValue(final AuthenticationSession session, final String fieldIdentifier, final IdentityData target) {
        final ClearValue request = new ClearValue(session, fieldIdentifier, target);
        execute(request);
        return request.getChanges();
    }

    ///////////////////////////////////////////////////////////////////
    // executeServerAction, executeClientAction
    ///////////////////////////////////////////////////////////////////

    public ServerActionResultData executeServerAction(
            final AuthenticationSession session,
            final String actionType,
            final String actionIdentifier,
            final ReferenceData target,
            final Data[] parameters) {
        final ExecuteServerAction request = new ExecuteServerAction(session, actionType, actionIdentifier, target, parameters);
        execute(request);
        return request.getActionResult();
    }

    public ClientActionResultData executeClientAction(final AuthenticationSession session, final ReferenceData[] data, final int[] types) {
        final ExecuteClientAction request = new ExecuteClientAction(session, data, types);
        execute(request);
        return request.getActionResult();
    }

    ///////////////////////////////////////////////////////////////////
    // getObject, resolveXxx
    ///////////////////////////////////////////////////////////////////

    public ObjectData getObject(final AuthenticationSession session, final Oid oid, final String specificationName) {
        final GetObject request = new GetObject(session, oid, specificationName);
        execute(request);
        return request.getObject();
    }

    public ObjectData resolveImmediately(final AuthenticationSession session, final IdentityData target) {
        final Resolve request = new Resolve(session, target);
        execute(request);
        return request.getUpdateData();
    }

    public Data resolveField(final AuthenticationSession session, final IdentityData target, final String name) {
        final ResolveField request = new ResolveField(session, target, name);
        execute(request);
        return request.getUpdateData();
    }
    
    ///////////////////////////////////////////////////////////////////
    // findInstances, hasInstances
    ///////////////////////////////////////////////////////////////////

    public ObjectData[] findInstances(final AuthenticationSession session, final CriteriaData criteria) {
        final FindInstances request = new FindInstances(session, criteria);
        execute(request);
        return request.getInstances();
    }

    public boolean hasInstances(final AuthenticationSession session, final String fullName) {
        final HasInstances request = new HasInstances(session, fullName);
        execute(request);
        return request.getFlag();
    }

    ///////////////////////////////////////////////////////////////////
    // services
    ///////////////////////////////////////////////////////////////////

    public IdentityData oidForService(final AuthenticationSession session, final String id) {
        final OidForService request = new OidForService(session, id);
        execute(request);
        return request.getData();
    }

    ///////////////////////////////////////////////////////////////////
    // getProperties
    ///////////////////////////////////////////////////////////////////

    public Properties getProperties() {
        final GetProperties request = new GetProperties();
        execute(request);
        return request.getProperties();
    }


    ///////////////////////////////////////////////////////////////////
    // Helpers: execute
    ///////////////////////////////////////////////////////////////////

    private void execute(final Request request) {
        synchronized (connection) {
            final Response response = connection.executeRemotely(request);
            if (request.getId() != response.getId()) {
                throw new NakedObjectsRemoteException("Response out of sequence with respect to the request: " + request.getId()
                        + " & " + response.getId() + " respectively");
            }
            LOG.debug("response " + response);
            request.setResponse(response.getObject());
        }
    }

    protected final Response executeRemotely(final Request request) {
        return null;
    }
    
    ///////////////////////////////////////////////////////////////////
    // Dependencies (injected)
    ///////////////////////////////////////////////////////////////////
    
    /**
     * .NET Property
     * 
     * @property
     */
    public void set_Connection(final CommandClientConnection connection) {
        setConnection(connection);
    }

    public void setConnection(final CommandClientConnection connection) {
        this.connection = connection;
    }


}
// Copyright (c) Naked Objects Group Ltd.
