package org.nakedobjects.plugins.remoting.command.transport.socket.server;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.SocketException;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.authentication.AuthenticationSession;
import org.nakedobjects.metamodel.commons.debug.DebugInfo;
import org.nakedobjects.metamodel.commons.debug.DebugString;
import org.nakedobjects.plugins.remoting.command.server.ServerConnection;
import org.nakedobjects.plugins.remoting.command.shared.requests.Authenticate;
import org.nakedobjects.plugins.remoting.command.shared.requests.Request;
import org.nakedobjects.plugins.remoting.command.shared.requests.Response;
import org.nakedobjects.runtime.context.NakedObjectsContext;
import org.nakedobjects.runtime.system.internal.monitor.Monitor;

public class Worker implements Runnable {
    private static final Logger LOG = Logger.getLogger(Worker.class);
    private static final Logger ACCESS_LOG = Logger.getLogger("access_log");
    private static int nextId = 1;
    
    private final WorkerPool poolToReturnTo;
    private final int id = nextId++;

    private ServerConnection connection;
    
    private String debugRequest;
    private String debugSession;
    private String debugResponse;
    private String debugContextId;
    private DebugInfo[] debugDetails;
    
    private long responseTime;
    private boolean running = true;

    public Worker(final WorkerPool pool) {
        this.poolToReturnTo = pool;
    }

    public synchronized void gracefulStop() {
        running = false;
    }

    public synchronized void setConnection(final ServerConnection connection) {
        this.connection = connection;
        notify();
    }

    public boolean isAvailable() {
        return connection == null;
    }

    public synchronized void run() {
        running: while (running) {
            while (connection == null) {
                try {
                    wait();
                } catch (final InterruptedException e) {
                    if (!running) {
                        LOG.info("Request to stop : " + toString());
                        break running;
                    }
                }
            }

            try {
                // TODO client should request to disconnect. then session should be closed
                final long start = System.currentTimeMillis();
                final Request request = connection.awaitRequest();
                AuthenticationSession authenticationSession = null;
                try {
                    debugRequest = request.getId() + " - " + request.toString();
                    authenticationSession = request.getSession();

                    if (authenticationSession == null) {
                        debugSession = "(none)";
                        debugContextId = "(none)";
                        
                        if (!(request instanceof Authenticate)) {
                            // TODO: should throw some sort of exception here.
                        }
                    } else {
                        NakedObjectsContext.openSession(authenticationSession);
                        
                        debugSession = authenticationSession.toString();
                        debugContextId = NakedObjectsContext.getSessionId();
                        debugDetails = NakedObjectsContext.debugSession();
                    }
                    
                    // the assumption here is that an ExecutionContext will have been
                    // opened if required (that is, in all cases except an Authenticate request.
                    request.execute(connection.getServer());
                    Response response = new Response(request);
                    
                    debugResponse = response.toString();
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("sending " + debugResponse);
                    }
                    connection.sendResponse(response);

                    final String message = "{" + authenticationSession.getUserName() + "|" + this + "}  " + request.toString();
                    ACCESS_LOG.info(message);
                    Monitor.addEvent("REQUEST", message, debugDetails);
                } catch (final Exception e) {
                    LOG.error("error during remote request", e);
                    StringWriter sw = new StringWriter();
                    e.printStackTrace(new PrintWriter(sw));
                    debugResponse = sw.toString();
                    
                    connection.sendResponse(e);
                } finally {
                    if (authenticationSession != null) {
                        NakedObjectsContext.closeSession();
                    } else {
                        // nothing to do
                    }
                    responseTime = System.currentTimeMillis() - start;
                }
            } catch (final SocketException e) {
                LOG.info("shutting down receiver (" + e + ")");
            } catch (final IOException e) {
                LOG.info("connection exception; closing connection", e);
            } finally {
                end();
            }
        }
        LOG.info("Stopping: " + toString());
    }

    private void end() {
        connection = null;
        poolToReturnTo.returnWorker(this);
    }

    public void debug(final DebugString debug) {
        debug.appendln("context Id", debugContextId);
        debug.appendln("session", debugSession);
        debug.appendln("request", debugRequest);
        debug.appendln("response", debugResponse);
        debug.appendln("duration", responseTime / 1000.0f + " secs.");
    /*    
        try {
        if (debugDetails != null) {
            for (DebugInfo info : debugDetails) {
                debug.appendTitle(info.debugTitle());
                info.debugData(debug);
            }
        }
        } catch (RuntimeException e) {
            debug.appendException(e);
        }
        */
    }

    @Override
    public String toString() {
        return "Worker#" + id;
    }
}

// Copyright (c) Naked Objects Group Ltd.
