package org.nakedobjects.plugins.remoting.command.transport.socket.client;

import static org.nakedobjects.plugins.remoting.command.transport.socket.shared.SocketTransportConstants.REMOTE_DEBUGGING_DEFAULT;
import static org.nakedobjects.plugins.remoting.command.transport.socket.shared.SocketTransportConstants.REMOTE_DEBUGGING_KEY;
import static org.nakedobjects.plugins.remoting.command.transport.socket.shared.SocketTransportConstants.REMOTE_HOST_DEFAULT;
import static org.nakedobjects.plugins.remoting.command.transport.socket.shared.SocketTransportConstants.REMOTE_HOST_KEY;
import static org.nakedobjects.plugins.remoting.command.transport.socket.shared.SocketTransportConstants.REMOTE_KEEPALIVE_DEFAULT;
import static org.nakedobjects.plugins.remoting.command.transport.socket.shared.SocketTransportConstants.REMOTE_KEEPALIVE_KEY;
import static org.nakedobjects.plugins.remoting.command.transport.socket.shared.SocketTransportConstants.REMOTE_PORT_DEFAULT;
import static org.nakedobjects.plugins.remoting.command.transport.socket.shared.SocketTransportConstants.REMOTE_PORT_KEY;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ConnectException;
import java.net.MalformedURLException;
import java.net.Socket;
import java.net.UnknownHostException;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.plugins.remoting.command.client.CommandClientConnection;
import org.nakedobjects.plugins.remoting.command.shared.marshal.ClientMarshaller;
import org.nakedobjects.plugins.remoting.command.shared.marshal.ClientMarshallerOwner;
import org.nakedobjects.plugins.remoting.command.shared.marshal.ConnectionException;
import org.nakedobjects.plugins.remoting.command.shared.requests.Request;
import org.nakedobjects.plugins.remoting.command.transport.socket.shared.ProfilingInputStream;
import org.nakedobjects.plugins.remoting.command.transport.socket.shared.ProfilingOutputStream;
import org.nakedobjects.plugins.remoting.shared.NakedObjectsRemoteException;
import org.nakedobjects.plugins.remoting.command.shared.requests.Response;
import org.nakedobjects.runtime.context.NakedObjectsContext;
import org.nakedobjects.runtime.persistence.ConcurrencyException;



/**
 * Concrete implementation of {@link CommandClientConnection} that delegates to
 * {@link ClientMarshaller} supplied in {@link CommandClientConnectionImpl constructor}.
 */
public class CommandClientConnectionImpl  implements CommandClientConnection, ClientMarshallerOwner {

    private static final Logger LOG = Logger.getLogger(CommandClientConnectionImpl.class);

    private final ClientMarshaller marshaller;
    
    private Socket socket;
    private int port;
    private String host;
    private ProfilingOutputStream profilingOutputStream;
    private ProfilingInputStream profilingInputStream;
    private boolean keepAlive;
    private boolean debugging;


    //////////////////////////////////////////////////////////
    // Constructor
    //////////////////////////////////////////////////////////

    public CommandClientConnectionImpl(final ClientMarshaller marshaller) {
        this.marshaller = marshaller;
        this.marshaller.setClientOwner(this);
    }
    

    protected ClientMarshaller getMarshaller() {
        return marshaller;
    }
    


    
    //////////////////////////////////////////////////////////
    // Configuration
    //////////////////////////////////////////////////////////

    protected boolean isDebugging() {
        return debugging;
    }

    public boolean isKeepAlive() {
        return keepAlive;
    }

    //////////////////////////////////////////////////////////
    // init, shutdown
    //////////////////////////////////////////////////////////

    public void init() {
        port = getConfiguration().getInteger(REMOTE_HOST_KEY, REMOTE_PORT_DEFAULT);
        host = getConfiguration().getString(REMOTE_PORT_KEY, REMOTE_HOST_DEFAULT);
        keepAlive = getConfiguration().getBoolean(REMOTE_KEEPALIVE_KEY, REMOTE_KEEPALIVE_DEFAULT);
        debugging = getConfiguration().getBoolean(REMOTE_DEBUGGING_KEY, REMOTE_DEBUGGING_DEFAULT);
        LOG.info("connections will be made to " + host + " " + port);
    }

    public void shutdown() {
        disconnect();
    }


    //////////////////////////////////////////////////////////
    // reconnect, connect, disconnect
    //////////////////////////////////////////////////////////

    public void reconnect() {
        disconnect();
        connect();
    }

    private void connect() {
        if (socket == null) {
            try {
                socket = new Socket(host, port);
                InputStream input = socket.getInputStream();
                OutputStream output = socket.getOutputStream();
                if (isDebugging()) {
                    output = profilingOutputStream = new ProfilingOutputStream(output);
                    input = profilingInputStream = new ProfilingInputStream(input);
                }
                openStreams(input, output);
                LOG.debug("connection established " + socket);
            } catch (final MalformedURLException e) {
                throw new ConnectionException("Connection failure", e);
            } catch (final UnknownHostException e) {
                throw new ConnectionException("Connection failure", e);
            } catch (final ConnectException e) {
                throw new ConnectionException("Failed to connect to " + host + "/" + port, e);
            } catch (final IOException e) {
                throw new ConnectionException("Connection failure", e);
            }
        }
    }

    /**
     * Delegates to {@link #getMarshaller()}.
     */
    private void openStreams(InputStream input, OutputStream output) throws IOException {
        marshaller.openStreams(input, output);
    }


    private void disconnect() {
        if (socket != null) {
            try {
                socket.close();
                socket = null;
            } catch (final IOException e) {
                LOG.error("failed to close connection", e);
            }
        }
    }


    //////////////////////////////////////////////////////////
    // executeRemotely
    //////////////////////////////////////////////////////////

    public Response executeRemotely(final Request request) {
        connect();

        Object response;
        try {
            response = request(request);
            if (profilingOutputStream != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug(profilingOutputStream.getSize() + " bytes sent in " + profilingOutputStream.getTime() + " seconds");
                }
                profilingOutputStream.resetTimer();
            }
            if (profilingInputStream != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug(profilingInputStream.getSize() + " bytes received in " + profilingInputStream.getTime() + " seconds");
                }
                profilingInputStream.resetTimer();
            }
            if (!isKeepAlive()) {
                disconnect();
            }
        } catch (final IOException e) {
            disconnect();
            throw new ConnectionException("Failed request", e);
        }
        if (response instanceof ConcurrencyException) {
            throw (ConcurrencyException) response;
        } else if (response instanceof Exception) {
            throw new NakedObjectsRemoteException("Exception occurred on server", (Throwable) response);
        } else {
            return (Response) response;
        }
    }



    /**
     * Hook method, delegates to {@link #getMarshaller()}.
     * 
     * @param request
     * @return
     * @throws IOException
     */
    private Object request(final Request request) throws IOException {
        return marshaller.request(request);
    }



    //////////////////////////////////////////////////////////
    // Dependencies (from singleton)
    //////////////////////////////////////////////////////////
    
    private NakedObjectConfiguration getConfiguration() {
        return NakedObjectsContext.getConfiguration();
    }


}
// Copyright (c) Naked Objects Group Ltd.
