package org.nakedobjects.remoting.server;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import org.jmock.Expectations;
import org.jmock.Mockery;
import org.jmock.integration.junit4.JMock;
import org.jmock.integration.junit4.JUnit4Mockery;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.nakedobjects.metamodel.adapter.NakedObject;
import org.nakedobjects.metamodel.adapter.oid.Oid;
import org.nakedobjects.metamodel.adapter.version.Version;
import org.nakedobjects.metamodel.authentication.AuthenticationSession;
import org.nakedobjects.remoting.server.ServerFacadeImpl;
import org.nakedobjects.remoting.shared.data.DummyClientResultData;
import org.nakedobjects.remoting.shared.data.DummyIdentityData;
import org.nakedobjects.remoting.shared.data.DummyObjectData;
import org.nakedobjects.remoting.shared.data.KnownObjects;
import org.nakedobjects.remoting.shared.encoding.object.ObjectEncoder;
import org.nakedobjects.remoting.shared.encoding.object.data.ClientActionResultData;
import org.nakedobjects.remoting.shared.encoding.object.data.ObjectData;
import org.nakedobjects.remoting.shared.encoding.object.data.ReferenceData;
import org.nakedobjects.remoting.shared.transaction.ClientTransactionEvent;
import org.nakedobjects.runtime.authentication.AuthenticationManager;
import org.nakedobjects.runtime.context.NakedObjectsContext;
import org.nakedobjects.runtime.persistence.ConcurrencyException;
import org.nakedobjects.runtime.testsystem.TestProxyNakedObject;
import org.nakedobjects.runtime.testsystem.TestProxySession;
import org.nakedobjects.runtime.testsystem.TestProxySystem;
import org.nakedobjects.runtime.testsystem.TestProxyVersion;


@RunWith(JMock.class)
public class ServerFacadeImpl_ClientActionTest  {


    private Mockery mockery = new JUnit4Mockery();

    private AuthenticationManager mockAuthenticationManager;
    private ObjectEncoder mockEncoder;

    private ServerFacadeImpl server;
    private AuthenticationSession session;
    private TestProxySystem system;

    /*
     * Testing the Distribution implementation ServerDistribution. This uses the encoder to unmarshall objects
     * and then calls the persistor and reflector; all of which should be mocked.
     */
    @Before
    public void setUp() throws Exception {
        BasicConfigurator.configure();
        LogManager.getRootLogger().setLevel(Level.OFF);

        mockAuthenticationManager = mockery.mock(AuthenticationManager.class);
        mockEncoder = mockery.mock(ObjectEncoder.class);

        server = new ServerFacadeImpl(mockAuthenticationManager);
        server.setEncoder(mockEncoder);

        server.init();


        system = new TestProxySystem();
        system.init();

        session = NakedObjectsContext.getAuthenticationSession();
    }

    @After
    public void tearDown() throws Exception {
        system.shutdown();
    }

    @Test
    public void testExecuteClientActionWithNoWork() {
        
        mockery.checking(new Expectations() {
            {
                one(mockEncoder).encodeClientActionResult(
                        with(equalTo(new ObjectData[0])), 
                        with(equalTo(new Version[0])), 
                        with(equalTo(new ObjectData[0])));
                will(returnValue(new DummyClientResultData(new ObjectData[0], new Version[0], null)));
            }
        });


        // don't start xactn here, since within call.
        final ClientActionResultData result = server.executeClientAction(session, new ReferenceData[0], new int[0]);

        assertEquals(0, result.getPersisted().length);
        assertEquals(0, result.getChanged().length);
    }

    @Test
    public void testExecuteClientActionWhereObjectChanged() {
        final NakedObject adapter = system.createPersistentTestObject();

        final DummyObjectData data = new DummyObjectData(adapter.getOid(), "none", true, new TestProxyVersion(1));

        // prepare the update data to return
        mockery.checking(new Expectations() {
            {
                one(mockEncoder).decode(data, new KnownObjects());
                will(returnValue(adapter));
                
            }
        });

        // results returned in their own container
        final DummyClientResultData results = new DummyClientResultData(new ObjectData[0], new Version[0], null);
        mockery.checking(new Expectations() {
            {
                one(mockEncoder).encodeClientActionResult(
                        with(equalTo(new ReferenceData[1])), 
                        with(equalTo(new Version[] { new TestProxyVersion(2) })),
                        with(equalTo(new ObjectData[0])));
                will(returnValue(results));
            }
        });

        // don't start xactn here, since within call.
        final ClientActionResultData result = 
            server.executeClientAction(
                    session, new ReferenceData[] { data }, new int[] { ClientTransactionEvent.CHANGE });
        final NakedObject object = 
            NakedObjectsContext.getPersistenceSession().loadObject(adapter.getOid(), adapter.getSpecification());
        

        assertEquals(new TestProxyVersion(2), object.getVersion());

        assertEquals(results, result);
    }


    @Test
    public void testExecuteClientActionWhereObjectMadePersistent() {
        final NakedObject adapter = system.createTransientTestObject();

        final DummyObjectData data = new DummyObjectData(adapter.getOid(), "none", true, new TestProxyVersion(1));

        // restore the object on the server
        mockery.checking(new Expectations() {
            {
                one(mockEncoder).decode(data, new KnownObjects());
                will(returnValue(adapter));
                
                one(mockEncoder).encodeIdentityData(adapter);
                will(returnValue(null));
            }
        });

        // return results
        final DummyClientResultData results = new DummyClientResultData(new ObjectData[0], new Version[0], new ObjectData[0]);
        mockery.checking(new Expectations() {
            {
                one(mockEncoder).encodeClientActionResult(
                        with(equalTo(new ReferenceData[1])), 
                        with(equalTo(new Version[1])), 
                        with(equalTo(new ObjectData[0])));
                will(returnValue(results));
            }
        });
        
        // don't start xactn here, since within call.

        final ClientActionResultData result = 
            server.executeClientAction(
                    session, new ReferenceData[] { data }, new int[] { ClientTransactionEvent.ADD });

        final NakedObject object = NakedObjectsContext.getPersistenceSession().loadObject(adapter.getOid(),
                adapter.getSpecification());
        

        assertEquals(results, result);
        assertEquals(adapter, object);
        assertEquals(new TestProxyVersion(1), object.getVersion());
    }

    @Test
    public void testExecuteClientActionFailsWithConcurrencyError() {
        final NakedObject adapter = system.createPersistentTestObject();
        adapter.setOptimisticLock(new TestProxyVersion(7));

        final Oid oid = adapter.getOid();
        final DummyIdentityData identityData = new DummyIdentityData(oid, TestProxyNakedObject.class.getName(),
                new TestProxyVersion(6));

        try {
            server.executeClientAction(new TestProxySession(), new ReferenceData[] { identityData },
                    new int[] { ClientTransactionEvent.DELETE });
            fail();
        } catch (final ConcurrencyException expected) {}
    }

    @Test
    public void testExecuteClientActionWhereObjectDeleted() {
        final NakedObject adapter = system.createPersistentTestObject();

        final Oid oid = adapter.getOid();
        final DummyIdentityData identityData = new DummyIdentityData(oid, TestProxyNakedObject.class.getName(),
                new TestProxyVersion(1));

        // return results
        final DummyClientResultData results = new DummyClientResultData(new ObjectData[0], new Version[0], null);
        mockery.checking(new Expectations() {
            {
                one(mockEncoder).encodeClientActionResult(
                        with(equalTo(new ReferenceData[1])), 
                        with(equalTo(new Version[1])), 
                        with(equalTo(new ObjectData[0])));
                will(returnValue(results));
            }
        });

        // don't start xactn here, since within call.

        final ClientActionResultData result = 
            server.executeClientAction(
                    new TestProxySession(), new ReferenceData[] { identityData }, new int[] { ClientTransactionEvent.DELETE });

        assertEquals(results, result);
    }

}
// Copyright (c) Naked Objects Group Ltd.
