package org.nakedobjects.runtime.boot;

import org.nakedobjects.metamodel.config.loader.ConfigurationLoaderDefault;
import org.nakedobjects.metamodel.config.loader.NotFoundPolicy;
import org.nakedobjects.runtime.boot.systemfactory.NakedObjectSystemFactory;
import org.nakedobjects.runtime.system.DeploymentType;
import org.nakedobjects.runtime.system.NakedObjectsSystem;
import org.nakedobjects.runtime.system.SystemConstants;


/**
 * Boots up a {@link NakedObjectsSystem} using the supplied {@link NakedObjectSystemFactory} to actually
 * create the {@link NakedObjectsSystem}.
 */
public abstract class NakedObjectsBootstrapperAbstract {

    private final NakedObjectSystemFactory systemFactory;

    public NakedObjectsBootstrapperAbstract(NakedObjectSystemFactory systemFactory) {
        this.systemFactory = systemFactory;
    }

    public NakedObjectSystemFactory getSystemFactory() {
        return systemFactory;
    }

    /**
     * Intended to be called by subclasses; actually does the work of creating the system (using the
     * {@link #getSystemFactory() system factory} provided in the constructor, and then
     * {@link NakedObjectsSystem#init() initialises} the system and starts client and/or server.
     */
    protected void bootSystem(final BootSystemParameter boot) {
        final ConfigurationLoaderDefault configurationLoader = createConfigurationLoader(boot);
        final NakedObjectsSystem system = systemFactory.createSystem(configurationLoader, boot.loggingLevel, boot.deploymentType,
                boot.splash);

        system.init();

        // specific to this bootstrap mechanism
        startClientOrServer(boot.user, boot.password, system);
    }

    private ConfigurationLoaderDefault createConfigurationLoader(final BootSystemParameter boot) {
        final ConfigurationLoaderDefault configurationLoader = new ConfigurationLoaderDefault();
        addConfigurationFiles(configurationLoader, boot.deploymentType, boot.config);
        addConfigurationProperties(configurationLoader, boot.additionalProperties);

        configurationLoader.add(SystemConstants.REQUESTED_REFLECTOR_KEY, boot.reflector);
        configurationLoader.add(SystemConstants.REQUESTED_DOMAIN_OBJECT_PERSISTENCE_MECHANISM_KEY, boot.persistenceMechanism);
        configurationLoader.add(SystemConstants.REQUESTED_CONNECTION_KEY, boot.connection);
        configurationLoader.add(SystemConstants.REQUESTED_VIEWER_KEY, boot.viewer);
        configurationLoader.add(SystemConstants.REQUESTED_FIXTURE_KEY, boot.fixture);
        return configurationLoader;
    }

    private void startClientOrServer(final String user, final String password, final NakedObjectsSystem system) {
        system.startClient(user, password);
        system.startServer();
    }

    private void addConfigurationFiles(
            final ConfigurationLoaderDefault loader,
            final DeploymentType deploymentType,
            final String config) {
        if (config != null) {
            loader.addConfigurationFile(config, NotFoundPolicy.FAIL_FAST);
        }
        String type = deploymentType.name().toLowerCase();
        loader.addConfigurationFile(type + ".properties", NotFoundPolicy.CONTINUE);
    }

    private void addConfigurationProperties(final ConfigurationLoaderDefault loader, final String[] additionalProperties) {
        if (additionalProperties != null) {
            for (int i = 0; i < additionalProperties.length; i += 2) {
                loader.add(additionalProperties[i], additionalProperties[i + 1]);
            }
        }
    }

}
// Copyright (c) Naked Objects Group Ltd.
