package org.nakedobjects.runtime.boot.cmdline;

import org.apache.commons.cli.Options;
import org.apache.log4j.Level;
import org.nakedobjects.runtime.boot.BootSystemParameter;
import org.nakedobjects.runtime.boot.NakedObjectsBootstrapperAbstract;
import org.nakedobjects.runtime.boot.RootLogging;
import org.nakedobjects.runtime.boot.cmdline.internal.ArgumentsOptionsBuilder;
import org.nakedobjects.runtime.boot.cmdline.internal.ArgumentsParser;
import org.nakedobjects.runtime.boot.cmdline.internal.BootPrinter;
import org.nakedobjects.runtime.boot.cmdline.internal.CommandLineValidator;
import org.nakedobjects.runtime.boot.systemfactory.usinginstallers.NakedObjectSystemUsingInstallersFactory;
import org.nakedobjects.runtime.installers.InstallerRepository;
import org.nakedobjects.runtime.system.DeploymentType;
import org.nakedobjects.runtime.system.NakedObjectsSystem;
import org.nakedobjects.runtime.system.Splash;
import org.nakedobjects.runtime.system.internal.NakedObjectsLoggingInitializer;


/**
 * Starts up a {@link NakedObjectsSystem} with the components specified on the command line, if any, using
 * {@link NakedObjectSystemUsingInstallersFactory installers} to interpret the strings into actual components.
 */
public final class NakedObjectsBootstrapperCmdLine extends
        NakedObjectsBootstrapperAbstract {

    private NakedObjectsLoggingInitializer loggingInitializer;
    private final CommandLineValidator commandLineValidator;
    private final BootPrinter printer;

    public NakedObjectsBootstrapperCmdLine() {
        this(new NakedObjectSystemUsingInstallersFactory());
    }

    protected NakedObjectsBootstrapperCmdLine(final NakedObjectSystemUsingInstallersFactory systemFactory) {
        super(systemFactory);
        loggingInitializer = new NakedObjectsLoggingInitializer();
        printer = new BootPrinter();
        commandLineValidator = new CommandLineValidator();
    }

    @Override
    public NakedObjectSystemUsingInstallersFactory getSystemFactory() {
    	return (NakedObjectSystemUsingInstallersFactory) super.getSystemFactory();
    }
    
    public void run(final String[] args) {
        loggingInitializer.configureLogging();
        Level loggingLevel = ArgumentsParser.loggingLevel(args);
        RootLogging.setRootLoggingLevel(loggingLevel);

        getSystemFactory().init();

		final Options options = buildOptions(getSystemFactory().getInstallerLookup());
        ArgumentsParser argumentsParser = parseArguments(options, args, printer);
        if (argumentsParser == null) {
            return;
        }

        final DeploymentType deploymentType = argumentsParser.getDeploymentType();
        final String connection = argumentsParser.getConnection();
        final String objectPersistor = argumentsParser.getObjectPersistor();
        final String userProfilePersistor = argumentsParser.getUserProfilePersistor();
        final String reflector = argumentsParser.getReflector();
        final String user = argumentsParser.getUser();
        final String password = argumentsParser.getPassword();
        final String config = argumentsParser.getConfig();
        final String[] additionalProperties = argumentsParser.getAdditionalProperties();
        final String viewer = argumentsParser.getViewer();
        final Splash splash = Splash.lookup(argumentsParser.isSplash());
        final String fixture = argumentsParser.getFixture();

        if (!commandLineValidator.validateCommandLineArgs(options, printer, deploymentType, connection, objectPersistor,
                userProfilePersistor, user, password)) {
            return;
        }

        bootSystem(new BootSystemParameter(loggingLevel, deploymentType, connection, reflector, objectPersistor,
                userProfilePersistor, user, password, config, additionalProperties, viewer, splash, fixture));
    }

    private Options buildOptions(InstallerRepository installerRepository) {
        ArgumentsOptionsBuilder optionsBuilder = new ArgumentsOptionsBuilder(installerRepository);
        final Options options = new Options();
        optionsBuilder.addBasicOptions(options);
        optionsBuilder.addComponentOptions(options);
        return options;
    }

    /**
     * @return {@link ArgumentsParser} if parsed ok, otherwise <tt>null</tt>.
     */
    private ArgumentsParser parseArguments(Options options, String[] args, BootPrinter printer) {
        ArgumentsParser argumentsParser = new ArgumentsParser(printer);
        return argumentsParser.parse(options, args) ? argumentsParser : null;
    }

}
// Copyright (c) Naked Objects Group Ltd.
