package org.nakedobjects.runtime.persistence;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatState;

import java.util.List;

import org.nakedobjects.applib.clock.Clock;
import org.nakedobjects.applib.fixtures.FixtureClock;
import org.nakedobjects.metamodel.spec.NakedObjectSpecification;
import org.nakedobjects.metamodel.specloader.SpecificationLoader;
import org.nakedobjects.runtime.system.DeploymentType;


/**
 * Implementation that just delegates to a supplied {@link PersistenceSessionFactory}.
 */
public abstract class PersistenceSessionFactoryDelegating implements PersistenceSessionFactory {

    private final DeploymentType deploymentType;
    private final PersistenceSessionFactoryDelegate persistenceSessionFactoryDelegate;
    private SpecificationLoader specificationLoader;
    private List<Object> serviceList;
    

    public PersistenceSessionFactoryDelegating(
            final DeploymentType deploymentType,
            final PersistenceSessionFactoryDelegate persistenceSessionFactoryDelegate) {
        this.deploymentType = deploymentType;
        this.persistenceSessionFactoryDelegate = persistenceSessionFactoryDelegate;
    }

    public DeploymentType getDeploymentType() {
        return deploymentType;
    }

    public PersistenceSessionFactoryDelegate getDelegate() {
        return persistenceSessionFactoryDelegate;
    }

    public PersistenceSession createPersistenceSession() {
        return persistenceSessionFactoryDelegate.createPersistenceSession(this);
    }


    public final void init() {
    	// check prereq dependencies injected
    	ensureThatState(specificationLoader, is(notNullValue()));
    	ensureThatState(serviceList, is(notNullValue()));
    	
        
        // a bit of a workaround, but required if anything in the metamodel (for example, a
        // ValueSemanticsProvider for a date value type) needs to use the Clock singleton
        // we do this after loading the services to allow a service to prime a different clock
        // implementation (eg to use an NTP time service).
        if (!deploymentType.isProduction() && !Clock.isInitialized()) {
        	FixtureClock.initialize();
        }

        primeSpecificationCache(serviceList);
        doInit(serviceList);
    }

    private void primeSpecificationCache(final List<Object> serviceList) {
        for (Object service : serviceList) {
            specificationLoader.loadSpecification(service.getClass());
        }
    }

    /**
     * Perform implementation-specific initialization.
     * 
     * <p>
     * Prior to this call, the {@link SpecificationLoader} will already have been primed with all discoverable
     * domain classes. Some/many implementations may be able to iterate through the {@link NakedObjectSpecification} 
     * cache to determine all discoverable classes requiring mapping.
     * 
     * <p>
     * That said, the list of services is provided for implementations that wish to perform their
     * mapping through some other mechanism, based on the service list.
     */
    protected void doInit(List<Object> serviceList) {}

    public final void shutdown() {
        doShutdown();
        // other
    }

    /**
     * Optional hook method for shutdown.
     */
    protected void doShutdown() {}

    
    ////////////////////////////////////////////////////////
    // Dependencies (injected via setters)
    ////////////////////////////////////////////////////////

    public List<Object> getServices() {
        return serviceList;
    }

    public void setServices(List<Object> serviceList) {
    	this.serviceList = serviceList;
    }

    /**
     * @see #setSpecificationLoader(SpecificationLoader)
     */
    public SpecificationLoader getSpecificationLoader() {
        return specificationLoader;
    }

    /**
     * Injected prior to {@link #init()}.
     */
    public void setSpecificationLoader(SpecificationLoader specificationLoader) {
        this.specificationLoader = specificationLoader;
    }

    
    

}

// Copyright (c) Naked Objects Group Ltd.
