package org.nakedobjects.runtime.persistence.adaptermanager;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatArg;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatState;

import java.util.Iterator;

import org.apache.log4j.Logger;
import org.nakedobjects.applib.Identifier;
import org.nakedobjects.metamodel.adapter.NakedObject;
import org.nakedobjects.metamodel.adapter.ResolveState;
import org.nakedobjects.metamodel.adapter.oid.AggregatedOid;
import org.nakedobjects.metamodel.adapter.oid.Oid;
import org.nakedobjects.metamodel.adapter.version.Version;
import org.nakedobjects.metamodel.commons.debug.DebugInfo;
import org.nakedobjects.metamodel.commons.debug.DebugString;
import org.nakedobjects.metamodel.commons.ensure.Assert;
import org.nakedobjects.metamodel.facets.object.aggregated.AggregatedFacet;
import org.nakedobjects.metamodel.facets.object.value.ValueFacet;
import org.nakedobjects.metamodel.spec.NakedObjectSpecification;
import org.nakedobjects.metamodel.spec.identifier.Identified;
import org.nakedobjects.metamodel.specloader.SpecificationLoader;
import org.nakedobjects.metamodel.specloader.SpecificationLoaderAware;
import org.nakedobjects.runtime.persistence.PersistenceSession;
import org.nakedobjects.runtime.persistence.adapterfactory.AdapterFactory;
import org.nakedobjects.runtime.persistence.adapterfactory.AdapterFactoryAware;
import org.nakedobjects.runtime.persistence.adaptermanager.internal.OidAdapterHashMap;
import org.nakedobjects.runtime.persistence.adaptermanager.internal.OidAdapterMap;
import org.nakedobjects.runtime.persistence.adaptermanager.internal.PojoAdapterHashMap;
import org.nakedobjects.runtime.persistence.adaptermanager.internal.PojoAdapterMap;
import org.nakedobjects.runtime.persistence.oidgenerator.OidGenerator;
import org.nakedobjects.runtime.persistence.oidgenerator.OidGeneratorAware;


public class AdapterManagerDefault implements AdapterManagerExtended, DebugInfo, AdapterFactoryAware, SpecificationLoaderAware,
        OidGeneratorAware {

    private static final Logger LOG = Logger.getLogger(AdapterManagerDefault.class);

    /**
     * Optionally injected, otherwise will default.
     */
    protected PojoAdapterMap pojoAdapterMap;
    /**
     * Optionally injected, otherwise will default.
     */
    protected OidAdapterMap oidAdapterMap;

    /**
     * Injected using dependency injection.
     */
    private AdapterFactory<?> adapterFactory;

    /**
     * Injected using dependency injection.
     */
    private SpecificationLoader specificationLoader;

    /**
     * Injected using dependency injection.
     * 
     * <p>
     * TODO: sort out the generics on this.
     */
    private OidGenerator oidGenerator;

    // //////////////////////////////////////////////////////////////////
    // constructor
    // //////////////////////////////////////////////////////////////////

    public AdapterManagerDefault() {
    // does nothing
    }

    // //////////////////////////////////////////////////////////////////
    // open, close
    // //////////////////////////////////////////////////////////////////

    public void open() {
        ensureThatState(adapterFactory, is(notNullValue()));
        ensureThatState(specificationLoader, is(notNullValue()));
        ensureThatState(oidGenerator, is(notNullValue()));

        if (oidAdapterMap == null) {
            oidAdapterMap = new OidAdapterHashMap();
        }
        if (pojoAdapterMap == null) {
            pojoAdapterMap = new PojoAdapterHashMap();
        }

        oidAdapterMap.open();
        pojoAdapterMap.open();
    }

    public void close() {
        oidAdapterMap.close();
        pojoAdapterMap.close();
    }

    // //////////////////////////////////////////////////////////////////
    // reset
    // //////////////////////////////////////////////////////////////////

    public void reset() {
        oidAdapterMap.reset();
        pojoAdapterMap.reset();
    }

    // //////////////////////////////////////////////////////////////////
    // Iterable
    // //////////////////////////////////////////////////////////////////

    public Iterator<NakedObject> iterator() {
        return getPojoAdapterMap().iterator();
    }

    // //////////////////////////////////////////////////////////////////
    // Backdoor
    // //////////////////////////////////////////////////////////////////

    public NakedObject addExistingAdapter(final NakedObject nakedObject) {
        map(nakedObject);
        return nakedObject;
    }

    // //////////////////////////////////////////////////////////////////
    // Adapter lookup
    // //////////////////////////////////////////////////////////////////

    public NakedObject getAdapterFor(final Object pojo) {
        ensureThatArg(pojo, is(notNullValue()));

        return getPojoAdapterMap().getAdapter(pojo);
    }

    public NakedObject getAdapterFor(final Oid oid) {
        ensureThatArg(oid, is(notNullValue()));
        ensureMapsConsistent(oid);

        return getOidAdapterMap().getAdapter(oid);
    }

    // //////////////////////////////////////////////////////////////////
    // Adapter lookup/creation
    // //////////////////////////////////////////////////////////////////

    /**
     * {@inheritDoc}
     * 
     * <p>
     * Looks up {@link #getAdapterFor(Object)} or returns a new transient
     */
    public NakedObject adapterFor(final Object pojo) {
        return adapterFor(pojo, (NakedObject) null, (Identified) null);
    }

    public NakedObject adapterFor(final Object pojo, final NakedObject ownerAdapter, Identified identified) {

        // attempt to locate adapter for the pojo
        final NakedObject adapter = getAdapterFor(pojo);
        if (adapter != null) {
            return adapter;
        }

        // need to create (and possibly map) the adapter.
        final NakedObjectSpecification noSpec = getSpecificationLoader().loadSpecification(pojo.getClass());

        // we create value facets as standalone (so not added to maps)
        if (noSpec.containsFacet(ValueFacet.class)) {
            return createStandaloneAdapter(pojo);
        }

        // aggregated objects are either intrinsically aggregate (on their spec) or
        // the reference to them are aggregated.
        //
        // can only do this if have been given an ownerAdapter & identified arguments to act as context.
        if (ownerAdapter != null && identified != null) {
            if (specIsIntrisicallyAggregated(noSpec)) {
                return map(createAggregatedObjectAdapter(ownerAdapter, identified.getIdentifier(), pojo));
            }

            if (aggregatedReference(identified) || specIsIntrisicallyAggregated(noSpec)) {
                return map(createAggregatedObjectAdapter(ownerAdapter, identified.getIdentifier(), pojo));
            }
        }

        // root objects
        return map(createRootAdapter(pojo));
    }

    private boolean specIsIntrisicallyAggregated(final NakedObjectSpecification noSpec) {
        return noSpec.containsFacet(AggregatedFacet.class);
    }

    private boolean aggregatedReference(Identified identified) {
        return identified != null && identified.containsFacet(AggregatedFacet.class);
    }

    public NakedObject recreateAdapter(final Oid oid, final Object pojo) {

        // attempt to locate adapter for the pojo
        final NakedObject adapterLookedUpByPojo = getAdapterFor(pojo);
        if (adapterLookedUpByPojo != null) {
            return adapterLookedUpByPojo;
        }

        // attempt to locate adapter for the Oid
        final NakedObject adapterLookedUpByOid = getAdapterFor(oid);
        if (adapterLookedUpByOid != null) {
            return adapterLookedUpByOid;
        }

        return map(createAdapter(pojo, oid));
    }

    // //////////////////////////////////////////////////////////////////
    // adapter maintenance
    // //////////////////////////////////////////////////////////////////

    public void remapUpdated(final Oid oid) {
        ensureThatArg(oid.hasPrevious(), is(true));

        final Oid previousOid = oid.getPrevious();
        if (LOG.isDebugEnabled()) {
            LOG.debug("remapping oid: " + oid + " with previous oid of: " + previousOid);
        }

        final NakedObject lookedUpAdapter = oidAdapterMap.getAdapter(previousOid);
        if (lookedUpAdapter == null) {
            LOG.warn("could not locate previousOid: " + previousOid);
            return;
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("removing previous oid" + previousOid);
        }
        oidAdapterMap.remove(previousOid);

        // we can't replace the Oid on the looked-up adapter, so instead we
        // just make it the same value as the Oid we were originally passed in
        final Oid lookedUpAdapterOid = lookedUpAdapter.getOid();
        lookedUpAdapterOid.copyFrom(oid);

        // finally re-map the adapter
        oidAdapterMap.add(lookedUpAdapterOid, lookedUpAdapter);
    }

    // //////////////////////////////////////////////////////////////////
    // adapter deletion
    // //////////////////////////////////////////////////////////////////

    /**
     * Removes the specified object from both the identity-adapter map, and the pojo-adapter map.
     * 
     * <p>
     * This indicates that the object is no longer in use, and therefore that no objects exists within the
     * system.
     * 
     * <p>
     * If an {@link NakedObject adapter} is removed while its pojo still is referenced then a subsequent
     * interaction of that pojo will create a different {@link NakedObject adapter}, in a
     * {@link ResolveState#TRANSIENT transient} state.
     * 
     * <p>
     * TODO: should do a cascade remove of any aggregated objects.
     */
    public void removeAdapter(final NakedObject adapter) {
        ensureMapsConsistent(adapter);

        if (LOG.isDebugEnabled()) {
            LOG.debug("removing adapter: " + adapter);
        }

        unmap(adapter);
    }

    // //////////////////////////////////////////////////////////////////
    // Persist API
    // //////////////////////////////////////////////////////////////////

    /**
     * {@inheritDoc}
     * 
     * <p>
     * Note that there is no management of {@link Version}s here. That is because the
     * {@link PersistenceSession} is expected to manage this. (In practice this is done by the
     * <tt>NakedObjectStore</tt> implementation delegated by the <tt>PersistenceSessionObjectStore</tt>, and
     * propogated back to client-side as required).
     */
    public void remapAsPersistent(final NakedObject adapter) {
        ensureMapsConsistent(adapter);

        // although the Oid reference doesn't change, the Oid internal values will change
        final Oid oid = adapter.getOid();
        if (LOG.isDebugEnabled()) {
            LOG.debug("remapAsPersistent: " + oid);
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("removing from maps");
        }

        boolean removed = getOidAdapterMap().remove(oid);
        if (!removed) {
            LOG.warn("could not remove oid: " + oid);
            // should we fail here with a more serious error?
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("updating the Oid");
        }
        getOidGenerator().convertTransientToPersistentOid(oid);

        // finally re-map the adapter
        if (LOG.isDebugEnabled()) {
            LOG.debug("readding into maps; oid is now: " + oid);
        }
        getOidAdapterMap().add(oid, adapter);

        // update the adapter's state
        adapter.changeState(ResolveState.RESOLVED);

        if (LOG.isDebugEnabled()) {
            LOG.debug("made persistent " + adapter + "; was " + oid.getPrevious());
        }

    }

    // //////////////////////////////////////////////////////////////////
    // TestSupport
    // //////////////////////////////////////////////////////////////////

    /**
     * For testing purposes only.
     */
    public NakedObject testCreateTransient(final Object pojo, final Oid oid) {
        if (!oid.isTransient()) {
            throw new IllegalArgumentException(
                    "Oid should be transient; use standard API to recreate adapters for persistent Oids");
        }
        return map(createAdapter(pojo, oid));
    }

    // ///////////////////////////////////////////////////////////////////////////
    // Helpers
    // ///////////////////////////////////////////////////////////////////////////

    /**
     * Creates a new root {@link NakedObject adapter} for the supplied domain object.
     * 
     * <p>
     * The default implementation will always create a new transient {@link NakedObject adapter}, using the
     * {@link OidGenerator}. However, the method has <tt>protected</tt> visibility so can be overridden if
     * required.
     * 
     * <p>
     * For example, some object stores (eg Hibernate) may be able to infer from the pojo itself what the
     * {@link Oid} and persistence state of the object is.
     */
    protected NakedObject createRootAdapter(final Object pojo) {
        Oid transientOid = getOidGenerator().createTransientOid(pojo);
        return createAdapter(pojo, transientOid);
    }

    /**
     * Creates an {@link NakedObject adapter} with an {@link AggregatedOid} (so that its version and its
     * persistence are the same as its owning parent).
     * 
     * <p>
     * Should only be called if the pojo is known not to be {@link #getAdapterFor(Object) mapped}.
     */
    public NakedObject createAggregatedObjectAdapter(NakedObject parent, Identifier identifier, Object pojo) {
        ensureMapsConsistent(parent);

        Assert.assertNotNull(pojo);

        // persistence of aggregated follows the parent
        final Oid aggregatedOid = new AggregatedOid(parent.getOid(), identifier.getMemberName());
        NakedObject aggregatedAdapter = createAdapter(pojo, aggregatedOid);

        // same locking as parent
        aggregatedAdapter.setOptimisticLock(parent.getVersion());

        return aggregatedAdapter;
    }

    /**
     * Creates a {@link NakedObject adapter} with no {@link Oid}.
     * 
     * <p>
     * Should only be called if the pojo is known not to be {@link #getAdapterFor(Object) mapped}, and for
     * immutable value types referenced.
     */
    private NakedObject createStandaloneAdapter(Object pojo) {
        return createAdapter(pojo, null);
    }

    /**
     * Helper method that creates {@link NakedObject adapter} (but does not {@link #map(NakedObject) add}) an
     * {@link NakedObject adapter}, and sets its {@link ResolveState} based on the {@link Oid}.
     * 
     * <p>
     * The {@link ResolveState} state will be:
     * <ul>
     * <li> {@link ResolveState#TRANSIENT} if the {@link Oid} is {@link Oid#isTransient() transient}.
     * <li> {@link ResolveState#GHOST} if the {@link Oid} is persistent (not {@link Oid#isTransient()
     * transient}).
     * <li> {@link ResolveState#STANDALONE} if no {@link Oid} was supplied.
     * </ul>
     */
    private NakedObject createAdapter(final Object pojo, final Oid oid) {
        NakedObject adapter = getAdapterFactory().createAdapter(pojo, oid);
        if (oid == null) {
            adapter.changeState(ResolveState.STANDALONE);
        } else {
            adapter.changeState(oid.isTransient() ? ResolveState.TRANSIENT : ResolveState.GHOST);
        }
        return adapter;
    }

    // //////////////////////////////////////////////////////////////////////////
    // Helpers: map & unmap
    // //////////////////////////////////////////////////////////////////////////

    private NakedObject map(final NakedObject nakedObject) {
        Assert.assertNotNull(nakedObject);
        final Object object = nakedObject.getObject();
        Assert.assertFalse("POJO Map already contains object", object, getPojoAdapterMap().containsPojo(object));

        if (LOG.isDebugEnabled()) {
            // don't interact with the underlying object because may be a ghost
            // and would trigger a resolve
            // don't call toString() on nakedobject because calls hashCode on underlying object,
            // may also trigger a resolve.
            LOG.debug("adding identity for nakedobject with oid=" + nakedObject.getOid());
        }

        // new design... always map naked objects provided not standalone.
        if (nakedObject.getResolveState().isStandalone()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("not mapping standalone adapter");
            }
            return nakedObject;
        }

        // add all aggregated collections
        NakedObjectSpecification noSpec = nakedObject.getSpecification();
        if (!nakedObject.isAggregated() || nakedObject.isAggregated() && !noSpec.isImmutable()) {
            getPojoAdapterMap().add(object, nakedObject);
        }

        // order is important - add to pojo map first, then identity map
        getOidAdapterMap().add(nakedObject.getOid(), nakedObject);

        return nakedObject;
    }

    private void unmap(final NakedObject adapter) {
        ensureMapsConsistent(adapter);

        final Oid oid = adapter.getOid();
        if (oid != null) {
            getOidAdapterMap().remove(oid);
        }
        getPojoAdapterMap().remove(adapter);
    }

    // //////////////////////////////////////////////////////////////////////////
    // Helpers: ensure invariants
    // //////////////////////////////////////////////////////////////////////////

    /**
     * Fail early if any problems.
     */
    private void ensureMapsConsistent(final NakedObject adapter) {
        // it isn't possible to guaranteee that the Oid will have no previous, because
        // if running standalone or server-side then we don't know that we can clear the previous
        //
        // ensureThat(adapter.getOid().hasPrevious(), is(false), "adapter's Oid has a previous value.");

        if (adapter.getResolveState().isStandalone()) {
            return;
        }
        ensurePojoAdapterMapConsistent(adapter);
        ensureOidAdapterMapConsistent(adapter);
    }

    /**
     * Fail early if any problems.
     */
    private void ensureMapsConsistent(final Oid oid) {
        ensureThatArg(oid, is(notNullValue()));

        // it isn't possible to guaranteee that the Oid will have no previous, because
        // if running standalone or server-side then we don't know that we can clear the previous
        //
        // ensureThat(oid.hasPrevious(), is(false), "adapter's Oid has a previous value.");

        NakedObject adapter = getOidAdapterMap().getAdapter(oid);
        if (adapter == null) {
            return;
        }
        ensureOidAdapterMapConsistent(adapter);
        ensurePojoAdapterMapConsistent(adapter);
    }

    private void ensurePojoAdapterMapConsistent(final NakedObject adapter) {
        Object adapterPojo = adapter.getObject();
        NakedObject adapterAccordingToPojoAdapterMap = getPojoAdapterMap().getAdapter(adapterPojo);
        ensureThatArg(adapter, is(adapterAccordingToPojoAdapterMap), "mismatch in PojoAdapterMap: adapter's Pojo: "
                + adapterPojo + ", \n" + "provided adapter: " + adapter + "; \n" + " but map's adapter was : "
                + adapterAccordingToPojoAdapterMap);
    }

    private void ensureOidAdapterMapConsistent(final NakedObject adapter) {
        Oid adapterOid = adapter.getOid();
        NakedObject adapterAccordingToOidAdapterMap = getOidAdapterMap().getAdapter(adapterOid);
        ensureThatArg(adapter, is(adapterAccordingToOidAdapterMap), "mismatch in OidAdapter map: " + "adapter's Oid: "
                + adapterOid + ", " + "provided adapter: " + adapter + "; " + "map's adapter: " + adapterAccordingToOidAdapterMap);
    }

    // //////////////////////////////////////////////////////////////////
    // Replaced code
    // (just commented out for now to act as a reference)
    // //////////////////////////////////////////////////////////////////

    // /**
    // * TODO: to go through
    // */
    // public NakedObject adapterFor(Object pojo) {
    // return adapterFor(pojo, (Oid)null, (Version)null);
    // }

    // replacing this code, just commented out for now to act as a reference.

    // public NakedObject adapterFor(final Object domainObject, final Oid oid, final Version version) {
    // if (oid != null) {
    // return adapterFor(domainObject, oid);
    // }
    //        
    // if (domainObject == null) {
    // return null;
    // }
    //
    // final NakedObject adapter = identityMap.getAdapterFor(domainObject);
    // if (adapter != null) {
    // return adapter;
    // }
    //
    // return createAdapterForNewObjectAndMap(domainObject);
    // }

    // replacing this code, just commented out for now to act as a reference.

    // /**
    // * Creates a new adapter with a new transient OID, and then added to the identity map.
    // *
    // * <p>
    // * The {@link NakedObject adapter's} {@link NakedObject#getResolveState() resolve state}
    // * is set to either {@link ResolveState#TRANSIENT} or {@link ResolveState#STANDALONE}
    // * as required.
    // *
    // * <p>
    // * <b><i>REVIEW: think this is wrong; we should set to
    // * {@link ResolveState#STANDALONE} is the {@link NakedObject adapter} has been created
    // * with no identity ({@link NakedObject#getOid() is null}), not based on the {@link
    // NakedObjectSpecification#isAggregated()}</i></b>
    // */
    // private NakedObject createAdapterForNewObjectAndMap(final Object domainObject) {
    // final Oid transientOid = getOidGenerator().createTransientOid(domainObject);
    // final NakedObject adapter = getAdapterFactory().createAdapter(domainObject, transientOid);
    // if (LOG.isDebugEnabled()) {
    // // don't call toString() because may still be a ghost and any interaction could resolve
    // LOG.debug("creating adapter (transient) for adapter with oid=" + adapter.getOid());
    // }
    // identityMap.addAdapter(adapter);
    //        
    // final NakedObjectSpecification specification = adapter.getSpecification();
    // if (specification.isAggregated()) {
    // adapter.changeState(ResolveState.STANDALONE);
    // } else {
    // adapter.changeState(ResolveState.TRANSIENT);
    // }
    //
    // return adapter;
    // }

    // replacing this code, just commented out for now to act as a reference.

    // private NakedObject adapterFor(final Object domainObject, final Oid oid) {
    // NakedObject adapter = getAdapterFor(oid);
    // if (adapter != null) {
    //            
    // // TODO: REVIEW: was tripping the exception setting a Boolean on a @NotPersistable.
    // // is it safe for immutable values to use equals(.) instead?
    //            
    // NakedObjectSpecification noSpec = adapter.getSpecification();
    // ValueFacet valueFacet = noSpec.getFacet(ValueFacet.class);
    // ImmutableFacet immutableFacet = valueFacet != null? valueFacet.getFacet(ImmutableFacet.class): null;
    // if (immutableFacet != null) {
    // if (!adapter.getObject().equals(domainObject)) {
    // throw new
    // AdapterException("Mapped adapter (of immutable value type) is for a domain object with different value: "
    // + domainObject + "; " + adapter);
    // }
    // } else {
    // if (adapter.getObject() != domainObject) {
    // throw new AdapterException("Mapped adapter is for different domain object: " + domainObject + "; " +
    // adapter);
    // }
    // }
    // return adapter;
    // }
    //        
    // adapter = getAdapterFor(domainObject);
    // if (adapter != null) {
    // if (!adapter.getOid().equals(oid)) {
    // throw new AdapterException("Mapped adapter has oid: " + oid + "; " + adapter);
    // }
    // return adapter;
    // }
    // return createAdapterAndMap(domainObject, oid);
    // }

    // /**
    // * Should only be called if it is known that the domainObject and Oid do not
    // * correspond to any existing {@link NakedObject}.
    // */
    // private NakedObject createAdapterAndMap(final Object domainObject, final Oid oid) {
    // return map(createAdapter(domainObject, oid));
    // }

    // //////////////////////////////////////////////////////////////////
    // debug
    // //////////////////////////////////////////////////////////////////

    public String debugTitle() {
        return "Identity map (adapter manager)";
    }

    public void debugData(final DebugString debug) {
        debug.appendTitle(pojoAdapterMap.debugTitle());
        pojoAdapterMap.debugData(debug);
        debug.appendln();

        debug.appendTitle(oidAdapterMap.debugTitle());
        oidAdapterMap.debugData(debug);

    }

    // ////////////////////////////////////////////////////////////////////
    // injectInto
    // ////////////////////////////////////////////////////////////////////

    public void injectInto(Object candidate) {
        if (AdapterManagerAware.class.isAssignableFrom(candidate.getClass())) {
            AdapterManagerAware cast = AdapterManagerAware.class.cast(candidate);
            cast.setAdapterManager(this);
        }
    }

    // //////////////////////////////////////////////////////////////////////////
    // Optionally Injected: OidAdapterMap
    // //////////////////////////////////////////////////////////////////////////

    private OidAdapterMap getOidAdapterMap() {
        return oidAdapterMap;
    }

    /**
     * For dependency injection.
     * 
     * <p>
     * If not injected, will be instantiated within {@link #init()} method.
     */
    public void setOidAdapterMap(final OidAdapterMap identityAdapterMap) {
        this.oidAdapterMap = identityAdapterMap;
    }

    // //////////////////////////////////////////////////////////////////////////
    // Optionally Injected: IdentityAdapterMap
    // //////////////////////////////////////////////////////////////////////////

    private PojoAdapterMap getPojoAdapterMap() {
        return pojoAdapterMap;
    }

    /**
     * For dependency injection.
     * 
     * <p>
     * If not injected, will be instantiated within {@link #init()} method.
     */
    public void setPojoAdapterMap(final PojoAdapterMap pojoAdapterMap) {
        this.pojoAdapterMap = pojoAdapterMap;
    }

    // /////////////////////////////////////////////////////////////////
    // Dependencies (injected)
    // /////////////////////////////////////////////////////////////////

    /**
     * @see #setAdapterFactory(AdapterFactory)
     */
    public AdapterFactory<?> getAdapterFactory() {
        return adapterFactory;
    }

    /**
     * Injected.
     */
    public void setAdapterFactory(AdapterFactory<?> adapterFactory) {
        this.adapterFactory = adapterFactory;
    }

    /**
     * @see #setSpecificationLoader(SpecificationLoader)
     */
    public SpecificationLoader getSpecificationLoader() {
        return specificationLoader;
    }

    /**
     * Injected.
     */
    public void setSpecificationLoader(SpecificationLoader specificationLoader) {
        this.specificationLoader = specificationLoader;
    }

    /**
     * @see #setOidGenerator(OidGenerator)
     */
    public OidGenerator getOidGenerator() {
        return oidGenerator;
    }

    /**
     * Injected.
     */
    public void setOidGenerator(OidGenerator<?> oidGenerator) {
        this.oidGenerator = oidGenerator;
    }

}

// Copyright (c) Naked Objects Group Ltd.
