package org.nakedobjects.runtime.persistence.objectstore;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatArg;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatState;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.adapter.NakedObject;
import org.nakedobjects.metamodel.adapter.ResolveState;
import org.nakedobjects.metamodel.adapter.oid.Oid;
import org.nakedobjects.metamodel.commons.debug.DebugString;
import org.nakedobjects.metamodel.commons.ensure.Assert;
import org.nakedobjects.metamodel.commons.lang.ToString;
import org.nakedobjects.metamodel.criteria.InstancesCriteria;
import org.nakedobjects.metamodel.facets.object.callbacks.LoadedCallbackFacet;
import org.nakedobjects.metamodel.facets.object.callbacks.LoadingCallbackFacet;
import org.nakedobjects.metamodel.facets.object.callbacks.RemovedCallbackFacet;
import org.nakedobjects.metamodel.facets.object.callbacks.RemovingCallbackFacet;
import org.nakedobjects.metamodel.facets.object.callbacks.UpdatedCallbackFacet;
import org.nakedobjects.metamodel.facets.object.callbacks.UpdatingCallbackFacet;
import org.nakedobjects.metamodel.services.ServicesInjector;
import org.nakedobjects.metamodel.spec.NakedObjectSpecification;
import org.nakedobjects.metamodel.spec.SpecificationFacets;
import org.nakedobjects.metamodel.spec.feature.NakedObjectAssociation;
import org.nakedobjects.metamodel.util.CallbackUtils;
import org.nakedobjects.runtime.context.NakedObjectsContext;
import org.nakedobjects.runtime.persistence.NotPersistableException;
import org.nakedobjects.runtime.persistence.PersistenceSessionAbstract;
import org.nakedobjects.runtime.persistence.PersistenceSessionFactory;
import org.nakedobjects.runtime.persistence.adapterfactory.AdapterFactory;
import org.nakedobjects.runtime.persistence.adaptermanager.AdapterManager;
import org.nakedobjects.runtime.persistence.adaptermanager.AdapterManagerExtended;
import org.nakedobjects.runtime.persistence.objectfactory.ObjectFactory;
import org.nakedobjects.runtime.persistence.objectstore.algorithm.PersistAlgorithm;
import org.nakedobjects.runtime.persistence.objectstore.algorithm.ToPersistObjectSet;
import org.nakedobjects.runtime.persistence.objectstore.transaction.CreateObjectCommand;
import org.nakedobjects.runtime.persistence.objectstore.transaction.DestroyObjectCommand;
import org.nakedobjects.runtime.persistence.objectstore.transaction.ObjectStoreTransactionManager;
import org.nakedobjects.runtime.persistence.objectstore.transaction.SaveObjectCommand;
import org.nakedobjects.runtime.persistence.oidgenerator.OidGenerator;
import org.nakedobjects.runtime.transaction.NakedObjectTransactionManager;
import org.nakedobjects.runtime.transaction.updatenotifier.UpdateNotifier;


public class PersistenceSessionObjectStore extends PersistenceSessionAbstract implements ToPersistObjectSet {

    private static final Logger LOG = Logger.getLogger(PersistenceSessionObjectStore.class);

    private final PersistAlgorithm persistAlgorithm;
    private final ObjectStorePersistence objectStore;


    
    /**
     * Initialize the object store so that calls to this object store access persisted objects and persist
     * changes to the object that are saved.
     */
    public PersistenceSessionObjectStore(
            final PersistenceSessionFactory persistenceSessionFactory, 
            final AdapterFactory<?> adapterFactory, 
            final ObjectFactory objectFactory, 
            final ServicesInjector servicesInjector, 
            final OidGenerator oidGenerator,
            final AdapterManagerExtended identityMap, 
            final PersistAlgorithm persistAlgorithm, 
            final ObjectStorePersistence objectStore) {
        
        super(persistenceSessionFactory, adapterFactory, objectFactory, servicesInjector, oidGenerator, identityMap);
        if (LOG.isDebugEnabled()) {
            LOG.debug("creating " + this);
        }
        
        ensureThatArg(persistAlgorithm, is(not(nullValue())), "persist algorithm required");
        ensureThatArg(objectStore, is(not(nullValue())), "object store required");
        
        this.persistAlgorithm = persistAlgorithm;
        this.objectStore = objectStore;
    }




    // ///////////////////////////////////////////////////////////////////////////
    // init, shutdown
    // ///////////////////////////////////////////////////////////////////////////

    @Override
    protected void doOpen() {

        ensureThatState(objectStore, is(notNullValue()), "object store required");
        ensureThatState(getTransactionManager(), is(notNullValue()), "transaction manager required");
        ensureThatState(persistAlgorithm, is(notNullValue()), "persist algorithm required");

        this.injectInto(objectStore); // as a hydrator
        getAdapterManager().injectInto(objectStore);
        getSpecificationLoader().injectInto(objectStore);
        getTransactionManager().injectInto(objectStore);
        
        getOidGenerator().injectInto(objectStore);
        
        objectStore.open();
    }


    public boolean isFixturesInstalled() {
        return objectStore.isFixturesInstalled();
    }


    protected void doClose() {
        objectStore.close();
    }


    
    public void testReset() {
        objectStore.reset();
        getAdapterManager().reset();
        super.testReset();
    }


    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        LOG.info("finalizing object manager");
    }

    

    
    // ///////////////////////////////////////////////////////////////////////////
    // Loading Objects
    // ///////////////////////////////////////////////////////////////////////////

    public NakedObject loadObject(final Oid oid, final NakedObjectSpecification specification) {
        ensureThatArg(oid, is(notNullValue()));
        ensureThatArg(specification, is(notNullValue()));

        NakedObject adapter = getAdapterManager().getAdapterFor(oid);
        if (adapter == null) {
            // the object store will map for us, using its hydrator (calls back to #recreateAdapter)
            adapter = objectStore.getObject(oid, specification);
        }
        return adapter;
    }

    /**
     * Does nothing.
     */
    public void reload(final NakedObject object) {}

    // ///////////////////////////////////////////////////////////////////////////
    // Resolving Objects
    // ///////////////////////////////////////////////////////////////////////////

    public void resolveField(final NakedObject objectAdapter, final NakedObjectAssociation field) {
        if (field.getSpecification().isMutableAggregated()) {
            return;
        }
        final NakedObject referenceAdapter = field.get(objectAdapter);
        if (referenceAdapter == null || referenceAdapter.getResolveState().isResolved()) {
            return;
        }
        if (!referenceAdapter.isPersistent()) {
            return;
        }
        if (LOG.isInfoEnabled()) {
            // don't log object - it's toString() may use the unresolved field or unresolved collection
            LOG.info("resolve field " + objectAdapter.getSpecification().getShortName() + "." + field.getId() + ": "
                    + referenceAdapter.getSpecification().getShortName() + " " + referenceAdapter.getResolveState().code() + " "
                    + referenceAdapter.getOid());
        }
        objectStore.resolveField(objectAdapter, field);
    }

    public void resolveImmediately(final NakedObject adapter) {
    	// synchronize on the current session
    	// because getting race conditions, I think betweeen different UI threads when
    	// running with DnD viewer + in-memory object store + cglib bytecode enhancement
    	synchronized (NakedObjectsContext.getAuthenticationSession()) {
            final ResolveState resolveState = adapter.getResolveState();
            if (resolveState.canChangeTo(ResolveState.RESOLVING)) {
                Assert.assertFalse("only resolve object that is not yet resolved", adapter, resolveState.isResolved());
                Assert.assertTrue("only resolve object that is persistent", adapter, adapter.isPersistent());
                if (LOG.isInfoEnabled()) {
                    // don't log object - it's toString() may use the unresolved field, or unresolved collection
                    LOG.info("resolve immediately: " + adapter.getSpecification().getShortName() + " "
                            + resolveState.code() + " " + adapter.getOid());
                }
                CallbackUtils.callCallback(adapter, LoadingCallbackFacet.class);
                objectStore.resolveImmediately(adapter);
                CallbackUtils.callCallback(adapter, LoadedCallbackFacet.class);
            }
		}
    }

    // ///////////////////////////////////////////////////////////////////////////
    // Saving & Destroying
    // ///////////////////////////////////////////////////////////////////////////

    public void objectChanged(final NakedObject adapter) {
        final ResolveState resolveState = adapter.getResolveState();
        if (resolveState.respondToChangesInPersistentObjects()) {
            final NakedObjectSpecification specification = adapter.getSpecification();
            
            if (SpecificationFacets.isAlwaysImmutable(specification)
                    || (SpecificationFacets.isImmutableOncePersisted(specification) && adapter.isPersistent())) {
            	// previously used to throw new ObjectPersistenceException("cannot change immutable object");
            	// however, since the the bytecode enhancers effectively make calling objectChanged()
            	// the responsibility of the framework, we may as well now do the check here and ignore
            	// if doesn't apply.
            	
                return;
            }
            CallbackUtils.callCallback(adapter, UpdatingCallbackFacet.class);
            final SaveObjectCommand saveObjectCommand = objectStore.createSaveObjectCommand(adapter);
            getTransactionManager().addCommand(saveObjectCommand);
            CallbackUtils.callCallback(adapter, UpdatedCallbackFacet.class);
            getTransactionManager().getTransaction().getUpdateNotifier().addChangedObject(adapter);
        }
        if (resolveState.respondToChangesInPersistentObjects() || adapter.isTransient()) {
            adapter.fireChangedEvent();
            getUpdateNotifier().addChangedObject(adapter);
        }
    }





    /**
     * Makes a naked object persistent. The specified object should be stored away via this object store's
     * persistence mechanism, and have an new and unique OID assigned to it.  The object, should also be 
     * added to the {@link AdapterManager} as the object is implicitly 'in use'.
     * 
     * <p>
     * If the object has any associations then each of these, where they aren't already persistent, should
     * also be made persistent by recursively calling this method.
     * 
     * <p>
     * If the object to be persisted is a collection, then each element of that collection, that is not
     * already persistent, should be made persistent by recursively calling this method.
     * 
     * @see #madePersistent(NakedObject)
     */
    public void makePersistent(final NakedObject adapter) {
        if (adapter.isPersistent()) {
            throw new NotPersistableException("Object already persistent: " + adapter);
        }
        if (!adapter.getSpecification().persistability().isPersistable()) {
            throw new NotPersistableException("Object is not persistable: " + adapter);
        }
        final NakedObjectSpecification specification = adapter.getSpecification();
        if (specification.isService()) {
            throw new NotPersistableException("Cannot persist services: " + adapter);
        }

        persistAlgorithm.makePersistent(adapter, this);
    }

    /**
     * Removes the specified object from the system. The specified object's data should be removed from the
     * persistence mechanism.
     */
    public void destroyObject(final NakedObject object) {
        LOG.info("destroyObject " + object);
        CallbackUtils.callCallback(object, RemovingCallbackFacet.class);
        final DestroyObjectCommand command = objectStore.createDestroyObjectCommand(object);
        getTransactionManager().addCommand(command);
        CallbackUtils.callCallback(object, RemovedCallbackFacet.class);
    }

    // ///////////////////////////////////////////////////////////////////////////
    // Finding Objects
    // ///////////////////////////////////////////////////////////////////////////

    @Override
    protected NakedObject[] getInstances(final InstancesCriteria criteria) {
        LOG.info("getInstances matching " + criteria);
        final NakedObject[] instances = objectStore.getInstances(criteria);
        clearAllDirty();
        return instances;
    }

    /**
     * Checks whether there are any instances of the specified type. The object store should look for
     * instances of the type represented by <variable>type </variable> and return <code>true</code> if there
     * are, or <code>false</code> if there are not.
     */
    public boolean hasInstances(final NakedObjectSpecification specification) {
        LOG.info("hasInstances of " + specification.getShortName());
        return objectStore.hasInstances(specification);
    }


    // ///////////////////////////////////////////////////////////////////////////
    // Services (for Dependency Injection)
    // ///////////////////////////////////////////////////////////////////////////

    @Override
    protected Oid getOidForService(final String name) {
        return objectStore.getOidForService(name);
    }

    @Override
    protected void registerService(final String name, final Oid oid) {
        objectStore.registerService(name, oid);
    }

    // ///////////////////////////////////////////////////////////////////////////
    // PersistentObjectAdder
    // ///////////////////////////////////////////////////////////////////////////

    
    /**
     * Just downcasts.
     */
    public ObjectStoreTransactionManager getTransactionManager() {
        return (ObjectStoreTransactionManager) super.getTransactionManager();
    }

    /**
     *  Uses the {@link ObjectStore} to {@link ObjectStore#createCreateObjectCommand(NakedObject) create}
     *  a {@link CreateObjectCommand}, and adds to the {@link NakedObjectTransactionManager}.
     */
    public void addPersistedObject(final NakedObject object) {
        getTransactionManager().addCommand(objectStore.createCreateObjectCommand(object));
    }

    /**
     * Callback from the {@link PersistAlgorithm} to indicate that the {@link NakedObject adapter}
     * should now be marked as persistent.
     * 
     * <p>
     * The {@link PersistAlgorithm} is called from {@link #makePersistent(NakedObject)}.
     * 
     * 
     * @see #madePersistent(NakedObject)
     */
    public void madePersistent(final NakedObject object) {
        getAdapterManager().remapAsPersistent(object);
    }

    // ///////////////////////////////////////////////////////////////////////////
    // Debugging
    // ///////////////////////////////////////////////////////////////////////////

    @Override
    public void debugData(final DebugString debug) {
        super.debugData(debug);

        debug.appendTitle("Persistor");
        getTransactionManager().debugData(debug);
        debug.appendln("Persist Algorithm", persistAlgorithm);
        debug.appendln("Object Store", objectStore);
        debug.appendln();

        objectStore.debugData(debug);
    }

    public String debugTitle() {
        return "Object Store Persistor";
    }

    @Override
    public String toString() {
        final ToString toString = new ToString(this);
        if (objectStore != null) {
            toString.append("objectStore", objectStore.name());
        }
        if (persistAlgorithm != null) {
            toString.append("persistAlgorithm", persistAlgorithm.name());
        }
        return toString.toString();
    }



    // ///////////////////////////////////////////////////////////////////////////
    // Helpers
    // ///////////////////////////////////////////////////////////////////////////

    /**
     * Helper.
     */
    private UpdateNotifier getUpdateNotifier() {
        return getTransactionManager().getTransaction().getUpdateNotifier();
    }


    // ///////////////////////////////////////////////////////////////////////////
    // Dependency Injection: constructor
    // ///////////////////////////////////////////////////////////////////////////

    /**
     * Injected by constructor.
     */
    public ObjectStorePersistence getObjectStore() {
        return objectStore;
    }

    /**
     * Injected by constructor.
     */
    public PersistAlgorithm getPersistAlgorithm() {
        return persistAlgorithm;
    }




    




}
// Copyright (c) Naked Objects Group Ltd.
