package org.nakedobjects.runtime.persistence.oidgenerator.simple;

import java.io.Serializable;

import org.nakedobjects.metamodel.adapter.oid.Oid;
import org.nakedobjects.metamodel.commons.encoding.ByteDecoder;
import org.nakedobjects.metamodel.commons.encoding.ByteEncoder;
import org.nakedobjects.metamodel.commons.encoding.Encodable;
import org.nakedobjects.metamodel.commons.ensure.Assert;


public final class SerialOid implements Oid, Encodable, Serializable {

    private static final long serialVersionUID = 1L;
    
    static enum State {
        PERSISTENT,
        TRANSIENT;
        public boolean isTransient() {
            return this == TRANSIENT;
        }
    }

    public static SerialOid createPersistent(final long serialNo) {
        return new SerialOid(serialNo, State.PERSISTENT);
    }

    public static SerialOid createTransient(final long serialNo) {
        return new SerialOid(serialNo, State.TRANSIENT);
    }

    private int hashCode;
    private State state;
    private SerialOid previous;
    private long serialNo;
    private String toString;

    //////////////////////////////////////////////
    // Constructor
    //////////////////////////////////////////////
    
    private SerialOid(final long serialNo, final State state) {
        this.serialNo = serialNo;
        this.state = state;
        cacheState();
    }

    public SerialOid(final ByteDecoder decoder) {
        serialNo = decoder.getLong();
        state = decodeState(decoder);
        final boolean hasPrevious = decoder.getBoolean();
        if (hasPrevious) {
            final long previousSerialNo = decoder.getLong();
            previous = new SerialOid(previousSerialNo, decodeState(decoder));
        }
        cacheState();
    }

    //////////////////////////////////////////////
    // isTransient
    //////////////////////////////////////////////

    public boolean isTransient() {
        return state.isTransient();
    }


    //////////////////////////////////////////////
    // encode, decode
    //////////////////////////////////////////////

    public void encode(final ByteEncoder encoder) {
        encoder.add(serialNo);
        encodeState(encoder);
        encoder.add(previous != null);
        if (previous != null) {
            encoder.add(previous.serialNo);
            previous.encodeState(encoder);
        }
    }

    private void encodeState(final ByteEncoder encoder) {
        encoder.add(state.isTransient());
    }

    /**
     * Reads boolean from decoder (<tt>true</tt> meaning is transient),
     * and converts to state.
     */
    private static State decodeState(final ByteDecoder decoder) {
        return decoder.getBoolean()?State.TRANSIENT:State.PERSISTENT;
    }


    //////////////////////////////////////////////
    // copyFrom
    //////////////////////////////////////////////

    public void copyFrom(final Oid oid) {
        Assert.assertTrue(oid instanceof SerialOid);
        final SerialOid from = (SerialOid) oid;
        this.serialNo = from.serialNo;
        this.state = from.state;
        cacheState();
    }


    //////////////////////////////////////////////
    // Previous
    //////////////////////////////////////////////

    public Oid getPrevious() {
        return previous;
    }

    public boolean hasPrevious() {
        return previous != null;
    }

    public void clearPrevious() {
        previous = null;
    }


    //////////////////////////////////////////////
    // SerialNo (not API)
    //////////////////////////////////////////////

    public void makePersistent(final long serialNo) {
        Assert.assertTrue(state.isTransient());
        previous = new SerialOid(this.serialNo, state);
        this.serialNo = serialNo;
        this.state = State.PERSISTENT;
        cacheState();
    }

    public long getSerialNo() {
        return serialNo;
    }

    
    //////////////////////////////////////////////
    // equals, hashCode
    //////////////////////////////////////////////

    private void cacheState() {
        hashCode = 17;
        hashCode = 37 * hashCode + (int) (serialNo ^ (serialNo >>> 32));
        hashCode = 37 * hashCode + (isTransient() ? 0 : 1);
        toString = (isTransient() ? "T" : "") + "OID#" + Long.toHexString(serialNo).toUpperCase() + (previous == null ? "" : "+");
    }


    /*
     * public void setPrevious(SerialOid previous) { Assert.assertNull(previous); this.previous = previous; }
     */

    @Override
    public boolean equals(final Object other) {
        if (other == this) {
            return true;
        }
        if (getClass() != other.getClass()) {
            return false;
        }
        return equals((SerialOid)other);
    }

    /**
     * Overloaded to allow compiler to link directly if we know the compile-time type.
     * (possible performance improvement - called 166,000 times in normal ref data fixture.
     */
    public boolean equals(final SerialOid other) {
        return other.serialNo == serialNo && other.state == state;
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public String toString() {
        return toString;
    }

}
// Copyright (c) Naked Objects Group Ltd.
