package org.nakedobjects.runtime.system;

import java.io.File;
import java.util.List;

import org.apache.log4j.Logger;
import org.nakedobjects.applib.fixtures.LogonFixture;
import org.nakedobjects.metamodel.commons.about.AboutNakedObjects;
import org.nakedobjects.metamodel.commons.component.Installer;
import org.nakedobjects.metamodel.commons.component.NoopUtils;
import org.nakedobjects.metamodel.commons.debug.DebugInfo;
import org.nakedobjects.metamodel.commons.debug.DebugString;
import org.nakedobjects.metamodel.commons.threads.ThreadRunner;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.metamodel.specloader.NakedObjectReflector;
import org.nakedobjects.runtime.authentication.AuthenticationManager;
import org.nakedobjects.runtime.authentication.AuthenticationRequest;
import org.nakedobjects.runtime.authentication.PasswordAuthenticationRequest;
import org.nakedobjects.runtime.client.NakedObjectClient;
import org.nakedobjects.runtime.context.NakedObjectsContext;
import org.nakedobjects.runtime.context.NakedObjectsContextStatic;
import org.nakedobjects.runtime.context.NakedObjectsContextThreadLocal;
import org.nakedobjects.runtime.fixturesinstaller.FixturesInstaller;
import org.nakedobjects.runtime.imageloader.TemplateImageLoader;
import org.nakedobjects.runtime.installers.InstallerLookup;
import org.nakedobjects.runtime.persistence.PersistenceSessionFactory;
import org.nakedobjects.runtime.remoting.ServerListener;
import org.nakedobjects.runtime.session.NakedObjectSession;
import org.nakedobjects.runtime.session.NakedObjectSessionFactory;
import org.nakedobjects.runtime.system.internal.InitialisationSession;
import org.nakedobjects.runtime.system.internal.NakedObjectsLocaleInitializer;
import org.nakedobjects.runtime.system.internal.NakedObjectsTimeZoneInitializer;
import org.nakedobjects.runtime.system.internal.SplashWindow;
import org.nakedobjects.runtime.system.internal.monitor.HttpServerMonitor;
import org.nakedobjects.runtime.system.internal.monitor.SocketServerMonitor;
import org.nakedobjects.runtime.userprofile.UserProfileStore;


/**
 * 
 */
public abstract class NakedObjectsSystemAbstract implements NakedObjectsSystem {

    public static final Logger LOG = Logger.getLogger(NakedObjectsSystemAbstract.class);

    private static final int SPLASH_DELAY_DEFAULT = 6;

    private final NakedObjectsLocaleInitializer localeInitializer;
    private final NakedObjectsTimeZoneInitializer timeZoneInitializer;
    private final DeploymentType deploymentType;

    private boolean hideSplash = false;
    private SplashWindow splashWindow;

    private FixturesInstaller fixtureInstaller;
    private NakedObjectClient client;
    private List<ServerListener> listeners;

    private boolean initialized = false;

    private NakedObjectSessionFactory sessionFactory;

    private LogonFixture logonFixture;

    // ///////////////////////////////////////////
    // Constructors
    // ///////////////////////////////////////////

    public NakedObjectsSystemAbstract(final DeploymentType deploymentType) {
        this(deploymentType, new NakedObjectsLocaleInitializer(), new NakedObjectsTimeZoneInitializer());
    }

    public NakedObjectsSystemAbstract(
            final DeploymentType deploymentType,
            final NakedObjectsLocaleInitializer localeInitializer,
            final NakedObjectsTimeZoneInitializer timeZoneInitializer) {
        this.deploymentType = deploymentType;
        this.localeInitializer = localeInitializer;
        this.timeZoneInitializer = timeZoneInitializer;
    }

    // ///////////////////////////////////////////
    // DeploymentType
    // ///////////////////////////////////////////

    public DeploymentType getDeploymentType() {
        return deploymentType;
    }

    // ///////////////////////////////////////////
    // init, shutdown
    // ///////////////////////////////////////////

    public void init() {

        if (initialized) {
            throw new IllegalStateException("Already initialized");
        } else {
            initialized = true;
        }

        LOG.info("initialising naked objects system");
        LOG.info("working directory: " + new File(".").getAbsolutePath());
        LOG.info("root path: " + getConfiguration().rootPath());

        localeInitializer.initLocale(getConfiguration());
        timeZoneInitializer.initTimeZone(getConfiguration());

        int splashDelay = SPLASH_DELAY_DEFAULT;
        try {
            TemplateImageLoader splashLoader = obtainTemplateImageLoader();
            splashLoader.init();
            showSplash(splashLoader);

            // REVIEW is this the best way to make the configuration available?
            NakedObjectsContext.setConfiguration(getConfiguration());
            
            sessionFactory = doCreateSessionFactory(deploymentType);
            
            NakedObjectsContext.setConfiguration(sessionFactory.getConfiguration());

            initContext(sessionFactory);
            sessionFactory.init();

            NakedObjectsContext.openSession(new InitialisationSession());
            fixtureInstaller = obtainFixturesInstaller();
            
            if (fixtureInstaller != null && !NoopUtils.isNoop(fixtureInstaller)) {
				fixtureInstaller.installFixtures();
            	
            	// only allow logon fixtures if not in production mode.
            	if (!deploymentType.isProduction()) {
            		logonFixture = fixtureInstaller.getLogonFixture();
            	}
            }
            NakedObjectsContext.closeSession();

            AuthenticationManager authenticationManager = sessionFactory.getAuthenticationManager();
            client = obtainClient();
            if (client != null) {
                client.setAuthenticationManager(authenticationManager);
            }

            listeners = obtainServerListeners();
            startListening(listeners, authenticationManager);

        } catch (NakedObjectSystemException ex) {
            LOG.error("failed to initialise", ex);
            splashDelay = 0;
            throw new RuntimeException(ex);
        } finally {
            removeSplash(splashDelay);
        }

    }

    /**
     * TODO: this needs to become pluggable. There currently isn't anyway to use the
     * <tt>NakedObjectsContextPipe</tt> implementation.
     * @param nakedObjectConfiguration 
     */
    private void initContext(NakedObjectSessionFactory sessionFactory) {
        if (!getDeploymentType().isMultithreaded()) {
            if (!sessionFactory.getDeploymentType().isProduction()) {
                // autocloseable, replaceable
                NakedObjectsContextStatic.createRelaxedInstance(sessionFactory);
            } else {
                NakedObjectsContextStatic.createInstance(sessionFactory);
            }
        } else {
            NakedObjectsContextThreadLocal.createInstance(sessionFactory);
        }

    }

    public void shutdown() {
        LOG.info("shutting down system");
        if (listeners != null) {
            for (final ServerListener listener : listeners) {
                listener.stop();
            }
        }
        NakedObjectsContext.closeAllSessions();
    }

    // ///////////////////////////////////////////
    // Hook:
    // ///////////////////////////////////////////

    /**
     * Hook method; the returned implementation is expected to use the same general approach as the subclass
     * itself.
     * 
     * <p>
     * So, for example, <tt>NakedObjectsSystemUsingInstallers</tt> uses the {@link InstallerLookup} mechanism
     * to find its components. The corresponding <tt>ExecutionContextFactoryUsingInstallers</tt> object
     * returned by this method should use {@link InstallerLookup} likewise.
     */
    protected abstract NakedObjectSessionFactory doCreateSessionFactory(final DeploymentType deploymentType)
            throws NakedObjectSystemException;

    // ///////////////////////////////////////////
    // startClient
    // ///////////////////////////////////////////

    public Thread startClient(final String user, final String password) {
        final DeploymentType deploymentType = getDeploymentType();
        if (!deploymentType.hasViewer()) {
        	return null;
        } 
        Runnable target = new Runnable() {
            public void run() {
                // REVIEW is it valid to allow all types of clients to be loaded when running
                // as a server - should we allow HTML but not DND?
                AuthenticationRequest authentationRequest = null;
                if (user != null) {
                    // TODO if user name set up then authenticate user here, and then set authenticateUser
                    // flag to false so not prompted. Also allow name to be passed to client to be used in
                    // its
                    // prompt
                    authentationRequest = new PasswordAuthenticationRequest(user, password);
                }

                runClient(deploymentType, authentationRequest);
            }
        };
        return new ThreadRunner().startThread(target, "Client");
        
    }

    // ///////////////////////////////////////////
    // startServer
    // ///////////////////////////////////////////

    public void startServer() {
        if (getDeploymentType().shouldMonitor()) {
            Runnable serverMonitorRunnable = new Runnable() {
                public void run() {
                    final SocketServerMonitor serverMonitor = new SocketServerMonitor();
                    serverMonitor.setTarget(NakedObjectsSystemAbstract.this);
                    serverMonitor.listen();
                }
            };
            new ThreadRunner().startThread(serverMonitorRunnable, "Monitor-1");
            Runnable httpServerMonitorRunnable = new Runnable() {
                public void run() {
                    final HttpServerMonitor httpServerMonitor = new HttpServerMonitor();
                    httpServerMonitor.setTarget(NakedObjectsSystemAbstract.this);
                    httpServerMonitor.listen();
                }
            };
            new ThreadRunner().startThread(httpServerMonitorRunnable, "Monitor-2");
        }
    }

    // ///////////////////////////////////////////
    // Configuration
    // ///////////////////////////////////////////

    /**
     * Populated after {@link #init()}.
     */
    public NakedObjectSessionFactory getSessionFactory() {
        return sessionFactory;
    }

    // ///////////////////////////////////////////
    // Configuration
    // ///////////////////////////////////////////

    public abstract NakedObjectConfiguration getConfiguration();

    // ///////////////////////////////////////////
    // TemplateImageLoader
    // ///////////////////////////////////////////

    protected abstract TemplateImageLoader obtainTemplateImageLoader() throws NakedObjectSystemException;

    // ///////////////////////////////////////////
    // Reflector
    // ///////////////////////////////////////////

    protected abstract NakedObjectReflector obtainReflector(DeploymentType deploymentType) throws NakedObjectSystemException;

    // ///////////////////////////////////////////
    // PersistenceSessionFactory
    // ///////////////////////////////////////////

    protected abstract PersistenceSessionFactory obtainPersistenceSessionFactory(DeploymentType deploymentType)
            throws NakedObjectSystemException;

    // ///////////////////////////////////////////
    // Fixtures
    // ///////////////////////////////////////////

    /**
     * This is the only {@link Installer} that is used by any (all) subclass implementations, because it
     * effectively <i>is</i> the component we need (as opposed to a builder/factory of the component we need).
     * 
     * <p>
     * The fact that the component <i>is</i> an installer (and therefore can be {@link InstallerLookup} looked
     * up} is at this level really just an incidental implementation detail useful for the subclass that uses
     * {@link InstallerLookup} to create the other components.
     */
    protected abstract FixturesInstaller obtainFixturesInstaller() throws NakedObjectSystemException;

    // ///////////////////////////////////////////
    // Authentication Manager
    // ///////////////////////////////////////////

    protected abstract AuthenticationManager obtainAuthenticationManager(DeploymentType deploymentType)
            throws NakedObjectSystemException;

    
    // ///////////////////////////////////////////
    // UserProfileLoader
    // ///////////////////////////////////////////

    protected abstract UserProfileStore obtainUserProfileStore();

    // ///////////////////////////////////////////
    // Services
    // ///////////////////////////////////////////

    protected abstract List<Object> obtainServices();


    // ///////////////////////////////////////////
    // Client
    // ///////////////////////////////////////////

    /**
     * May be <tt>null</tt>.
     */
    public NakedObjectClient getClient() {
        return client;
    }

    protected abstract NakedObjectClient obtainClient() throws NakedObjectSystemException;

    public void runClient(final DeploymentType deploymentType, final AuthenticationRequest authenticationRequest) {

        NakedObjectClient client = getClient();
        if (client != null) {
            try {
                client.setDeploymentType(deploymentType);
                client.setLogonFixture(logonFixture);
                client.setAuthenticationRequestViaArgs(authenticationRequest);
                client.run();
            } catch (final Exception e) {
                LOG.error("client start up problem", e);
                removeSplash(0); // remove immediately.
            }
        }

    }

    // ///////////////////////////////////////////
    // Fixtures Installer
    // ///////////////////////////////////////////

    public FixturesInstaller getFixturesInstaller() {
        return fixtureInstaller;
    }

    // ///////////////////////////////////////////
    // Server Listeners
    // ///////////////////////////////////////////

    protected abstract List<ServerListener> obtainServerListeners() throws NakedObjectSystemException;

    private void startListening(List<ServerListener> listeners, AuthenticationManager authenticationManager) {
        for (final ServerListener listener : listeners) {
            listener.setAuthenticationManager(authenticationManager);
            final Thread thread = new Thread() {
                @Override
                public void run() {
                    listener.listen();
                }
            };
            thread.start();
        }
    }

    // ///////////////////////////////////////////
    // Splash
    // ///////////////////////////////////////////

    private void showSplash(TemplateImageLoader imageLoader) {
        boolean noSplash = getConfiguration().getBoolean(SystemConstants.NOSPLASH_KEY, SystemConstants.NOSPLASH_DEFAULT)
                || hideSplash;
        if (!noSplash) {
            splashWindow = new SplashWindow(imageLoader);
        }
    }

    public void setHideSplash(final boolean hideSplash) {
        this.hideSplash = hideSplash;
    }

    private void removeSplash(final int delay) {
        if (splashWindow != null) {
            if (delay == 0) {
                splashWindow.removeImmediately();
            } else {
                splashWindow.toFront();
                splashWindow.removeAfterDelay(delay);
            }
        }
    }

    // ///////////////////////////////////////////
    // debugging
    // ///////////////////////////////////////////

    private void debug(final DebugString debug, final Object object) {
        if (object instanceof DebugInfo) {
            final DebugInfo d = (DebugInfo) object;
            debug.appendTitle(d.debugTitle());
            d.debugData(debug);
        } else {
            debug.appendln(object.toString());
            debug.appendln("... no further debug information");
        }
    }

    public DebugInfo debugSection(String selectionName) {
        //DebugInfo deb;
        if (selectionName.equals("Configuration")) {
             return getConfiguration();
        } /*else if (selectionName.equals("Overview")) {
            debugOverview(debug);
        } else  if (selectionName.equals("Authenticator")) {
            deb = NakedObjectsContext.getAuthenticationManager();
        } else if (selectionName.equals("Reflector")) {
            deb = NakedObjectsContext.getSpecificationLoader();
        } else if (selectionName.equals("Contexts")) {
            deb = debugListContexts(debug);
        } else if (selectionName.equals("Listeners")) {
            deb = debugListeners(debug);
        } else if (selectionName.equals("Events")) {

            deb = debugListeners(debug);
        } else {
            deb = debugDisplayContext(selectionName, debug);
        }*/
        return null;
    }
    
    private void debugDisplayContext(final String selector, final DebugString debug) {
        final NakedObjectSession d = NakedObjectsContext.getSession(selector);
        if (d != null) {
            d.debugAll(debug);
        } else {
            debug.appendln("No context: " + selector);
        }
    }

    private void debugListContexts(final DebugString debug) {
        final String[] contextIds = NakedObjectsContext.getInstance().allSessionIds();
        for (int i = 0; i < contextIds.length; i++) {
            debug.appendln(contextIds[i]);
            debug.appendln("-----");
            final NakedObjectSession d = NakedObjectsContext.getSession(contextIds[i]);
            d.debug(debug);
            debug.appendln();
        }
    }

    private void debugListeners(final DebugString debug) {
        for (final ServerListener listener : listeners) {
            debug(debug, listener);
        }
    }

    public String[] debugSectionNames() {
        final String[] general = new String[] { "Overview", "Authenticator", "Configuration", "Reflector", "Listeners", "Requests",
                "Contexts" };
        final String[] contextIds = NakedObjectsContext.getInstance().allSessionIds();
        final String[] combined = new String[general.length + contextIds.length];
        System.arraycopy(general, 0, combined, 0, general.length);
        System.arraycopy(contextIds, 0, combined, general.length, contextIds.length);
        return combined;
    }

    private void debugOverview(final DebugString debug) {
        try {
            debug.appendln(AboutNakedObjects.getFrameworkName());
            debug.appendln(AboutNakedObjects.getFrameworkVersion());
            if (AboutNakedObjects.getApplicationName() != null) {
                debug.appendln("application: " + AboutNakedObjects.getApplicationName());
            }
            if (AboutNakedObjects.getApplicationVersion() != null) {
                debug.appendln("version" + AboutNakedObjects.getApplicationVersion());
            }

            final String user = System.getProperty("user.name");
            final String system = System.getProperty("os.name") + " (" + System.getProperty("os.arch") + ") "
                    + System.getProperty("os.version");
            final String java = System.getProperty("java.vm.name") + " " + System.getProperty("java.vm.version");
            debug.appendln("user: " + user);
            debug.appendln("os: " + system);
            debug.appendln("java: " + java);
            debug.appendln("working directory: " + new File(".").getAbsolutePath());

            debug.appendTitle("System Installer");
            debug.appendln("Fixture Installer", fixtureInstaller == null ? "none" : fixtureInstaller.getClass().getName());

            debug.appendTitle("System Components");
            debug.appendln("Authentication manager", NakedObjectsContext.getAuthenticationManager().getClass().getName());
            debug.appendln("Configuration", getConfiguration().getClass().getName());

            final DebugInfo[] inf = NakedObjectsContext.debugSystem();
            for (int i = 0; i < inf.length; i++) {
                if (inf[i] != null) {
                    inf[i].debugData(debug);
                }
            }
        } catch (RuntimeException e) {
            debug.appendException(e);
        }
    }

}
// Copyright (c) Naked Objects Group Ltd.
