package org.nakedobjects.runtime.system.installers;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.metamodel.specloader.FacetDecoratorInstaller;
import org.nakedobjects.metamodel.specloader.NakedObjectReflector;
import org.nakedobjects.metamodel.specloader.NakedObjectReflectorInstaller;
import org.nakedobjects.runtime.authentication.AuthenticationManager;
import org.nakedobjects.runtime.authentication.AuthenticationManagerInstaller;
import org.nakedobjects.runtime.client.NakedObjectClient;
import org.nakedobjects.runtime.client.NakedObjectClientInstaller;
import org.nakedobjects.runtime.fixturesinstaller.FixturesInstaller;
import org.nakedobjects.runtime.imageloader.TemplateImageLoader;
import org.nakedobjects.runtime.imageloader.TemplateImageLoaderInstaller;
import org.nakedobjects.runtime.installers.InstallerLookup;
import org.nakedobjects.runtime.persistence.PersistenceMechanismInstaller;
import org.nakedobjects.runtime.persistence.PersistenceSessionFactory;
import org.nakedobjects.runtime.persistence.internal.RuntimeContextFromSession;
import org.nakedobjects.runtime.persistence.services.ServicesInstaller;
import org.nakedobjects.runtime.remoting.ClientConnectionInstaller;
import org.nakedobjects.runtime.remoting.ServerListener;
import org.nakedobjects.runtime.remoting.ServerListenerInstaller;
import org.nakedobjects.runtime.session.NakedObjectSessionFactory;
import org.nakedobjects.runtime.session.NakedObjectSessionFactoryDefault;
import org.nakedobjects.runtime.system.DeploymentType;
import org.nakedobjects.runtime.system.NakedObjectSystemException;
import org.nakedobjects.runtime.system.NakedObjectsSystemDefault;
import org.nakedobjects.runtime.system.SystemConstants;
import org.nakedobjects.runtime.transaction.facetdecorator.standard.StandardTransactionFacetDecoratorInstaller;
import org.nakedobjects.runtime.userprofile.UserProfileLoader;
import org.nakedobjects.runtime.userprofile.UserProfileLoaderDefault;
import org.nakedobjects.runtime.userprofile.UserProfileStore;
import org.nakedobjects.runtime.userprofile.UserProfileStoreInstaller;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatArg;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatState;


public class NakedObjectsSystemUsingInstallers extends NakedObjectsSystemDefault {

    public static final Logger LOG = Logger.getLogger(NakedObjectsSystemUsingInstallers.class);

    private final InstallerLookup installerLookup;

    private AuthenticationManagerInstaller authenticatorInstaller;
    private NakedObjectReflectorInstaller reflectorInstaller;
    private ServicesInstaller servicesInstaller;
    private UserProfileStoreInstaller userProfileStoreInstaller;
    private PersistenceMechanismInstaller persistenceMechanismInstaller;
    private NakedObjectClientInstaller clientInstaller;
    private FixturesInstaller fixtureInstaller;

    private final List<ServerListenerInstaller> serverListenerInstallers = new ArrayList<ServerListenerInstaller>();


    // ///////////////////////////////////////////
    // Constructors
    // ///////////////////////////////////////////

    public NakedObjectsSystemUsingInstallers(final DeploymentType deploymentType, final InstallerLookup installerLookup) {
        super(deploymentType);
        ensureThatArg(installerLookup, is(not(nullValue())));
        this.installerLookup = installerLookup;
    }

    // ///////////////////////////////////////////
    // InstallerLookup
    // ///////////////////////////////////////////

    /**
     * As per {@link #NakedObjectsSystemUsingInstallers(DeploymentType, InstallerLookup) constructor}.
     */
    public InstallerLookup getInstallerLookup() {
        return installerLookup;
    }

    // ///////////////////////////////////////////
    // Create context hooks
    // ///////////////////////////////////////////

    public NakedObjectSessionFactory doCreateSessionFactory(final DeploymentType deploymentType)
            throws NakedObjectSystemException {
        final PersistenceSessionFactory persistenceSessionFactory = obtainPersistenceSessionFactory(deploymentType);
        final UserProfileLoader userProfileLoader = new UserProfileLoaderDefault(obtainUserProfileStore());
        return createSessionFactory(deploymentType, userProfileLoader, persistenceSessionFactory);
    }

    /**
     * Overloaded version designed to be called by subclasses that need to explicitly specify different
     * persistence mechanisms.
     * 
     * <p>
     * This is <i>not</i> a hook method, rather it is designed to be called <i>from</i> the
     * {@link #doCreateSessionFactory(DeploymentType) hook method}.
     */
    protected final NakedObjectSessionFactory createSessionFactory(
            final DeploymentType deploymentType,
            final UserProfileLoader userProfileLoader,
            final PersistenceSessionFactory persistenceSessionFactory) throws NakedObjectSystemException {

        final NakedObjectConfiguration configuration = getConfiguration();
        final AuthenticationManager authenticationManager = obtainAuthenticationManager(deploymentType);
        final TemplateImageLoader templateImageLoader = obtainTemplateImageLoader();
        final NakedObjectReflector reflector = obtainReflector(deploymentType);
        
        final List<Object> servicesList = obtainServices();

        // bind metamodel to the (runtime) framework
        // REVIEW: misplaced? seems like a side-effect... 
        reflector.setRuntimeContext(new RuntimeContextFromSession());

        return new NakedObjectSessionFactoryDefault(deploymentType, configuration, templateImageLoader, reflector,
                authenticationManager, userProfileLoader, persistenceSessionFactory, servicesList);

    }

    // ///////////////////////////////////////////
    // Configuration
    // ///////////////////////////////////////////

	/**
     * Returns a <i>snapshot</i> of the {@link NakedObjectConfiguration configuration} held by the
     * {@link #getInstallerLookup() installer lookup}.
     * 
     * @see InstallerLookup#getConfiguration()
     */
    @Override
    public NakedObjectConfiguration getConfiguration() {
        return installerLookup.getConfiguration();
    }

    // ///////////////////////////////////////////
    // Authentication
    // ///////////////////////////////////////////

    public void lookupAndSetAuthenticatorInstaller() {

        if (getDeploymentType().isClientSideOnly()) {
        	
        	NakedObjectConfiguration configuration = installerLookup.getConfiguration();
        	String connection = configuration.getString(SystemConstants.REQUESTED_CONNECTION_KEY);
        	
            ClientConnectionInstaller clientConnectionInstaller = installerLookup.clientConnectionInstaller(connection);
            if (clientConnectionInstaller != null) {
                setAuthenticatorInstaller(clientConnectionInstaller);
                return;
            }
            
        } else {
        	
            final AuthenticationManagerInstaller authenticatorInstaller = installerLookup.authenticatorInstaller(null);
            if (authenticatorInstaller != null) {
                setAuthenticatorInstaller(authenticatorInstaller);
            }
            
        }
    }

    /**
     * Set the type of connection to used to access the server.
     * 
     * <p>
     * Note that the {@link NakedObjectSessionFactoryUsingInstallers} also checks the
     * {@link ClientConnectionInstaller} twice over: to see if a <tt>PersistenceSessionProxy</tt> should be
     * used as a persistor, and for any {@link FacetDecoratorInstaller}s.
     */
    public void setAuthenticatorInstaller(final AuthenticationManagerInstaller authenticationManagerInstaller) {
        this.authenticatorInstaller = authenticationManagerInstaller;
    }

    protected AuthenticationManager obtainAuthenticationManager(DeploymentType deploymentType) {
        return authenticatorInstaller.createAuthenticationManager();
    }
    

    // ///////////////////////////////////////////
    // Fixtures
    // ///////////////////////////////////////////

    public void lookupAndSetFixturesInstaller() {
        NakedObjectConfiguration configuration = installerLookup.getConfiguration();
        String fixture = configuration.getString(SystemConstants.REQUESTED_FIXTURE_KEY);

        final FixturesInstaller fixturesInstaller = installerLookup.fixturesInstaller(fixture);
        if (fixturesInstaller != null) {
            this.fixtureInstaller = fixturesInstaller;
        }
    }

    public void setFixtureInstaller(FixturesInstaller fixtureInstaller) {
		this.fixtureInstaller = fixtureInstaller;
	}
    
    @Override
    protected FixturesInstaller obtainFixturesInstaller() throws NakedObjectSystemException {
        return fixtureInstaller;
    }

    // ///////////////////////////////////////////
    // Client
    // ///////////////////////////////////////////

    public void lookupAndSetClientInstaller() {
        NakedObjectConfiguration configuration = installerLookup.getConfiguration();
        String requestedName = configuration.getString(SystemConstants.REQUESTED_VIEWER_KEY);

        String defaultName = getDeploymentType().isServerSideOnly() ? null : "dnd";
        final NakedObjectClientInstaller clientInstaller = installerLookup.clientInstaller(requestedName, defaultName);
        if (clientInstaller != null) {
            setClientInstaller(clientInstaller);
        } else if (requestedName != null) {
            throw new NakedObjectException("Can't install client component: " + requestedName);
        } else if (defaultName != null) {
            throw new NakedObjectException("Can't install default client component: " + defaultName);
        }
    }

    public void setClientInstaller(NakedObjectClientInstaller clientInstaller) {
        this.clientInstaller = clientInstaller;
    }

    protected NakedObjectClient obtainClient() {
        return clientInstaller != null ? clientInstaller.createClient() : null;
    }

    // ///////////////////////////////////////////
    // Server Listeners
    // ///////////////////////////////////////////

    public void lookupAndSetServerListenerInstaller() {
        if (!getDeploymentType().isServerSideOnly()) {
            return;
        }
        NakedObjectConfiguration configuration = installerLookup.getConfiguration();
        String fromCmdLine = configuration.getString(SystemConstants.REQUESTED_CONNECTION_KEY);

        ServerListenerInstaller serverListenerInstaller = installerLookup.listenerInstaller(fromCmdLine);
        if (serverListenerInstaller != null) {
            addListenerInstaller(serverListenerInstaller);
        }
    }

    /**
     * Add this listener so that it can be {@link #createListenersAndStartListening() setup to start
     * listening}.
     * 
     * <p>
     * We're not doing a pub/sub observer here.
     */
    public void addListenerInstaller(final ServerListenerInstaller installer) {
        serverListenerInstallers.add(installer);
    }

    protected List<ServerListener> obtainServerListeners() {
        final List<ServerListener> listeners = new ArrayList<ServerListener>();
        for (final ServerListenerInstaller serverListenerInstaller : serverListenerInstallers) {
            final ServerListener listener = serverListenerInstaller.createListener();
            listeners.add(listener);
        }
        return listeners;
    }

    // ///////////////////////////////////////////
    // Template Image Loader
    // ///////////////////////////////////////////

    /**
     * Uses the {@link TemplateImageLoader} configured in {@link InstallerLookup}, if available, else falls
     * back to that of the superclass.
     */
    @Override
    protected TemplateImageLoader obtainTemplateImageLoader() {
        TemplateImageLoaderInstaller templateImageLoaderInstaller = installerLookup.templateImageLoaderInstaller(null);
        if (templateImageLoaderInstaller != null) {
            return templateImageLoaderInstaller.createLoader();
        } else {
            return super.obtainTemplateImageLoader();
        }
    }

    // ///////////////////////////////////////////
    // Reflector
    // ///////////////////////////////////////////

    public void setReflectorInstaller(final NakedObjectReflectorInstaller reflectorInstaller) {
        this.reflectorInstaller = reflectorInstaller;
    }

    @Override
    protected NakedObjectReflector obtainReflector(DeploymentType deploymentType) throws NakedObjectSystemException {
        if (reflectorInstaller == null) {
            NakedObjectConfiguration configuration = installerLookup.getConfiguration();
            String fromCmdLine = configuration.getString(SystemConstants.REQUESTED_REFLECTOR_KEY);
            reflectorInstaller = installerLookup.reflectorInstaller(fromCmdLine);
        }
        ensureThatState(reflectorInstaller, is(not(nullValue())),
                "reflector installer has not been injected and could not be looked up");

        // add in transaction support (if already in set then will be ignored)
        reflectorInstaller.addFacetDecoratorInstaller(installerLookup
                .getInstaller(StandardTransactionFacetDecoratorInstaller.class));

        // if acting as a client and there is a client connection installer, then add
        // it as a facet decorator installer also
        if (deploymentType.isClientSideOnly()) {
            String connection = getConfiguration().getString(SystemConstants.REQUESTED_CONNECTION_KEY);
            if (connection != null) {
                FacetDecoratorInstaller clientConnectionInstaller = installerLookup.clientConnectionInstaller(connection);
                reflectorInstaller.addFacetDecoratorInstaller(clientConnectionInstaller);
            }
        }

        return reflectorInstaller.createReflector();
    }

    
    // ///////////////////////////////////////////
    // Services
    // ///////////////////////////////////////////


    public void setServicesInstaller(ServicesInstaller servicesInstaller) {
		this.servicesInstaller = servicesInstaller;
	}
    
	@Override
	protected List<Object> obtainServices() {
        if (servicesInstaller == null) {
            servicesInstaller = installerLookup.servicesInstaller(null);
        }
        ensureThatState(servicesInstaller, is(not(nullValue())),
                "services installer has not been injected and could not be looked up");

        return servicesInstaller.getServices(getDeploymentType());
	}

	
    // ///////////////////////////////////////////
    // User Profile Loader/Store
    // ///////////////////////////////////////////
    

    public void lookupAndSetUserProfileFactoryInstaller() {
        NakedObjectConfiguration configuration = installerLookup.getConfiguration();
        String persistor = configuration.getString(SystemConstants.REQUESTED_USER_PROFILE_STORE_KEY);

           UserProfileStoreInstaller userProfilePersistenceMechanismInstaller = installerLookup.userProfilePersistenceMechanismInstaller(persistor, getDeploymentType());
            if (userProfilePersistenceMechanismInstaller != null) {
                setUserProfileStoreInstaller(userProfilePersistenceMechanismInstaller);
            }
        }
    
    public void setUserProfileStoreInstaller(UserProfileStoreInstaller userProfilestoreInstaller) {
        this.userProfileStoreInstaller = userProfilestoreInstaller;
    }
    
	protected UserProfileStore obtainUserProfileStore() {
		return userProfileStoreInstaller.createUserProfileStore(getConfiguration());
	}
    

    // ///////////////////////////////////////////
    // PersistenceSessionFactory
    // ///////////////////////////////////////////

    public void setPersistenceMechanismInstaller(final PersistenceMechanismInstaller persistenceMechanismInstaller) {
        this.persistenceMechanismInstaller = persistenceMechanismInstaller;
    }

    @Override
    protected PersistenceSessionFactory obtainPersistenceSessionFactory(DeploymentType deploymentType)
            throws NakedObjectSystemException {

        // if client side, attempt to look up connection (that is, a ProxyPersistor)
        if (persistenceMechanismInstaller == null && deploymentType.isClientSideOnly()) {
            String connection = getConfiguration().getString(SystemConstants.REQUESTED_CONNECTION_KEY);
            if (connection != null) {
                persistenceMechanismInstaller = installerLookup.clientConnectionInstaller(connection);
            }
        }
        // if nothing, look for a object store persistor
        if (persistenceMechanismInstaller == null) {
            String persistenceMechanism = getConfiguration().getString(SystemConstants.REQUESTED_DOMAIN_OBJECT_PERSISTENCE_MECHANISM_KEY);
            persistenceMechanismInstaller = installerLookup.persistenceMechanismInstaller(persistenceMechanism, deploymentType);
        }
        ensureThatState(persistenceMechanismInstaller, is(not(nullValue())),
                "persistor installer has not been injected and could not be looked up");

        return persistenceMechanismInstaller.createPersistenceSessionFactory(deploymentType);
    }


}
// Copyright (c) Naked Objects Group Ltd.
