package org.nakedobjects.runtime.transaction;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatState;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.commons.debug.DebugString;
import org.nakedobjects.runtime.session.NakedObjectSession;
import org.nakedobjects.runtime.transaction.messagebroker.MessageBroker;
import org.nakedobjects.runtime.transaction.messagebroker.MessageBrokerDefault;
import org.nakedobjects.runtime.transaction.updatenotifier.UpdateNotifier;
import org.nakedobjects.runtime.transaction.updatenotifier.UpdateNotifierDefault;

public abstract class NakedObjectTransactionManagerAbstract<T extends NakedObjectTransaction> implements NakedObjectTransactionManager {

    private static final Logger LOG = Logger.getLogger(NakedObjectTransactionManagerAbstract.class);

    private NakedObjectSession session;
    
    /**
     * Holds the current or most recently completed transaction.
     */
    private T transaction;

    //////////////////////////////////////////////////////////////////
    // constructor
    //////////////////////////////////////////////////////////////////

    
    public NakedObjectTransactionManagerAbstract() {
    }

    //////////////////////////////////////////////////////////////////
    // open, close
    //////////////////////////////////////////////////////////////////

    public void open() {
        ensureThatState(session, is(notNullValue()), "session is required");
    }


    public void close() {
        if (getTransaction() != null) {
            try {
                abortTransaction();
            } catch (final Exception e2) {
                LOG.error("failure during abort", e2);
            }
        }
        session = null;
    }


    ////////////////////////////////////////////////////////
    // current transaction (if any)
    ////////////////////////////////////////////////////////

    /**
     * Current transaction (if any).
     */
    public T getTransaction() {
        return transaction;
    }

    /**
     * Convenience method returning the {@link UpdateNotifier}
     * of the {@link #getTransaction() current transaction}.
     */
    protected UpdateNotifier getUpdateNotifier() {
        return getTransaction().getUpdateNotifier();
    }



    
    //////////////////////////////////////////////////////////////////
    // create transaction, + hooks
    //////////////////////////////////////////////////////////////////

    /**
     * Creates a new transaction and saves, to be accessible in {@link #getTransaction()}.
     */
    protected final T createTransaction() {
        this.transaction = createTransaction(createMessageBroker(), createUpdateNotifier());
        return transaction;
    }

    
    /**
     * Overridable hook.
     * 
     * <p>
     * The provided {@link MessageBroker} and {@link UpdateNotifier} are obtained from
     * the hook methods ({@link #createMessageBroker()} and {@link #createUpdateNotifier()}).
     * 
     * @see #createMessageBroker()
     * @see #createUpdateNotifier()
     */
    protected abstract T createTransaction(MessageBroker messageBroker, UpdateNotifier updateNotifier);

    /**
     * Overridable hook, used in {@link #createTransaction(MessageBroker, UpdateNotifier)
     * 
     * <p>
     * Called when a new {@link NakedObjectTransaction} is created.
     */
    protected MessageBroker createMessageBroker() {
        return new MessageBrokerDefault();
    }
    
    /**
     * Overridable hook, used in {@link #createTransaction(MessageBroker, UpdateNotifier)
     * 
     * <p>
     * Called when a new {@link NakedObjectTransaction} is created.
     */
    protected UpdateNotifier createUpdateNotifier() {
        return new UpdateNotifierDefault();
    }


    
    
    //////////////////////////////////////////////////////////////////
    // helpers
    //////////////////////////////////////////////////////////////////

    protected void ensureTransactionInProgress() {
        ensureThatState(
                getTransaction() != null && !getTransaction().getState().isComplete(), 
                is(true), "No transaction in progress");
    }

    protected void ensureTransactionNotInProgress() {
        ensureThatState(
                getTransaction() != null && !getTransaction().getState().isComplete(), 
                is(false), "Transaction in progress");
    }


    // ////////////////////////////////////////////////////////////////////
    // injectInto
    // ////////////////////////////////////////////////////////////////////

    public void injectInto(Object candidate) {
        if (NakedObjectTransactionManagerAware.class.isAssignableFrom(candidate.getClass())) {
            NakedObjectTransactionManagerAware cast = NakedObjectTransactionManagerAware.class.cast(candidate);
            cast.setTransactionManager(this);
        }
    }

    
    ////////////////////////////////////////////////////////
    // debugging
    ////////////////////////////////////////////////////////

    public void debugData(final DebugString debug) {
        debug.appendln("Transaction", getTransaction());
    }


    //////////////////////////////////////////////////////////////////
    // Dependencies (injected)
    //////////////////////////////////////////////////////////////////
    
    public NakedObjectSession getSession() {
        return session;
    }


    /**
     * Should be injected prior to {@link #open() opening}
     */
    public void setSession(NakedObjectSession session) {
        this.session = session;
    }


}


// Copyright (c) Naked Objects Group Ltd.
