package org.nakedobjects.runtime.authentication.standard;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.nakedobjects.metamodel.commons.ensure.Ensure.ensureThatArg;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.nakedobjects.metamodel.authentication.AuthenticationSession;
import org.nakedobjects.metamodel.commons.debug.DebugInfo;
import org.nakedobjects.metamodel.commons.debug.DebugString;
import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.metamodel.commons.lang.ToString;
import org.nakedobjects.runtime.authentication.AuthenticationManager;
import org.nakedobjects.runtime.authentication.AuthenticationRequest;
import org.nakedobjects.runtime.authentication.NoAuthenticatorException;
import org.nakedobjects.runtime.context.NakedObjectsContext;


public class AuthenticationManagerStandard implements AuthenticationManager, DebugInfo {

    private List<Authenticator> authenticators = new ArrayList<Authenticator>();
    private final Map<String,String> users = new HashMap<String,String>();
    private boolean createExploration;

    private RandomCodeGenerator randomCodeGenerator;
    private AuthenticationSession testSession;


    ////////////////////////////////////////////////////////////
    // init
    ////////////////////////////////////////////////////////////
    
    public final void init() {
        defaultRandomCodeGeneratorIfNecessary();
        if (authenticators.size() == 0) {
            throw new NakedObjectException("No authenticators specified");
        }
    }

    public void shutdown() {}


    
    ////////////////////////////////////////////////////////////
    // Session Management (including authenticate)
    ////////////////////////////////////////////////////////////

    public final AuthenticationSession authenticate(final AuthenticationRequest request) {
        if (testSession != null) {
            return testSession;
        }

        if (createExploration) {
            final AuthenticationSession session = new ExplorationSession();
            return session;
        }

        for(Authenticator authenticator: authenticators) {
            if (authenticator.canAuthenticate(request)) {
                AuthenticationSession user = null;
                if (authenticator.isValid(request)) {
                    do {
                        user = createSession(request);
                    } while (users.containsKey(user.getValidationCode()));
                    users.put(user.getValidationCode(), user.getUserName());
                }
                return user;
            }
        }
        throw new NoAuthenticatorException("No authenticator available for processing " + request.getClass().getName());
    }

    protected SimpleSession createSession(final AuthenticationRequest request) {
        String randomCode = randomCodeGenerator.generateRandomCode();
        return new SimpleSession(request.getName(), request.getRoles(), randomCode);
    }

    public final boolean isSessionValid(final AuthenticationSession session) {
        final String userName = users.get(session.getValidationCode());
        return userName == null ? false : userName.equals(session.getUserName());
    }

    public final void closeSession(final AuthenticationSession session) {
        users.remove(session.getValidationCode());
        NakedObjectsContext.closeSession();
    }


    ////////////////////////////////////////////////////////////
    // Authenticators
    ////////////////////////////////////////////////////////////

    /**
     * Adds an {@link Authenticator}.
     * 
     * <p>
     * Use either this or alternatively {@link #setAuthenticators(List) inject}
     * the full list of {@link Authenticator}s.
     */
    public final void addAuthenticator(final Authenticator authenticator) {
        authenticators.add(authenticator);
    }
    
    /**
     * Provide direct injection.
     * 
     * <p>
     * Use either this or programmatically {@link #addAuthenticator(Authenticator)}.
     */
    public void setAuthenticators(final List<Authenticator> authenticators) {
        this.authenticators = authenticators;
    }

    /**
     * Determines if this authenticator will simply create an exploration session object when the
     * {@link #authenticate(AuthenticationRequest)} method is called.
     */
    public void setCreateExploration(final boolean createExploration) {
        this.createExploration = createExploration;
    }

    
    public List<Authenticator> getAuthenticators() {
        return Collections.unmodifiableList(authenticators);
    }

    
    ////////////////////////////////////////////////////////////
    // RandomCodeGenerator
    ////////////////////////////////////////////////////////////

    private void defaultRandomCodeGeneratorIfNecessary() {
        if (randomCodeGenerator == null) {
            randomCodeGenerator = new RandomCodeGenerator10Chars();
        }
    }

    /**
     * The {@link RandomCodeGenerator} in use.
     */
    public RandomCodeGenerator getRandomCodeGenerator() {
        return randomCodeGenerator;
    }
    
    /**
     * For injection; will {@link #defaultRandomCodeGeneratorIfNecessary() default} otherwise.
     */
    public void setRandomCodeGenerator(final RandomCodeGenerator randomCodeGenerator) {
        ensureThatArg(randomCodeGenerator, is(notNullValue()), "randomCodeGenerator cannot be null");
        this.randomCodeGenerator = randomCodeGenerator;
    }
    
    ////////////////////////////////////////////////////////////
    // Debugging
    ////////////////////////////////////////////////////////////

    public String debugTitle() {
        return "Authentication Manager";
    }


    public void debugData(final DebugString debug) {
        debug.appendTitle("Authenticators");
        debug.indent();
        for(Authenticator authenticator: authenticators) {
            debug.appendln(authenticator.toString());
        }
        debug.unindent();

        debug.appendTitle("Users");
        debug.indent();
        for(String userName: users.values()) {
            debug.appendln(userName);
        }
        debug.unindent();
    }


    @Override
    public String toString() {
        final ToString str = ToString.createAnonymous(this);
        str.append("authenticators", authenticators.size());
        str.append("users", users.size());
        return str.toString();
    }

    
    
    public void testSetSession(AuthenticationSession authenticationSession) {
        this.testSession = authenticationSession;
    }



}

// Copyright (c) Naked Objects Group Ltd.
