package org.nakedobjects.runtime.authentication.standard.file;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.StringTokenizer;

import org.nakedobjects.metamodel.commons.ensure.Assert;
import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.runtime.authentication.AuthenticationRequest;
import org.nakedobjects.runtime.authentication.PasswordAuthenticationRequest;
import org.nakedobjects.runtime.authentication.standard.Authenticator;


public class PasswordFileAuthenticator implements Authenticator {
    private static final boolean FAILED_AUTHENTICATION = false;
    private static final boolean AUTHENTICATED = true;
    private final File passwordFile;

    public PasswordFileAuthenticator(final File file) {
        passwordFile = file;
    }

    private final void setRoles(final String line, final AuthenticationRequest request) {
        String[] roles;
        final StringTokenizer tokens = new StringTokenizer(line, "|", false);
        roles = new String[tokens.countTokens()];
        for (int i = 0; tokens.hasMoreTokens(); i++) {
            roles[i] = tokens.nextToken();
        }
        request.setRoles(roles);
    }

    public final boolean isValid(final AuthenticationRequest request) {
        final PasswordAuthenticationRequest passwordRequest = (PasswordAuthenticationRequest) request;
        final String username = passwordRequest.getName();
        if (username == null || username.equals("")) {
            return FAILED_AUTHENTICATION;
        }
        final String password = passwordRequest.getPassword();
        Assert.assertNotNull(password);

        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(new FileInputStream(passwordFile)));
            String line;
            while ((line = reader.readLine()) != null) {
                if (line.startsWith("#") || line.trim().length() == 0) {
                    continue;
                }
                final int posFirstColon = line.indexOf(':');
                if (posFirstColon == -1) {
                    throw new NakedObjectException("Invalid entry in password file - no colon (:) found on line: " + line);
                }
                final String name = line.substring(0, posFirstColon);
                if (name.equals(username)) {
                    final int posSecondColon = line.indexOf(':', posFirstColon + 1);
                    final String expected = line.substring(posFirstColon + 1, posSecondColon == -1 ? line.length()
                            : posSecondColon);
                    if (checkPassword(expected, password)) {
                        if (posSecondColon != -1) {
                            setRoles(line.substring(posSecondColon + 1), request);
                        }
                        return AUTHENTICATED;
                    } else {
                        return FAILED_AUTHENTICATION;
                    }
                }
            }
        } catch (final IOException e) {
            throw new NakedObjectException("Failed to open or read password file: " + passwordFile.getAbsolutePath());
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (final IOException ignore) {}
            }
        }

        return FAILED_AUTHENTICATION;
    }

    protected boolean checkPassword(final String expected, final String password) {
        return expected.equals(password);
    }

    public final boolean canAuthenticate(final AuthenticationRequest request) {
        return request instanceof PasswordAuthenticationRequest;
    }

}

// Copyright (c) Naked Objects Group Ltd.
