package org.nakedobjects.runtime.authentication.standard.ldap;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import javax.naming.AuthenticationException;
import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.commons.ensure.Assert;
import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.metamodel.config.ConfigurationConstants;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.runtime.authentication.AuthenticationRequest;
import org.nakedobjects.runtime.authentication.PasswordAuthenticationRequest;
import org.nakedobjects.runtime.authentication.standard.Authenticator;


public class LdapAuthenticator implements Authenticator {

    private static final String COM_SUN_JNDI_LDAP_CTX_FACTORY = "com.sun.jndi.ldap.LdapCtxFactory";

    private static final Logger LOG = Logger.getLogger(LdapAuthenticator.class);
    
    private static final String AUTH_LDAPSERVER_KEY = ConfigurationConstants.ROOT + "security.ldap.server";
    private static final String AUTH_LDAPDN_KEY = ConfigurationConstants.ROOT + "security.ldap.dn";
    
    private static String FILTER = "(objectclass=organizationalRole)";
    
    private final NakedObjectConfiguration configuration;
    private final String ldapProvider;
    private final String ldapDn;
    
    public LdapAuthenticator(final NakedObjectConfiguration configuration) {
        this.configuration = configuration;
        ldapProvider = this.configuration.getString(AUTH_LDAPSERVER_KEY);
        ldapDn = this.configuration.getString(AUTH_LDAPDN_KEY);
    }

    public boolean canAuthenticate(final AuthenticationRequest request) {
        return request instanceof PasswordAuthenticationRequest;
    }

    private void setRoles(final DirContext authContext, final AuthenticationRequest request, final String username)
            throws NamingException {
        final List<String> roles = new ArrayList<String>();
        final SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[] { "cn" });
        final String name = "uid=" + username + ", " + ldapDn;
        final NamingEnumeration<SearchResult> answer = authContext.search(name, FILTER, controls);
        while (answer.hasMore()) {
            final SearchResult result = (SearchResult) answer.nextElement();
            final String roleName = (String) result.getAttributes().get("cn").get(0);
            roles.add(roleName);
            LOG.debug("Adding role: " + roleName);
        }
        request.setRoles((String[]) roles.toArray(new String[roles.size()]));
    }

    public boolean isValid(final AuthenticationRequest request) {
        final PasswordAuthenticationRequest passwordRequest = (PasswordAuthenticationRequest) request;
        final String username = passwordRequest.getName();
        Assert.assertNotNull(username);
        if (username.equals("")) {
            LOG.debug("empty username");
            return false; // failed authentication
        }
        final String password = passwordRequest.getPassword();
        Assert.assertNotNull(password);

        final Hashtable<String,String> env = new Hashtable<String,String>(4);
        env.put(Context.INITIAL_CONTEXT_FACTORY, COM_SUN_JNDI_LDAP_CTX_FACTORY);
        env.put(Context.PROVIDER_URL, ldapProvider);
        env.put(Context.SECURITY_PRINCIPAL, "uid=" + username + ", " + ldapDn);
        env.put(Context.SECURITY_CREDENTIALS, password);

        DirContext authContext = null;
        try {
            authContext = new InitialDirContext(env);
            setRoles(authContext, request, username);
            return true;
        } catch (final AuthenticationException e) {
            return false;
        } catch (final NamingException e) {
            throw new NakedObjectException("Failed to authenticate using LDAP", e);
        } finally {
            try {
                if (authContext != null) {
                    authContext.close();
                }
            } catch (final NamingException e) {
                throw new NakedObjectException("Failed to authenticate using LDAP", e);
            }
        }
    }

}

// Copyright (c) Naked Objects Group Ltd.
