package org.nakedobjects.runtime.authorization.standard.file;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.nakedobjects.applib.Identifier;
import org.nakedobjects.metamodel.commons.ensure.Assert;
import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.metamodel.config.ConfigurationConstants;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.metamodel.spec.identifier.IdentifierFactory;
import org.nakedobjects.runtime.authorization.standard.Authorizor;


public class FileAuthorizor implements Authorizor {

    private static final String AUTH_BLACK_FILE_DEFAULT = "";

    private static final Logger LOG = Logger.getLogger(FileAuthorizor.class);

    private static final String AUTH_WHITE_FILE_KEY = ConfigurationConstants.ROOT + "security.whitelist.file";
    private static final String AUTH_WHITE_FILE_DEFAULT = "config/allow";
    
    private static final String AUTH_BLACK_FILE = ConfigurationConstants.ROOT + "security.blacklist.file";
    private static final String AUTH_LEARN = "security.learn";

    private static final String NONE = AUTH_BLACK_FILE_DEFAULT; // for backward compatibility
    private static final String RO = "-ro";
    private static final String RW = "-rw";
    

    private final Map<String,List<String>> whiteListMap = new HashMap<String,List<String>>();
    private final Map<String,List<String>> blackListMap = new HashMap<String,List<String>>();
    private final boolean learn;
    private final File whiteListFile;
    private File blackListFile;

    private NakedObjectConfiguration configuration;

    public FileAuthorizor(final NakedObjectConfiguration configuration) {
        this.configuration = configuration;
        learn = getConfiguration().getBoolean(ConfigurationConstants.ROOT + AUTH_LEARN, false);
        final String whiteListFileName = getConfiguration().getString(AUTH_WHITE_FILE_KEY,AUTH_WHITE_FILE_DEFAULT);
        Assert.assertTrue(whiteListFileName.length() > 0);
        whiteListFile = new File(whiteListFileName);
        if (!learn && !whiteListFile.canRead()) {
            throw new NakedObjectException("Cannot read white list security file: " + whiteListFileName);
        }
        // black list is optional ....
        final String blackListFileName = getConfiguration().getString(AUTH_BLACK_FILE, AUTH_BLACK_FILE_DEFAULT);
        if (blackListFileName.length() > 0) {
            blackListFile = new File(blackListFileName);
            if (!blackListFile.canRead()) {
                // ... but if it's there we should be able to read it
                throw new NakedObjectException("Cannot read black list security file: " + blackListFileName);
            }
        }
    }

    private void tokenizeLine(final Map<String,List<String>> map, final String line) {
        final StringTokenizer tokens = new StringTokenizer(line, ":", false);
        Assert.assertTrue(tokens.countTokens() == 2);
        final String token1 = tokens.nextToken();
        final String token2 = tokens.nextToken();
        final Identifier identifier = memberFromString(token1.trim());
        final List<String> roles = tokenizeRoles(token2);
        String identityString = identifier.toIdentityString(Identifier.CLASS_NAME_PARMS);
        map.put(identityString, roles);
    }

    public void buildMap(final Map<String,List<String>> map, final File file) {
        try {
            LOG.info("loading authorisation details from " + file.getAbsolutePath());
            final FileReader fileReader = new FileReader(file);
            final BufferedReader buffReader = new BufferedReader(fileReader);
            for (String line; (line = buffReader.readLine()) != null;) {
                tokenizeLine(map, line);
            }
            fileReader.close();
            buffReader.close();
        } catch (final Exception e) {
            throw new NakedObjectException(e);
        }
    }

    public void init() {
        if (learn) {
            return;
        }
        buildMap(whiteListMap, whiteListFile);
        if (blackListFile != null) {
            buildMap(blackListMap, blackListFile);
        }
    }

    private List<String> tokenizeRoles(final String allRoles) {
        final List<String> roles = new ArrayList<String>();
        final StringTokenizer tokens = new StringTokenizer(allRoles, "|", false);
        while (tokens.hasMoreTokens()) {
            String nextToken = tokens.nextToken();
            String trimmedNextToken = nextToken.trim();
            roles.add(trimmedNextToken);
        }
        return roles;
    }

    private Identifier memberFromString(final String identifier) {
        return IdentifierFactory.fromIdentityString(identifier);
    }

    private boolean isQualifiedMatch(final Map<String,List<String>> map, final String role, final String key, final String[] qualifiers) {
        if (map.containsKey(key)) {
            final List<String> roles = map.get(key);
            for (int i = 0; i < qualifiers.length; i++) {
                final String qualifiedRole = role + qualifiers[i];
                if (roles.contains(qualifiedRole)) {
                    return true;
                }
            }
        }
        return false;
    }

    private boolean isListed(final Map<String,List<String>> map, final String role, final Identifier member, final String[] qualifiers) {
        if (map.isEmpty()) {// quick fail
            return false;
        }
        for (int depth = Identifier.CLASS; depth <= Identifier.CLASS_NAME_PARMS; depth++) {

            if (isQualifiedMatch(map, role, member.toIdentityString(depth), qualifiers)) {
                return true;
            }
        }

        return false;
    }

    private boolean isWhiteListed(final String role, final Identifier member, final String[] qualifiers) {
        return isListed(whiteListMap, role, member, qualifiers);
    }

    private boolean isBlackListed(final String role, final Identifier member, final String[] qualifiers) {
        return isListed(blackListMap, role, member, qualifiers);
    }

    private boolean isAuthorised(final String role, final Identifier member, final String[] qualifiers) {
        if (learn) {
            return learn(role, member);
        }
        return isWhiteListed(role, member, qualifiers) && !isBlackListed(role, member, qualifiers);
    }

    public void shutdown() {
        if (learn) {
            writeMap();
        }
    }

    private void writeMap() {
        try {
            LOG.info("writing authorisation details to " + whiteListFile.getAbsolutePath());
            final FileWriter fileWriter = new FileWriter(whiteListFile);
            final BufferedWriter buffWriter = new BufferedWriter(fileWriter);
            Set<Entry<String, List<String>>> entrySet = whiteListMap.entrySet();
            for (int i = 0; i < entrySet.size(); i++) {
                final Map.Entry<String,List<String>> entry = (Map.Entry<String,List<String>>) entrySet.toArray()[i];
                final StringBuffer buff = new StringBuffer();
                buff.append(entry.getKey()).append(":");
                final List<String> roles = entry.getValue();
                for (int j = 0; j < roles.size(); j++) {
                    buff.append(roles.get(j));
                    if (j < roles.size() - 1) {
                        buff.append("|");
                    }
                }
                buffWriter.write(buff.toString());
                buffWriter.newLine();
            }
            buffWriter.flush();
            buffWriter.close();
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        }
    }

    private boolean learn(final String role, final Identifier member) {
        String identityString = member.toIdentityString(Identifier.CLASS_NAME_PARMS);
        if (whiteListMap.containsKey(identityString)) {
            final List<String> roles = (List<String>) whiteListMap.get(identityString);
            if (!roles.contains(role)) {
                roles.add(role);
            }
        } else {
            whiteListMap.put(identityString, Arrays.asList(new String[] { role }));
        }
        return true;
    }

    public boolean isUsable(final String role, final Identifier member) {
        return isAuthorised(role, member, new String[] { NONE, RW });
    }

    public boolean isVisible(final String role, final Identifier member) {
        return isAuthorised(role, member, new String[] { NONE, RO, RW });
    }

    
    public NakedObjectConfiguration getConfiguration() {
        return configuration;
    }

}
