package org.nakedobjects.runtime.installers;

import static org.nakedobjects.runtime.installers.InstallerLookupConstants.AUTHENTICATION_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.AUTHENTICATION_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.AUTHORISATION_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.AUTHORISATION_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.FIXTURES_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.FIXTURES_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.IMAGE_LOADER_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.IMAGE_LOADER_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.PERSISTOR_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.PERSISTOR_NON_PRODUCTION_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.PERSISTOR_PRODUCTION_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.REFLECTOR_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.REFLECTOR_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.REMOTING_CLIENT_CONNECTION_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.REMOTING_CLIENT_CONNECTION_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.SERVICES_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.SERVICES_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.USER_PROFILE_STORE_KEY;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.USER_PROFILE_STORE_NON_PRODUCTION_DEFAULT;
import static org.nakedobjects.runtime.installers.InstallerLookupConstants.USER_PROFILE_STORE_PRODUCTION_DEFAULT;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.nakedobjects.metamodel.commons.about.AboutNakedObjects;
import org.nakedobjects.metamodel.commons.about.ComponentDetails;
import org.nakedobjects.metamodel.commons.component.Installer;
import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.metamodel.commons.factory.InstanceCreationClassException;
import org.nakedobjects.metamodel.commons.factory.InstanceCreationException;
import org.nakedobjects.metamodel.commons.factory.InstanceFactory;
import org.nakedobjects.metamodel.commons.factory.UnavailableClassException;
import org.nakedobjects.metamodel.commons.lang.CastUtils;
import org.nakedobjects.metamodel.commons.lang.StringUtils;
import org.nakedobjects.metamodel.config.NakedObjectConfiguration;
import org.nakedobjects.metamodel.config.loader.ConfigurationLoader;
import org.nakedobjects.metamodel.config.loader.ConfigurationLoaderDefault;
import org.nakedobjects.metamodel.config.loader.NotFoundPolicy;
import org.nakedobjects.metamodel.config.provider.NakedObjectConfigurationProviderAware;
import org.nakedobjects.metamodel.specloader.FacetDecoratorInstaller;
import org.nakedobjects.metamodel.specloader.NakedObjectReflectorInstaller;
import org.nakedobjects.runtime.authentication.AuthenticationManagerInstaller;
import org.nakedobjects.runtime.authorization.AuthorizationFacetDecoratorInstaller;
import org.nakedobjects.runtime.client.NakedObjectClientInstaller;
import org.nakedobjects.runtime.fixturesinstaller.FixturesInstaller;
import org.nakedobjects.runtime.imageloader.TemplateImageLoaderInstaller;
import org.nakedobjects.runtime.persistence.PersistenceMechanismInstaller;
import org.nakedobjects.runtime.persistence.services.ServicesInstaller;
import org.nakedobjects.runtime.remoting.ClientConnectionInstaller;
import org.nakedobjects.runtime.remoting.ServerListenerInstaller;
import org.nakedobjects.runtime.system.DeploymentType;
import org.nakedobjects.runtime.userprofile.UserProfileStoreInstaller;

/**
 * This class retrieves named {@link Installer}s from those loaded at creation, updating
 * the {@link NakedObjectConfiguration} as it goes.
 * 
 * <p>
 * A list of possible classes are read in from the resource file <tt>framework.properties</tt> and instantiated.
 * Each installer has a unique name (with respect to its type) that will be compared when one of this classes 
 * methods are called.
 * 
 * <p>
 * Note that it <i>is</i> possible to use an {@link Installer} implementation even if it has not been registered
 * in <tt>framework.properties</tt> : just specify the {@link Installer}'s fully qualified class name.
 * 
 * <p>
 * The {@link NakedObjectConfiguration} is managed using a mutable representation, namely the
 * {@link ConfigurationLoaderDefault}.  Whenever an {@link Installer} is successfully looked up, any configuration file for
 * that {@link Installer} is merged into the current {@link ConfigurationLoaderDefault}.  
 * A call to {@link #getConfiguration()} will return a (cached) <i>snapshot</i> of the
 * current {@link NakedObjectConfiguration}, but loading further {@link Installer}s may
 * cause a subsequent call to return a larger set of keys. 
 */
public class InstallerLookupImpl implements InstallerLookup {
	
    private static final Logger LOG = Logger.getLogger(InstallerLookupImpl.class);
    public final String INSTALLER_REGISTRY_FILE = "installer-registry.properties";
    

    private List<Installer> installerList;

    /**
     * A mutable representation of the configuration.
     */
    private ConfigurationLoader configurationLoader;
    
    /**
     * Most recent snapshot of {@link NakedObjectConfiguration} obtained from
     * {@link #configurationLoader}.
     * 
     * <p>
     * Whenever further configuration is merged in, this cache is invalidated.
     */
    private NakedObjectConfiguration cachedConfiguration;
    private final Class<?> cls;

    //////////////////////////////////////////////////////////
    // Constructor / initialization
    //////////////////////////////////////////////////////////

    public InstallerLookupImpl(Class<?> cls) {
        this.cls = cls;
    }

    public void loadInstallers() {
        installerList = new ArrayList<Installer>();
        final InputStream in = getInstallerRegistryStream(INSTALLER_REGISTRY_FILE, cls);
        final BufferedReader reader = new BufferedReader(new InputStreamReader(in));
        try {
            String line;
            while ((line = reader.readLine()) != null) {
                final String className = StringUtils.firstWord(line);
                if (className.length() == 0 || className.startsWith("#")) {
                    continue;
                }
                try {
                    final Installer object = (Installer) InstanceFactory.createInstance(className);
                    LOG.debug("created component installer: " + object.getName() + " - " + className);
                    installerList.add(object);
                } catch (final UnavailableClassException e) {
                    LOG.info("component installer not found; it will not be available: " + className);
                } catch (final InstanceCreationClassException e) {
                    LOG.info("instance creation exception: " + e.getMessage());
                } catch (final InstanceCreationException e) {
                    throw e;
                }
            }
        } catch (final IOException e) {
            throw new NakedObjectException(e);
        } finally {
            close(reader);
        }

        List<ComponentDetails> installerVersionList = new ArrayList<ComponentDetails>();
        for (Installer installer : installerList) {
            installerVersionList.add(new InstallerVersion(installer));
        }
        AboutNakedObjects.setComponentDetails(installerVersionList);
    }
    
    
    //////////////////////////////////////////////////////////
    // Type-safe Lookups
    //////////////////////////////////////////////////////////

    public AuthenticationManagerInstaller authenticatorInstaller(String requested) {
        return getInstaller(AuthenticationManagerInstaller.class, requested, AUTHENTICATION_KEY, AUTHENTICATION_DEFAULT);
    }

    public AuthorizationFacetDecoratorInstaller authorisationInstaller(String requested) {
        return getInstaller(AuthorizationFacetDecoratorInstaller.class, requested, AUTHORISATION_KEY, AUTHORISATION_DEFAULT);
    }

    public FixturesInstaller fixturesInstaller(String requested) {
        return getInstaller(FixturesInstaller.class, requested, FIXTURES_KEY, FIXTURES_DEFAULT);
    }

    public TemplateImageLoaderInstaller templateImageLoaderInstaller(String requested) {
        return getInstaller(TemplateImageLoaderInstaller.class, requested, IMAGE_LOADER_KEY, IMAGE_LOADER_DEFAULT);
    }

    public PersistenceMechanismInstaller persistenceMechanismInstaller(final String requested, final DeploymentType deploymentType) {
        String persistorDefault = deploymentType.isExploring() || deploymentType.isPrototyping()? PERSISTOR_NON_PRODUCTION_DEFAULT: PERSISTOR_PRODUCTION_DEFAULT;
        return getInstaller(PersistenceMechanismInstaller.class, requested, PERSISTOR_KEY, persistorDefault);
    }

    public UserProfileStoreInstaller userProfilePersistenceMechanismInstaller(String requested, DeploymentType deploymentType) {
        String persistorDefault = deploymentType.isExploring() || deploymentType.isPrototyping()? USER_PROFILE_STORE_NON_PRODUCTION_DEFAULT: USER_PROFILE_STORE_PRODUCTION_DEFAULT;
        return getInstaller(UserProfileStoreInstaller.class, requested, USER_PROFILE_STORE_KEY, persistorDefault);
    }
    
    public NakedObjectReflectorInstaller reflectorInstaller(final String requested) {
        return getInstaller(NakedObjectReflectorInstaller.class, requested, REFLECTOR_KEY, REFLECTOR_DEFAULT);
    }

    /**
     * Client-side of <tt>remoting</tt>, specifying how to access the server.
     * 
     * <p>
     * This lookup is called in three different contexts:
     * <ul>
     * <li> the <tt>NakedObjectsExecutionContextFactoryUsingInstallers</tt> uses this to lookup
     *      the {@link PersistenceMechanismInstaller} (may be a <tt>ProxyPersistor</tt>)</li>
     * <li> the <tt>NakedObjectsExecutionContextFactoryUsingInstallers</tt> also uses this to lookup
     *      the {@link FacetDecoratorInstaller}; adds in remoting facets.</li>
     * <li> the <tt>NakedObjectsSystemUsingInstallers</tt> uses this to lookup the {@link AuthenticationManagerInstaller}.</li>
     * </ul>
     * 
     * <p>
     * In addition to the usual {@link #mergeConfigurationFor(Installer) merging} of any {@link Installer}-specific
     * configuration files, this lookup also merges in any {@link ClientConnectionInstaller#getRemoteProperties() remote properties}
     * available.
     */
    public ClientConnectionInstaller clientConnectionInstaller(final String requested) {
        ClientConnectionInstaller installer = getInstaller(ClientConnectionInstaller.class, requested, REMOTING_CLIENT_CONNECTION_KEY, REMOTING_CLIENT_CONNECTION_DEFAULT);
        if (installer != null) {
            final Properties properties = installer.getRemoteProperties();
            configurationLoader.add(properties);
        }
        return installer;
    }

    /**
     * Server-side of <tt>remoting</tt>
     */
    public ServerListenerInstaller listenerInstaller(final String name) {
        if (name == null) {
            throw new NakedObjectException("No server listener specified");
        }
        final ServerListenerInstaller component = (ServerListenerInstaller) getInstaller(ServerListenerInstaller.class, name);
        if (component == null) {
            throw new NakedObjectException("No server listener of type " + name);
        }
        return component;
    }

    public ServicesInstaller servicesInstaller(final String requestedImplementationName) {
        return getInstaller(ServicesInstaller.class, requestedImplementationName, SERVICES_KEY, SERVICES_DEFAULT);
    }

    public NakedObjectClientInstaller clientInstaller(final String name, final String defaultName) {
        String viewer;
        if (name == null) {
            viewer = getConfiguration().getString(InstallerLookupConstants.VIEWER_KEY, defaultName);
        } else {
            viewer = name;
        }
        if (viewer == null) {
            return null;
        }
        return getInstaller(NakedObjectClientInstaller.class, viewer);
    }


    //////////////////////////////////////////////////////////
    // Generic Lookups
    //////////////////////////////////////////////////////////
    
    @SuppressWarnings("unchecked")
    public <T extends Installer> T getInstaller(final Class<T> cls, final String implName) {
        if (implName == null) {
            throw new IllegalArgumentException("No name specified");
        }
        for (final Installer installer: installerList) {
            if (cls.isAssignableFrom(installer.getClass()) && 
                installer.getName().equals(implName)) {
                mergeConfigurationFor(installer);
                configure(installer);
                return (T)installer;
            }
        }
        return (T) getInstaller(implName);
    }

    @SuppressWarnings("unchecked")
	public Installer getInstaller(final String implClassName) {
        try {
            Installer installer = CastUtils.cast(InstanceFactory.createInstance(implClassName));
            if (installer != null) {
                mergeConfigurationFor(installer);
                configure(installer);
            }
            return installer;
        } catch (final InstanceCreationException e) {
            throw new InstanceCreationException("Specification error in " + INSTALLER_REGISTRY_FILE, e);
        } catch (final UnavailableClassException e) {
            return null;
        }
    }

    @SuppressWarnings("unchecked")
	public <T extends Installer> T getInstaller(final Class<T> installerCls) {
        try {
            T installer = (T)(InstanceFactory.createInstance(installerCls));
            if (installer != null) {
                mergeConfigurationFor(installer);
                configure(installer);
            }
            return installer;
        } catch (final InstanceCreationException e) {
            throw new InstanceCreationException("Specification error in " + INSTALLER_REGISTRY_FILE, e);
        } catch (final UnavailableClassException e) {
            return null;
        }
    }

    
    //////////////////////////////////////////////////////////
    // InstallerRepository impl.
    //////////////////////////////////////////////////////////

    public Installer[] getInstallers(final Class<?> cls) {
        final List<Installer> list = new ArrayList<Installer>();
        for (final Installer comp: installerList) {
            if (cls.isAssignableFrom(comp.getClass())) {
                list.add(comp);
            }
        }
        return (Installer[]) list.toArray(new Installer[list.size()]);
    }

    
    //////////////////////////////////////////////////////////
    // Helpers
    //////////////////////////////////////////////////////////
    
    
    private <T extends Installer> T getInstaller(Class<T> requiredType, String reqImpl, String key, String defaultImpl) {
        if (reqImpl == null) {
            reqImpl = getConfiguration().getString(key, defaultImpl);
        }
        T installer = getInstaller(requiredType, reqImpl);
        if (installer == null) {
            throw new  InstanceCreationException("Failed to load installer class " + reqImpl + " (of type " + requiredType.getName());
        }
        return installer;
    }


    private void close(final BufferedReader reader) {
        if (reader != null) {
            try {
                reader.close();
            } catch (final IOException e) {
                throw new NakedObjectException(e);
            }
        }
    }

    private InputStream getInstallerRegistryStream(final String componentFile, Class<?> cls) {
        final InputStream in = cls.getResourceAsStream("/" + componentFile);
        if (in == null) {
            throw new NakedObjectException("No resource found: " + componentFile);
        }
        return in;
    }


    
    // ////////////////////////////////////////////////////////////////////
    // injectInto
    // ////////////////////////////////////////////////////////////////////

    public void injectInto(Object candidate) {
        if (NakedObjectConfigurationProviderAware.class.isAssignableFrom(candidate.getClass())) {
            NakedObjectConfigurationProviderAware cast = NakedObjectConfigurationProviderAware.class.cast(candidate);
            cast.setConfigurationProvider(this);
        }
    }

    //////////////////////////////////////////////////////////
    // Configuration
    //////////////////////////////////////////////////////////

    public void setConfigurationLoader(final ConfigurationLoader configurationLoader) {
        this.configurationLoader = configurationLoader;
    }

    public NakedObjectConfiguration getConfiguration() {
        if (cachedConfiguration == null) {
            cachedConfiguration = configurationLoader.load();
        }
        return cachedConfiguration;
    }

    public <T> T configure(T candidate) {
        if (candidate instanceof NakedObjectConfigurationProviderAware) {
            NakedObjectConfigurationProviderAware providerAware = (NakedObjectConfigurationProviderAware) candidate;
            providerAware.setConfigurationProvider(this);
        }
        if (candidate instanceof InstallerLookupAware) {
            InstallerLookupAware installerLookupAware = (InstallerLookupAware) candidate;
            installerLookupAware.setInstallerLookup(this);
        }
        return candidate;
    }

    public void mergeConfigurationFor(Installer installer) {
        if (installer != null) {
            configurationLoader.addConfigurationFile(installer.getName() + ".properties", NotFoundPolicy.CONTINUE);
        }
        cachedConfiguration = null;
    }



}

// Copyright (c) Naked Objects Group Ltd.
