package org.nakedobjects.runtime;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.nakedobjects.metamodel.commons.exceptions.NakedObjectException;
import org.nakedobjects.metamodel.commons.lang.StringUtils;
import org.nakedobjects.metamodel.commons.threads.ThreadRunner;
import org.nakedobjects.metamodel.config.ConfigurationBuilder;
import org.nakedobjects.metamodel.config.ConfigurationBuilderDefault;
import org.nakedobjects.metamodel.config.ConfigurationPrimer;
import org.nakedobjects.runtime.installers.InstallerLookup;
import org.nakedobjects.runtime.installers.InstallerLookupDefault;
import org.nakedobjects.runtime.logging.NakedObjectsLoggingConfigurer;
import org.nakedobjects.runtime.options.BootPrinter;
import org.nakedobjects.runtime.options.OptionHandler;
import org.nakedobjects.runtime.options.standard.OptionHandlerAdditionalProperty;
import org.nakedobjects.runtime.options.standard.OptionHandlerConnector;
import org.nakedobjects.runtime.options.standard.OptionHandlerConfiguration;
import org.nakedobjects.runtime.options.standard.OptionHandlerDebug;
import org.nakedobjects.runtime.options.standard.OptionHandlerDeploymentType;
import org.nakedobjects.runtime.options.standard.OptionHandlerDiagnostics;
import org.nakedobjects.runtime.options.standard.OptionHandlerFixture;
import org.nakedobjects.runtime.options.standard.OptionHandlerHelp;
import org.nakedobjects.runtime.options.standard.OptionHandlerNoSplash;
import org.nakedobjects.runtime.options.standard.OptionHandlerPassword;
import org.nakedobjects.runtime.options.standard.OptionHandlerPersistor;
import org.nakedobjects.runtime.options.standard.OptionHandlerQuiet;
import org.nakedobjects.runtime.options.standard.OptionHandlerReflector;
import org.nakedobjects.runtime.options.standard.OptionHandlerUser;
import org.nakedobjects.runtime.options.standard.OptionHandlerUserProfileStore;
import org.nakedobjects.runtime.options.standard.OptionHandlerVerbose;
import org.nakedobjects.runtime.options.standard.OptionHandlerVersion;
import org.nakedobjects.runtime.options.standard.OptionHandlerViewer;
import org.nakedobjects.runtime.system.DeploymentType;
import org.nakedobjects.runtime.system.NakedObjectsSystem;
import org.nakedobjects.runtime.system.NakedObjectsSystemBootstrapper;
import org.nakedobjects.runtime.system.SystemConstants;
import org.nakedobjects.runtime.viewer.NakedObjectsViewer;
import org.nakedobjects.runtime.viewer.NakedObjectsViewerInstaller;
import org.nakedobjects.runtime.web.EmbeddedWebServer;
import org.nakedobjects.runtime.web.EmbeddedWebServerInstaller;
import org.nakedobjects.runtime.web.WebAppSpecification;


public class NakedObjects {
    private static final String DEFAULT_EMBEDDED_WEBSERVER = SystemConstants.WEBSERVER_DEFAULT;

    public static void main(final String[] args) {
        new NakedObjects().run(args);
    }

	private List<OptionHandler> optionHandlers = new ArrayList<OptionHandler>();
	private OptionHandlerDeploymentType flagHandlerDeploymentType;
	private OptionHandlerConnector flagHandlerClientConnection;
	private OptionHandlerPersistor flagHandlerPersistor;
	private OptionHandlerUser flagHandlerUser;
	private OptionHandlerPassword flagHandlerPassword;
	private OptionHandlerViewer flagHandlerViewer;

    public void run(final String[] args) {
        setupLoggingImmediately(args);
        InstallerLookup installerLookup = new InstallerLookupDefault(NakedObjects.class);

        addOptionHandlers(installerLookup);

        // add options
		final Options options = new Options();
		for (OptionHandler optionHandler : optionHandlers) {
			optionHandler.addOption(options);
		}
        
        // parse & handle flags
        BootPrinter printer = new BootPrinter();
        final CommandLineParser parser = new BasicParser();
        try {
        	CommandLine commandLine = parser.parse(options, args);
            for (OptionHandler optionHandler : optionHandlers) {
    			if (!optionHandler.handle(commandLine, printer, options)) {
    				return;
    			}
    		}
        } catch (final ParseException e) {
            printer.printErrorMessage(e.getMessage());
            printer.printHelp(options);
            return;
        }

        // adhoc validation
        DeploymentType deploymentType = flagHandlerDeploymentType.getDeploymentType();
		List<String> viewerNames = flagHandlerViewer.getViewerNames();
		List<String> connectorNames = flagHandlerClientConnection.getConnectorNames();
		String objectPersistorName = flagHandlerPersistor.getPersistorName();
		String user = flagHandlerUser.getUserName();
		String password = flagHandlerPassword.getPassword();

		if (!validateDeploymentTypeAndConnectors(options, printer, deploymentType, connectorNames)) { return; }
		if (!validateDeploymentTypeAndViewers(options, printer, deploymentType, viewerNames)) { return; }
		if (!validateDeploymentTypeAndPersistor(options, printer, deploymentType, objectPersistorName)) { return; }
		if (!validateUserAndPasswordCombo(options, printer, user, password));

		// prime configuration
		ConfigurationBuilder builder = createConfigurationBuilder();
        for (ConfigurationPrimer optionHandler : optionHandlers) {
        	optionHandler.primeConfigurationBuilder(builder);
		}
        
		// wire up and initialize installer lookup
		builder.injectInto(installerLookup);
		installerLookup.init();
		
        bootstrapSystemAndViewers(installerLookup, deploymentType, viewerNames);
    }

    
    public List<OptionHandler> getFlagHandlers() {
		return Collections.unmodifiableList(optionHandlers);
	}
    
	protected void addOptionHandlers(InstallerLookup installerLookup) {
		
		addOptionHandler(new OptionHandlerConfiguration());
		
		addOptionHandler(flagHandlerClientConnection = new OptionHandlerConnector(installerLookup));
		addOptionHandler(flagHandlerDeploymentType = createOptionHandlerDeploymentType());
		addOptionHandler(flagHandlerPersistor = new OptionHandlerPersistor(installerLookup));
		addOptionHandler(new OptionHandlerReflector(installerLookup));
		addOptionHandler(flagHandlerViewer = new OptionHandlerViewer(installerLookup));
		
		addOptionHandler(flagHandlerUser = new OptionHandlerUser());
		addOptionHandler(flagHandlerPassword = new OptionHandlerPassword());
		addOptionHandler(new OptionHandlerUserProfileStore(installerLookup));
		
		addOptionHandler(new OptionHandlerFixture());
		addOptionHandler(new OptionHandlerNoSplash());
		addOptionHandler(new OptionHandlerAdditionalProperty());
		
		addOptionHandler(new OptionHandlerDebug());
		addOptionHandler(new OptionHandlerDiagnostics());
		addOptionHandler(new OptionHandlerQuiet());
		addOptionHandler(new OptionHandlerVerbose());
		
		
		addOptionHandler(new OptionHandlerHelp());
		addOptionHandler(new OptionHandlerVersion());
		
	}

	protected OptionHandlerDeploymentType createOptionHandlerDeploymentType() {
		return new OptionHandlerDeploymentType();
	}

	protected boolean addOptionHandler(OptionHandler optionHandler) {
		return optionHandlers.add(optionHandler);
	}

	private boolean validateUserAndPasswordCombo(final Options options,
			final BootPrinter printer, final String user, final String password) {
		// check user and password
        if (password == null && user == null) {
    		return true;
    	} 
        printer.printErrorAndHelp(options, "A user name must be specified with a password");
		return false;
	}

	private boolean validateDeploymentTypeAndPersistor(
			final Options options,
			final BootPrinter printer, 
			final DeploymentType deploymentType,
			final String objectPersistor) {
		if (deploymentType.canSpecifyObjectStore() || StringUtils.isEmpty(objectPersistor)) {
    		return true;
    	} 
		printer.printErrorAndHelp(
				options, "Error: cannot specify an object store (persistor) for deployment type %s\n", deploymentType.name().toLowerCase());
		return false;
	}

	private boolean validateDeploymentTypeAndViewers(
			final Options options,
			final BootPrinter printer, 
			final DeploymentType deploymentType,
			List<String> viewers) {
		if (deploymentType.canSpecifyViewers(viewers)) {
    		return true;
    	} 
		printer.printErrorAndHelp(
				options, 
					"Error: cannot specify %s viewer%s for deployment type %s\n",
					(viewers.size() > 1 ? "more than one" : "any"), 
					(viewers.size() > 1 ? "" : "s"),
					deploymentType.name().toLowerCase());
		return false;
	}

	private boolean validateDeploymentTypeAndConnectors(
			final Options options,
			final BootPrinter printer, 
			final DeploymentType deploymentType,
			final List<String> connectors) {
		if (deploymentType.canSpecifyConnectors(connectors)) {
    		return true;
    	} 
		printer.printErrorAndHelp(
				options, 
					"Error: cannot specify %s connector%s for deployment type %s\n",
					(connectors.size() > 1 ? "more than one" : "any"), 
					(connectors.size() > 1 ? "" : "s"),
					deploymentType.name().toLowerCase());
		return false;
	}

	

    private void setupLoggingImmediately(final String[] args) {
        new NakedObjectsLoggingConfigurer().configureLogging(args);
    }

    /**
     * Hook method, possibly overridable.
     * 
     * <p>
     * The default implementation returns a {@link ConfigurationBuilderDefault}, which looks to
     * the <tt>config/</tt> directory, the <tt>src/main/webapp/WEB-INF</tt> directory, and then
     * finally to the classpath.  However, this could be a security concern in a production
     * environment; a user could edit the <tt>nakedobjects.properties</tt> config files to disable security,
     * for example.
     * 
     * <p>
     * This hook method therefore allows this {@link NakedObjects} class to be subclassed and setup 
     * to use a different {@link ConfigurationBuilder}.  For example, a security-conscious subclass
     * could return a {@link ConfigurationBuilder} that only reads from the classpath.  This would
     * allow the application to be deployed as a single sealed JAR that could not be tampered with.
     */
	protected ConfigurationBuilderDefault createConfigurationBuilder() {
		return new ConfigurationBuilderDefault();
	}

	/**
	 * Overridable.
	 */
	protected void bootstrapSystemAndViewers(
			InstallerLookup installerLookup,
			DeploymentType deploymentType, 
			List<String> viewerNames) {
		List<NakedObjectsViewer> viewers = lookupViewers(installerLookup, viewerNames, deploymentType);
		bootstrapSystem(installerLookup, deploymentType);

        // split viewers into web viewers and non-web viewers
        List<NakedObjectsViewer> webViewers = findWebViewers(viewers);
        List<NakedObjectsViewer> nonWebViewers = findNonWebViewers(viewers, webViewers);

        startNonWebViewers(nonWebViewers);
        startWebViewers(installerLookup, webViewers);
	}

	private List<NakedObjectsViewer> lookupViewers(
			InstallerLookup installerLookup, List<String> viewerNames,
			DeploymentType deploymentType) {
		// identify viewer(s) to start
		List<String> requestedViewers = new ArrayList<String>(viewerNames);

        // ask deployment type to ensure viewer installed if need be.
		deploymentType.addViewersIfAnyTo(requestedViewers);

        List<NakedObjectsViewer> viewers = new ArrayList<NakedObjectsViewer>();
        for (String requestedViewer : requestedViewers) {
            String defaultViewer = deploymentType.getDefaultViewer();
            final NakedObjectsViewerInstaller viewerInstaller = installerLookup.viewerInstaller(requestedViewer, defaultViewer);
            if (viewerInstaller == null) {
                throw new NakedObjectException("Can't find viewer: " + requestedViewer);
            }
            final NakedObjectsViewer viewer = viewerInstaller.createViewer();
            viewers.add(viewer);
        }
        return viewers;
	}

    /**
     * Bootstrap the {@link NakedObjectsSystem}, injecting into all {@link NakedObjectsViewer viewer}s.
     */
	private void bootstrapSystem(InstallerLookup installerLookup,
			DeploymentType deploymentType) {
		NakedObjectsSystemBootstrapper bootstrapper = new NakedObjectsSystemBootstrapper(installerLookup);
		bootstrapper.bootSystem(deploymentType);
	}

    private List<NakedObjectsViewer> findWebViewers(List<NakedObjectsViewer> viewers) {
        List<NakedObjectsViewer> webViewers = new ArrayList<NakedObjectsViewer>(viewers);
        CollectionUtils.filter(webViewers, new Predicate() {
            public boolean evaluate(Object object) {
                NakedObjectsViewer viewer = (NakedObjectsViewer) object;
                return viewer.getWebAppSpecification() != null;
            }
        });
        return webViewers;
    }

    private List<NakedObjectsViewer> findNonWebViewers(List<NakedObjectsViewer> viewers, List<NakedObjectsViewer> webViewers) {
        List<NakedObjectsViewer> nonWebViewers = new ArrayList<NakedObjectsViewer>(viewers);
        nonWebViewers.removeAll(webViewers);
        return nonWebViewers;
    }

    /**
     * Starts each (non web) {@link NakedObjectsViewer viewer} in its own thread.
     */
    private void startNonWebViewers(List<NakedObjectsViewer> viewers) {
        for (final NakedObjectsViewer viewer : viewers) {
            Runnable target = new Runnable() {
                public void run() {
                    viewer.init();
                }
            };
            new ThreadRunner().startThread(target, "Viewer");
        }
    }

    /**
     * Starts all the web {@link NakedObjectsViewer viewer}s in an instance of an {@link EmbeddedWebServer}.
     */
    private void startWebViewers(final InstallerLookup installerLookup, final List<NakedObjectsViewer> webViewers) {
        if (webViewers.size() == 0) {
            return;
        }

        // TODO: we could potentially offer pluggability here
        EmbeddedWebServerInstaller webServerInstaller = installerLookup.embeddedWebServerInstaller(DEFAULT_EMBEDDED_WEBSERVER);
        EmbeddedWebServer embeddedWebServer = webServerInstaller.createEmbeddedWebServer();
        for (final NakedObjectsViewer viewer : webViewers) {
            WebAppSpecification webContainerRequirements = viewer.getWebAppSpecification();
            embeddedWebServer.addWebAppSpecification(webContainerRequirements);
        }
        embeddedWebServer.init();
    }


    
    /////////////////////////////////////////////////////////////////////////////////////////



}
// Copyright (c) Naked Objects Group Ltd.
