package org.nakedobjects.runtime.logging;

import java.io.IOException;

import org.apache.log4j.Appender;
import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Layout;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.PropertyConfigurator;
import org.nakedobjects.metamodel.commons.resource.ResourceStreamSource;
import org.nakedobjects.metamodel.commons.resource.ResourceStreamSourceComposite;
import org.nakedobjects.metamodel.commons.resource.ResourceStreamSourceContextLoaderClassPath;
import org.nakedobjects.metamodel.commons.resource.ResourceStreamSourceFileSystem;
import org.nakedobjects.metamodel.config.ConfigurationConstants;
import org.nakedobjects.metamodel.config.internal.PropertiesReader;
import org.nakedobjects.runtime.options.Constants;

public class NakedObjectsLoggingConfigurer {

    /**
     * As per {@link #configureLogging(String, String[])}, and using {@link ConfigurationConstants#DEFAULT_CONFIG_DIRECTORY} as the
     * directory.
     * 
     * <p>
     * Only intended to be called from <tt>main()</tt>, so uses the
     * {@link ConfigurationConstants#DEFAULT_CONFIG_DIRECTORY default config directory}.
     */
    public void configureLogging(String[] args) {
        configureLogging(ConfigurationConstants.DEFAULT_CONFIG_DIRECTORY, args);
    }

    /**
     * As per {@link #configureLogging(String)}.
     * 
     * <p>
     * The root logging level can also be adjusted using command line arguments.
     * @param args 
     */
    public void configureLogging(final String configDirectory, String[] args) {
        configureLogging(configDirectory);

        // override if specified by command line arguments
        Level loggingLevel = loggingLevel(args);
		if (loggingLevel != null) {
			Logger.getRootLogger().setLevel(loggingLevel);
		}
    }

    /**
     * Sets up logging using either a logging file or (if cannot be found) some sensible defaults.
     * 
     * <p>
     * If a {@link LoggingConstants#LOGGING_CONFIG_FILE logging config file} can be located in the provided
     * directory, then that is used.  Otherwise, will set up the 
     * {@link Logger#getRootLogger() root logger} to {@link Level#WARN warning}, a
     * typical {@link PatternLayout} and logging to the {@link ConsoleAppender console}.
     */
	public void configureLogging(final String configDirectory) {
		configureLogging(
				new ResourceStreamSourceComposite(
						new ResourceStreamSourceFileSystem(configDirectory), 
						new ResourceStreamSourceContextLoaderClassPath()));
	}

	public void configureLogging(ResourceStreamSource resourceStreamSource) {
		try {
			PropertiesReader propertiesReader = new PropertiesReader(resourceStreamSource, LoggingConstants.LOGGING_CONFIG_FILE);
			PropertyConfigurator.configure(propertiesReader.getProperties());
		} catch (IOException ex) {
            configureFallbackLogging();
		}
	}

	private void configureFallbackLogging() {
		final Layout layout = new PatternLayout("%-5r [%-25.25c{1} %-10.10t %-5.5p]  %m%n");
		final Appender appender = new ConsoleAppender(layout);
		BasicConfigurator.configure(appender);
		Logger.getRootLogger().setLevel(Level.WARN);
		Logger.getLogger("ui").setLevel(Level.OFF);
	}


	private Level loggingLevel(final String[] args) {
		Level level = null;
		for (int i = 0; i < args.length; i++) {
			if (args[i].equals("-" + Constants.DEBUG_OPT)) {
				level = Level.DEBUG;
				break;
			} else if (args[i].equals("-" + Constants.QUIET_OPT)) {
				level = Level.ERROR;
				break;
			} else if (args[i].equals("-" + Constants.VERBOSE_OPT)) {
				level = Level.INFO;
				break;
			}
		}
		return level;
	}

}


// Copyright (c) Naked Objects Group Ltd.
