
/*
#
# Copyright 2013 OW2 Nanoko Project
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
*/


/*
# Hub Class
*/


/**
 The Hub Class.
 @class HUBU.Hub
 @classdesc The main **Hub** class. Each instance of this class is a _hub_ and so is able to receive components.
 All components need to be plugged to a hub to be _active_. This is the central piece of the h-ubu system.
 Hub are also components so can be plugged to other hubs.
*/


(function() {
  var Hub;

  HUBU.Hub = Hub = (function() {
    /**
    The component plugged to the hub.
    @type {Array}
    @memberOf HUBU.Hub
    @name #_components
    @private
    */

    Hub.prototype._components = null;

    /**
    Is the hub started.
    @name HUBU.Hub#_started
    @private
    */


    Hub.prototype._started = false;

    /**
    The list of extensions plugged on this hub.
    The extensions are created on the first hub access (either `start` or `registerComponent`)
    @name HUBU.Hub#_extensions
    @private
    */


    Hub.prototype._extensions = null;

    /**
    The parent hub, if set. The parent is given during the configure method.
    @name HUBU.Hub#_parentHub
    @private
    */


    Hub.prototype._parentHub = null;

    /**
    The hub name if set. `hub` by default. The root hub is named `root`.
    @name HUBU.Hub#_name
    @private
    */


    Hub.prototype._name = null;

    /**
    The hub constructor.
    */


    function Hub() {
      this._components = [];
      this._started = false;
      this._extensions = null;
    }

    /**
    Configures the hub. This method initializes all extensions if not already done.
    @method
    @name HUBU.Hub#configure
    @param {HUBU.Hub} parent the parent hub if exists. Sub-hubs have necessary one and only one parent hub.
    @param configuration optional parameter used to pass the component configuration. The configuration object is a simple
     key/value map.  @returns {HUBU.Hub} the hub
    */


    Hub.prototype.configure = function(parent, configuration) {
      var ext, name, _ref;
      if ((parent != null)) {
        this._parentHub = parent;
      }
      if (!(this._name != null)) {
        this._name = ((configuration != null ? configuration.component_name : void 0) != null) ? configuration.component_name : "hub";
      }
      if (!(this._extensions != null)) {
        this._extensions = [];
        _ref = getHubuExtensions();
        for (name in _ref) {
          ext = _ref[name];
          this._extensions.push(new ext(this));
        }
      } else {
        HUBU.logger.debug("Hub already initialized");
      }
      return this;
    };

    /**
    Gets the parent hub if set
    @method
    @name HUBU.Hub#getParentHub
    @returns {boolean} the parent hub is set, `null` otherwise.
    */


    Hub.prototype.getParentHub = function() {
      return this._parentHub;
    };

    /**
    Gets all plugged components.
    *Do not modified the result !*
    @method
    @name HUBU.Hub#getComponents
    @return {Array} the list of plugged components on the current hub.
    */


    Hub.prototype.getComponents = function() {
      return this._components;
    };

    /**
    Looks for one specific component plugged to the hub.
    This lookup is based on the component 'getComponentName' method.
    @method
    @name HUBU.Hub#getComponent
    @param {String} name the component name
    @return {HUBU.AbstractComponent} the component with the matching name or `null` if the component is not plugged.
    */


    Hub.prototype.getComponent = function(name) {
      var cmp, fc, n, _i, _len, _ref;
      if (!(name != null)) {
        return null;
      }
      _ref = this._components;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        cmp = _ref[_i];
        fc = cmp.getComponentName;
        if ((fc != null) && HUBU.UTILS.isFunction(fc)) {
          n = fc.apply(cmp, []);
          if (n === name) {
            return cmp;
          }
        }
      }
      return null;
    };

    /**
    Registers a new component on the hub.
    If the component already exists, the registration is ignored. The lookup is based on the `getComponentName` method.
    This method allows to configure the component.Once successfully registered, the hub call the 'configure' method on
    the component passing a reference on the hub and the configuration to the component.
    If component is `null`, the method throws an exception.
    @method
    @name HUBU.Hub#registerComponent
    @param {HUBU.AbstractComponent} component the component to register
    @param {Object} configuration the component configuration (optional).
    If the configuration contain the `component_name` key, the component takes this name.
    @return {HUBU.Hub} the current hub
    */


    Hub.prototype.registerComponent = function(component, configuration) {
      /* Validation
      */

      var ext, _i, _len, _ref;
      if (!(component != null)) {
        throw new Exception("Cannot register component - component is null");
      }
      if (!HUBU.UTILS.isComponent(component)) {
        if (component.getComponentName) {
          throw new Exception(component.getComponentName() + " is not a valid component");
        } else {
          throw new Exception(component + " is not a valid component");
        }
      }
      /* End of Validation
      */

      if (this._extensions === null) {
        this.configure();
      }
      if (this.getComponent(component.getComponentName()) != null) {
        HUBU.logger.info("Component " + component.getComponentName() + " already registered");
        return this;
      }
      this._components.push(component);
      if ((configuration != null) && (configuration.component_name != null)) {
        component["__name__"] = configuration.component_name;
        component.getComponentName = function() {
          return this["__name__"];
        };
      }
      if (!(component.__hub__ != null) && !(component.hub != null)) {
        component.__hub__ = this;
        component.hub = function() {
          return this.__hub__;
        };
      }
      HUBU.logger.debug("Registering component " + component.getComponentName());
      _ref = this._extensions;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        ext = _ref[_i];
        HUBU.UTILS.invoke(ext, "registerComponent", [component, configuration]);
      }
      HUBU.logger.debug("Configuring component " + component.getComponentName());
      component.configure(this, configuration);
      if (this._started) {
        HUBU.logger.debug("Starting component " + component.getComponentName());
        component.start();
      }
      HUBU.logger.debug("Component " + component.getComponentName() + " registered");
      return this;
    };

    /**
    Creates an instance of the given factory.
    This methods delegates most of the work on the _registerComponent_ method but create the component out of the given
    factory. The return of the factory must be a valid component.
    @name HUBU.Hub#createInstance
    @method
    @param {Function} factory the method used to create the instance. The return of this method must be a valid component,
    so be conform to the {HUBU.AbstractComponent} contract.
    @param {Object} configuration the component configuration (optional).
    If the configuration contain the `component_name` key, the component takes this name.
    @return {HUBU.Hub} the current hub
    */


    Hub.prototype.createInstance = function(factory, configuration) {
      /* Validation
      */

      var instance;
      if (!(factory != null)) {
        throw new Exception("Cannot create instance - the given factory / constructor is null");
      }
      if (!HUBU.UTILS.isFunction(factory)) {
        throw new Exception("Cannot create instance - the given factory " + "/ constructor is not a function");
      }
      /* End of validation
      */

      instance = new factory();
      return this.registerComponent(instance, configuration);
    };

    /**
    Unregisters the given component.
    If the component is not plugged to the hub, this method does nothing.
    @name HUBU.Hub#unregisterComponent
    @method
    @param {Object} component either the component object ({HUBU.AbstractComponent}) or the component name {String}
    @return {HUBU.Hub} the current hub.
    */


    Hub.prototype.unregisterComponent = function(component) {
      var cmp, ext, idx, _i, _len, _ref;
      if (!(component != null)) {
        return this;
      }
      cmp = null;
      if (HUBU.UTILS.typeOf(component) === "string") {
        cmp = this.getComponent(component);
        if (!(cmp != null)) {
          return this;
        }
      } else {
        if (!HUBU.UTILS.isComponent(component)) {
          throw new Exception("Cannot unregister component, it's not a valid component").add("component", component);
        } else {
          cmp = component;
        }
      }
      if (this._extensions === null) {
        this.configure();
      }
      idx = HUBU.UTILS.indexOf(this._components, cmp);
      if (idx !== -1) {
        _ref = this._extensions;
        for (_i = 0, _len = _ref.length; _i < _len; _i++) {
          ext = _ref[_i];
          HUBU.UTILS.invoke(ext, "unregisterComponent", [cmp]);
        }
        cmp.stop();
        this._components.splice(idx, 1);
      } else {
        HUBU.logger.info("Component " + cmp.getComponentName() + " not unregistered - not on the hub");
      }
      return this;
    };

    /**
    Starts the hub.
    This method calls start on all plugged components.
    This method does nothing is the hub is already started.
    @method
    @name HUBU.Hub#start
    @return {HUBU.Hub} the hub
    */


    Hub.prototype.start = function() {
      var cmp, ext, _i, _j, _len, _len1, _ref, _ref1;
      if (this._started) {
        return this;
      }
      if (this._extensions === null) {
        this.configure();
      }
      _ref = this._extensions;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        ext = _ref[_i];
        HUBU.UTILS.invoke(ext, "start", []);
      }
      this._started = true;
      _ref1 = this._components;
      for (_j = 0, _len1 = _ref1.length; _j < _len1; _j++) {
        cmp = _ref1[_j];
        cmp.start();
      }
      return this;
    };

    /**
    Stops the hub.
    This method calls stop on all plugged components.
    If the hub is not started, this methods does nothing.
    @method
    @name HUBU.Hub#stop
    @return {HUBU.Hub} the hub
    */


    Hub.prototype.stop = function() {
      var cmp, ext, _i, _j, _len, _len1, _ref, _ref1;
      if (!this._started) {
        return this;
      }
      this._started = false;
      _ref = this._components;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        cmp = _ref[_i];
        cmp.stop();
      }
      _ref1 = this._extensions;
      for (_j = 0, _len1 = _ref1.length; _j < _len1; _j++) {
        ext = _ref1[_j];
        HUBU.UTILS.invoke(ext, "start", []);
      }
      return this;
    };

    /**
    Checks whether the hub is started.
    @method
    @name HUBU.Hub#isStarted
    @return {boolean} `true` is the hub is started, `false` otherwise
    */


    Hub.prototype.isStarted = function() {
      return this._started;
    };

    /**
    Resets the hub.
    This method is generally used for testing as it reinitializes the hub state.
    @method
    @name HUBU.Hub#reset
    @return {HUBU.Hub} the current hub
    */


    Hub.prototype.reset = function() {
      var ext, name, _i, _len, _ref;
      this.stop();
      name = this._name;
      if (this._extensions === null) {
        this.configure();
      }
      _ref = this._extensions;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        ext = _ref[_i];
        HUBU.UTILS.invoke(ext, "reset", []);
      }
      this._components = [];
      this._extensions = null;
      this._name = name;
      return this;
    };

    /**
    Gets the hub name.
    @name HUBU.Hub#getComponentName
    @method
    @return {String} the hub's name
    */


    Hub.prototype.getComponentName = function() {
      return this._name;
    };

    return Hub;

  })();

  /* End of th Hub Class
  */


  /**
  Create the main Global hub, and the `hub` alias
  @desc The main global hub.
  @global
  @readonly
  */


  getGlobal().hub = new HUBU.Hub().configure(null, {
    component_name: "root"
  });

}).call(this);
