
/*
#
# Copyright 2013 OW2 Nanoko Project
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
*/


/*
# Hubu Eventing Extension
# This extension manages the event communications between components.
# It is recommended to use `topics` instead of direct events.
*/


(function() {
  var Eventing,
    __slice = [].slice;

  HUBU.Eventing = Eventing = (function() {

    Eventing.prototype._hub = null;

    Eventing.prototype._listeners = null;

    function Eventing(hubu) {
      var myExtension;
      this._hub = hubu;
      this._listeners = [];
      myExtension = this;
      /*
          # Gets the registered event listeners.
          # *Do not modified the result !*
          # @return the list of registered listeners.
      */

      this._hub.getListeners = function() {
        return myExtension._listeners;
      };
      /*
          # Registers an event listener.
          # This method can take either 2 or 3 arguments. The first one is *always* the component, so two signatures are possible:
          #
          # * `registerListener(component, match, callback)` where `match` is the matching function and `callback` is the
          # reception callback.
          # * `registerListener(component, conf)` where `conf` is an object containing `match` and `callback` specifying
          # respectively the event matching method and the reception callback.
          #
          # @param {HUBU.AbstractComponent} component : the component registering the listener
          # @param {Function} match : the method called to check if the event matches. This method must returns true or false.
          # @param {Function} callback : the callback method to invoke when a matching event is sent
          # @return the hub
      */

      this._hub.registerListener = function() {
        var component, conf;
        component = arguments[0], conf = 2 <= arguments.length ? __slice.call(arguments, 1) : [];
        if (conf.length >= 2) {
          myExtension.registerListener(component, conf[0], conf[1]);
          return this;
        } else {
          myExtension.registerListener(component, conf[1]);
          return this;
        }
      };
      /*
          # *Deprecated method*, use `registerListener` instead.
      */

      this._hub.registerConfigurableListener = function(component, conf) {
        HUBU.logger.warn("registerConfigurableListener is a deprecated method and may disappear at any time, use registerListener instead");
        myExtension.registerListener(component, conf);
        return this;
      };
      /*
          # Unregisters listeners for the given component. According to the arguments, several cases occur:
          # `component` can be either a String of a Component. In case of the string, we look up for the component using
          # `getComponentName`.
          # If `callback` is defined, only the listener matching component and callback will be unregistered. Otherwise all
          # listeners of the component will be unregistered.
          # If `component` is `null`, this methods does nothing.
          # @param {HUBU.AbstractComponent} component the component
          # @param {Function} callback the callback function (optional)
          # @return the current hub
      */

      this._hub.unregisterListener = function(component, callback) {
        myExtension.unregisterListener(component, callback);
        return this;
      };
      /*
          # Sends an event inside the hub. If component or event is null, the method does nothing. If not, the event processed
          # and sent to all matching listeners.
          # @param {HUBU.AbstractComponent} component the component sending the event
          # @param {Object} event the event
          # @return true if the event was delivered to at least one component, false otherwise
          # @methodOf HUBU.hubu
      */

      this._hub.sendEvent = function(component, event) {
        return myExtension.sendEvent(component, event);
      };
      /*
          # Subscribes to a specific topic.
          # @param {HUBU.AbstractComponent} component : the component registering the listener
          # @param {String} topic : the topic (Regexp)
          # @param {Function} callback : the callback method to invoke when a matching event is sent
          # @param {Function} filter : optional method to filter received events.
          # @return the current hub
          # @methodOf HUBU.hubu
      */

      this._hub.subscribe = function(component, topic, callback, filter) {
        myExtension.subscribe(component, topic, callback, filter);
        return this;
      };
      /*
          # Unsubscribes the subscriber.
          # @param {Object} component the component
          # @param {Function} callback the registered callback
          # @methodOf HUBU.hubu
          # @return the current hub
      */

      this._hub.unsubscribe = function(component, callback) {
        myExtension.unsubscribe(component, callback);
        return this;
      };
      /*
          # Publishes an event to a specific topic. If component, topic or event is null, the method does nothing. If not,
          # the event is processed and sent to all matching listeners.
          # @param {HUBU.AbstractComponent} component the component sending the event
          # @param {String} topic the topic
          # @param {Object} event the event
          # @return true if the event was delivered to at least one component, false otherwise
          # @methodOf HUBU.hubu
      */

      this._hub.publish = function(component, topic, event) {
        myExtension.publish(component, topic, event);
        return this;
      };
    }

    /* End of constructor
    */


    /*
      # Processes the given event sent by the given component. This methods checks who is interested by the event by
      # calling the match method, then the callback method is called.
      # The event is extended with the 'source' property indicating the component sending the event.
      # @param {Object} event
      # @param {AbstractComponent} component
      # @return true if the event was consumed by at least one component, false otherwise.
    */


    Eventing.prototype._processEvent = function(component, event) {
      var ev, listener, sent, _i, _len, _ref;
      if (!(event != null) || !(component != null)) {
        HUBU.logger.warn("Can't process event - component or event not defined");
        return false;
      }
      sent = false;
      _ref = this._listeners.slice();
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        listener = _ref[_i];
        if (!(listener.component !== component && HUBU.UTILS.indexOf(this._listeners, listener) !== -1)) {
          continue;
        }
        ev = !(event.clone != null) || event.clone ? HUBU.UTILS.clone(event) : event;
        ev.source = component;
        if (listener.match.apply(listener.component, [ev])) {
          listener.callback.apply(listener.component, [ev]);
          sent = true;
        }
      }
      return sent;
    };

    Eventing.prototype.registerListener = function() {
      var callback, component, listener, match, others;
      component = arguments[0], others = 2 <= arguments.length ? __slice.call(arguments, 1) : [];
      match = null;
      callback = null;
      switch (others.length) {
        case 2:
          match = others[0];
          callback = others[1];
          break;
        case 1:
          match = others[0].match;
          callback = others[0].callback;
      }
      if (!(component != null) || !(match != null) || !(callback != null)) {
        throw new Exception("Cannot register event listener, component or match or callback is/are not defined").add("component", component).add("match", match).add("callback", callback);
      }
      if (!HUBU.UTILS.isComponentPlugged(component, this._hub)) {
        throw new Exception("Cannot register event listener, the component is not plugged on the hub");
      }
      listener = {
        "component": component,
        "callback": callback,
        "match": match
      };
      return this._listeners.push(listener);
    };

    Eventing.prototype.unregisterListener = function(component, callback) {
      var cmp, listener, toRemove, _i, _j, _k, _len, _len1, _len2, _ref, _ref1, _results;
      if (!(component != null)) {
        HUBU.logger.warn("Cannot unregister listener - component not defined");
        return false;
      }
      cmp = this.getComponent(component);
      if (!(cmp != null)) {
        HUBU.logger.warn("Cannot unregister listener - component not plugged on the hub");
        return false;
      }
      toRemove = [];
      if ((callback != null)) {
        _ref = this._listeners;
        for (_i = 0, _len = _ref.length; _i < _len; _i++) {
          listener = _ref[_i];
          if (listener.component === cmp && listener.callback === callback) {
            toRemove.push(listener);
          }
        }
      } else {
        _ref1 = this._listeners;
        for (_j = 0, _len1 = _ref1.length; _j < _len1; _j++) {
          listener = _ref1[_j];
          if (listener.component === cmp) {
            toRemove.push(listener);
          }
        }
      }
      _results = [];
      for (_k = 0, _len2 = toRemove.length; _k < _len2; _k++) {
        listener = toRemove[_k];
        _results.push(this._listeners = HUBU.UTILS.removeElementFromArray(this._listeners, listener));
      }
      return _results;
    };

    /*
      # Helper method retriving a component object from the given argument.
      # If the argument is a String, it performs a lookup by name
      # If the argument is a component object, it just checks the conformity
    */


    Eventing.prototype.getComponent = function(obj) {
      if (HUBU.UTILS.typeOf(obj) === "string") {
        return this._hub.getComponent(obj);
      }
      if (HUBU.UTILS.isComponent(obj)) {
        return obj;
      }
      return null;
    };

    Eventing.prototype.sendEvent = function(component, event) {
      if (!(component != null) || !(event != null)) {
        HUBU.logger.warn("Cannot send event, component or/and event are undefined");
        return;
      }
      return this._processEvent(component, event);
    };

    Eventing.prototype.subscribe = function(component, topic, callback, filter) {
      var match, regex;
      if (!(component != null) || !(topic != null) || !(callback != null)) {
        HUBU.logger.warn("Cannot subscribe to topic, component or/and topic and/or callback are undefined");
        return;
      }
      regex = new RegExp(topic);
      match = null;
      if (!(filter != null) || !HUBU.UTILS.isFunction(filter)) {
        match = function(event) {
          return regex.test(event.topic);
        };
      } else {
        match = function(event) {
          return regex.test(event.topic) && filter(event);
        };
      }
      return this.registerListener(component, match, callback);
    };

    Eventing.prototype.unsubscribe = function(component, callback) {
      return this.unregisterListener(component, callback);
    };

    Eventing.prototype.publish = function(component, topic, event) {
      if (!(component != null) || !(topic != null) || !(event != null)) {
        HUBU.logger.info("Cannot publish event - component and/or topic and/or event are missing");
        return false;
      }
      event.topic = topic;
      return this.sendEvent(component, event);
    };

    Eventing.prototype.reset = function() {
      return this._listeners = [];
    };

    Eventing.prototype.unregisterComponent = function(cmp) {
      return this.unregisterListener(cmp);
    };

    return Eventing;

  })();

  /* End of the Eventing class
  */


  getHubuExtensions().eventing = Eventing;

}).call(this);
