
/*
#
# Copyright 2013 OW2 Nanoko Project
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
*/


/*
# Hubu Service-Orientation Extension
*/


/**
 The Service Component class
 @class HUBU.ServiceComponent
 @classdesc This class represents _Service Components_. Service Components are one of the main concept of h-ubu. It
represents the published and required services for a specific h-ubu's component.

  The constructor.
  Initializes the service component. By default, there are no service dependencies and no provided services.
 @param {HUBU.AbstractComponent} component the underlying component.
*/


(function() {
  var ProvidedService, ServiceComponent, ServiceDependency, ServiceOrientation;

  HUBU.ServiceComponent = ServiceComponent = (function() {
    /**
    _STOPPED_ state.
    A stopped service component does not published any services, and required services are not tracked and injected.
    It's also the initial state of service components.
    @type {Number}
    @memberOf HUBU.ServiceComponent
    @name STOPPED
    */

    ServiceComponent.STOPPED = 0;

    /**
    _INVALID_ state.
    A service component is invalid if a mandatory service dependency is not resolved.
    An invalid service component does not publish its services but required services are tracked.
    Once started, service component are in this state.
    @type {Number}
    @memberOf HUBU.ServiceComponent
    @name INVALID
    */


    ServiceComponent.INVALID = 1;

    /**
    _VALID_ state.
    A service component is valid if all mandatory service dependencies are resolved.
    A valid service component publishes its services, required services are tracked and injected.
    The service component stays in this state as long as all mandatory services dependencies are resolved.
    @type {Number}
    @memberOf HUBU.ServiceComponent
    @name VALID
    */


    ServiceComponent.VALID = 2;

    /**
    The underlying component.
    @type {HUBU.AbstractComponent}
    @memberOf HUBU.ServiceComponent
    @name #_component
    @private
    */


    ServiceComponent.prototype._component = null;

    /**
    The provided services.
    @type {HUBU.ProvidedService}
    @memberOf HUBU.ServiceComponent
    @name #_providedServices
    @private
    */


    ServiceComponent.prototype._providedServices = null;

    /**
    The required services.
    @type {HUBU.ServiceDependency}
    @memberOf HUBU.ServiceComponent
    @name #_requiredServices
    @private
    */


    ServiceComponent.prototype._requiredServices = null;

    /**
    The current state of the service component.
    @type {Number}
    @memberOf HUBU.ServiceComponent
    @name #_state
    @private
    */


    ServiceComponent.prototype._state = 0;

    function ServiceComponent(component) {
      this._component = component;
      this._providedServices = [];
      this._requiredServices = [];
      this._state = ServiceComponent.STOPPED;
    }

    /**
    Gets the underlying components
    @method
    @memberOf HUBU.ServiceComponent
    @name #getComponent
    @returns {HUBU.AbstractComponent} the underlying component
    */


    ServiceComponent.prototype.getComponent = function() {
      return this._component;
    };

    /**
    Gets the current state
    @method
    @memberOf HUBU.ServiceComponent
    @name #getState
    @returns {Number} the current state
    */


    ServiceComponent.prototype.getState = function() {
      return this._state;
    };

    /**
    Adds a provided service.
    Dependending on the current state the provided service is started, validated or invalidated
    @method
    @memberOf HUBU.ServiceComponent
    @name #addProvidedService
    @param {HUBU.ProvidedService} ps the provided service to add.
    */


    ServiceComponent.prototype.addProvidedService = function(ps) {
      if (HUBU.UTILS.indexOf(this._providedServices, ps) === -1) {
        this._providedServices.push(ps);
        ps.setServiceComponent(this);
        if (this._state > ServiceComponent.STOPPED) {
          ps.onStart();
        }
        if (this._state === ServiceComponent.VALID) {
          ps.onValidation();
        }
        if (this._state === ServiceComponent.INVALID) {
          return ps.onInvalidation();
        }
      }
    };

    /**
    Removes a provided service. Does nothing if the provided service is not found. If found the provided service is stopped.
    @method
    @memberOf HUBU.ServiceComponent
    @name #removeProvidedService
    @param {HUBU.ProvidedService} ps the provided service to add.
    */


    ServiceComponent.prototype.removeProvidedService = function(ps) {
      if (HUBU.UTILS.indexOf(this._providedServices, ps) !== -1) {
        HUBU.UTILS.removeElementFromArray(this._providedServices, ps);
        return ps.onStop();
      }
    };

    /**
    Adds a required service.
    Depending on the current state, the dependency is started.
    @method
    @memberOf HUBU.ServiceComponent
    @name #addRequireService
    @param {HUBU.ServiceDependency} req the service dependency to add
    */


    ServiceComponent.prototype.addRequiredService = function(req) {
      if (HUBU.UTILS.indexOf(this._requiredServices, req) === -1) {
        this._requiredServices.push(req);
        req.setServiceComponent(this);
        if (this._state > ServiceComponent.STOPPED) {
          req.onStart();
          return this.computeState();
        }
      }
    };

    /**
    Removes a service dependency.
    The dependency is stopped, the current state is recomputed.
    If the dependency is not found, this method does nothing.
    @method
    @memberOf HUBU.ServiceComponent
    @name #removeRequiredService
    @param {HUBU.ProvidedService} ps the provided service to add.
    */


    ServiceComponent.prototype.removeRequiredService = function(req) {
      if (HUBU.UTILS.indexOf(this._requiredServices, req) > -1) {
        HUBU.UTILS.removeElementFromArray(this._requiredServices, req);
        req.onStop();
        if (this._state > ServiceComponent.STOPPED) {
          return this.computeState();
        }
      }
    };

    /**
    Computes the state of the current service component.
    The state is valid if and only if all mandatory required services are fulfilled.
    If there is a transition the _validate_ and _invalidate_ callbacks are called.
    @method
    @memberOf HUBU.ServiceComponent
    @name #computeState
    @returns {Number} the new state
    */


    ServiceComponent.prototype.computeState = function() {
      var isValid, oldState, req, _i, _len, _ref;
      isValid = true;
      _ref = this._requiredServices;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        req = _ref[_i];
        isValid = isValid && req.isValid();
      }
      oldState = this._state;
      this._state = isValid ? ServiceComponent.VALID : ServiceComponent.INVALID;
      if (this._state > oldState && this._state === ServiceComponent.VALID) {
        this._validate();
      } else if (this._state < oldState && this._state === ServiceComponent.INVALID) {
        this._invalidate();
      }
      return this._state;
    };

    /**
    Validates the service component.
    Invokes _onValidation_ on all provided service.
    @method
    @memberOf HUBU.ServiceComponent
    @name #_validate
    @private
    */


    ServiceComponent.prototype._validate = function() {
      var prov, _i, _len, _ref, _ref1, _results;
      HUBU.logger.debug("Validate instance " + ((_ref = this._component) != null ? _ref.getComponentName() : void 0));
      _ref1 = this._providedServices;
      _results = [];
      for (_i = 0, _len = _ref1.length; _i < _len; _i++) {
        prov = _ref1[_i];
        _results.push(prov.onValidation());
      }
      return _results;
    };

    /**
    Invalidates the service component.
    Invokes _onInvalidation_ on all provided service.
    @method
    @memberOf HUBU.ServiceComponent
    @name #_invalidate
    @private
    */


    ServiceComponent.prototype._invalidate = function() {
      var prov, _i, _len, _ref, _results;
      HUBU.logger.debug("Invalidate instance");
      _ref = this._providedServices;
      _results = [];
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        prov = _ref[_i];
        _results.push(prov.onInvalidation());
      }
      return _results;
    };

    /**
    Starting callback.
    @method
    @memberOf HUBU.ServiceComponent
    @name #onStart
    */


    ServiceComponent.prototype.onStart = function() {
      var prov, req, _i, _j, _len, _len1, _ref, _ref1;
      _ref = this._requiredServices;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        req = _ref[_i];
        req.onStart();
      }
      _ref1 = this._providedServices;
      for (_j = 0, _len1 = _ref1.length; _j < _len1; _j++) {
        prov = _ref1[_j];
        prov.onStart();
      }
      return this.computeState();
    };

    /**
    Stopping callback.
    @method
    @memberOf HUBU.ServiceComponent
    @name #onStop
    */


    ServiceComponent.prototype.onStop = function() {
      var prov, req, _i, _j, _len, _len1, _ref, _ref1;
      _ref = this._providedServices;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        prov = _ref[_i];
        prov.onStop();
      }
      _ref1 = this._requiredServices;
      for (_j = 0, _len1 = _ref1.length; _j < _len1; _j++) {
        req = _ref1[_j];
        req.onStop();
      }
      return this._state = ServiceComponent.STOPPED;
    };

    /**
    Gets a service dependency by name
    @method
    @memberOf HUBU.ServiceComponent
    @name #getServiceDependencyByName
    @param {String} name the dependency
    @return {HUBU.ServiceDependency} the service dependency, `null` if no service dependencies match the name
    */


    ServiceComponent.prototype.getServiceDependencyByName = function(name) {
      var dep, _i, _len, _ref;
      _ref = this._requiredServices;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        dep = _ref[_i];
        if (dep.getName() === name) {
          return dep;
        }
      }
    };

    return ServiceComponent;

  })();

  HUBU.ServiceDependency = ServiceDependency = (function() {
    var _listener, _refs, _serviceComponent, _state;

    ServiceDependency.UNRESOLVED = 0;

    ServiceDependency.RESOLVED = 1;

    ServiceDependency.prototype._component = null;

    ServiceDependency.prototype._contract = null;

    ServiceDependency.prototype._filter = null;

    ServiceDependency.prototype._aggregate = false;

    ServiceDependency.prototype._optional = false;

    ServiceDependency.prototype._field = null;

    ServiceDependency.prototype._bind = null;

    ServiceDependency.prototype._unbind = null;

    ServiceDependency.prototype._name = null;

    ServiceDependency.prototype._hub = null;

    _listener = null;

    _state = null;

    _refs = [];

    _serviceComponent = null;

    function ServiceDependency(component, contract, filter, aggregate, optional, field, bind, unbind, name, hub) {
      var self;
      this._component = component;
      this._contract = contract;
      this._filter = filter;
      this._aggregate = aggregate;
      this._optional = optional;
      this._field = field;
      this._name = name != null ? name : this._contract;
      if (bind != null) {
        this._bind = HUBU.UTILS.isFunction(bind) ? bind : this._component[bind];
        if (!(this._bind != null)) {
          throw new Exception("Bind method " + bind + " not found on component");
        }
      }
      if (unbind != null) {
        this._unbind = HUBU.UTILS.isFunction(unbind) ? unbind : this._component[unbind];
        if (!(this._unbind != null)) {
          throw new Exception("Unbind method " + unbind + " not found on component");
        }
      }
      this._hub = hub;
      this._state = HUBU.ServiceDependency.UNRESOLVED;
      this._refs = [];
      self = this;
      this._listener = {
        contract: this._contract,
        filter: function(ref) {
          return ref.getProperty("service.publisher") !== self._component && (!(self._filter != null) || self._filter(ref));
        },
        listener: function(event) {
          switch (event.getType()) {
            case SOC.ServiceEvent.REGISTERED:
              return self._onServiceArrival(event.getReference());
            case SOC.ServiceEvent.MODIFIED:
              return self._onServiceModified(event.getReference());
            case SOC.ServiceEvent.UNREGISTERING:
              return self._onServiceDeparture(event.getReference());
            case SOC.ServiceEvent.MODIFIED_ENDMATCH:
              return self._onServiceDeparture(event.getReference());
          }
        }
      };
    }

    /* End Constructor
    */


    ServiceDependency.prototype.setServiceComponent = function(sc) {
      return this._serviceComponent = sc;
    };

    ServiceDependency.prototype.onStart = function() {
      this._state = HUBU.ServiceDependency.UNRESOLVED;
      this._startTracking();
      return this._computeDependencyState();
    };

    ServiceDependency.prototype.onStop = function() {
      this._stopTracking();
      this._ungetAllServices();
      this._refs = [];
      return this._state = HUBU.ServiceDependency.UNRESOLVED;
    };

    ServiceDependency.prototype._ungetAllServices = function() {
      var entry, _i, _len, _ref, _results;
      _ref = this._refs;
      _results = [];
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        if (!(entry.service != null)) {
          continue;
        }
        entry.service = null;
        _results.push(this._hub.ungetService(this._component, entry.reference));
      }
      return _results;
    };

    ServiceDependency.prototype._startTracking = function() {
      var ref, refs, _i, _len, _results;
      this._hub.registerServiceListener(this._listener);
      refs = this._hub.getServiceReferences(this._contract, this._filter);
      _results = [];
      for (_i = 0, _len = refs.length; _i < _len; _i++) {
        ref = refs[_i];
        _results.push(this._onServiceArrival(ref));
      }
      return _results;
    };

    ServiceDependency.prototype._stopTracking = function() {
      return this._hub.unregisterServiceListener(this._listener);
    };

    ServiceDependency.prototype.isValid = function() {
      return this._state === HUBU.ServiceDependency.RESOLVED;
    };

    ServiceDependency.prototype.getName = function() {
      return this._name;
    };

    ServiceDependency.prototype.getContract = function() {
      return this._contract;
    };

    ServiceDependency.prototype.getFilter = function() {
      return this._filter;
    };

    ServiceDependency.prototype.isAggregate = function() {
      return this._aggregate;
    };

    ServiceDependency.prototype.isOptional = function() {
      return this._optional;
    };

    ServiceDependency.prototype._computeDependencyState = function() {
      var oldState;
      oldState = this._state;
      if (this._optional || this._refs.length > 0) {
        this._state = HUBU.ServiceDependency.RESOLVED;
      } else {
        this._state = HUBU.ServiceDependency.UNRESOLVED;
      }
      if (oldState !== this._state) {
        return this._serviceComponent.computeState();
      }
    };

    ServiceDependency.prototype._onServiceArrival = function(ref) {
      var entry, refEntry, _i, _len, _ref;
      HUBU.logger.debug("Service arrival detected for " + this._component.getComponentName());
      _ref = this._refs;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        if (entry.reference === ref) {
          refEntry = entry;
        }
      }
      if (!(refEntry != null)) {
        refEntry = {
          reference: ref,
          service: null
        };
        this._refs.push(refEntry);
        this._computeDependencyState();
        if (this._aggregate) {
          return this._inject(refEntry);
        } else {
          if (this._refs.length === 1) {
            return this._inject(refEntry);
          }
        }
      }
    };

    ServiceDependency.prototype._onServiceDeparture = function(ref) {
      var entry, newRef, refEntry, _i, _len, _ref;
      HUBU.logger.debug("Service departure detected for " + this._component.getComponentName());
      _ref = this._refs;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        if (entry.reference === ref) {
          refEntry = entry;
        }
      }
      if (refEntry != null) {
        HUBU.UTILS.removeElementFromArray(this._refs, refEntry);
        if (refEntry.service != null) {
          this._deinject(refEntry);
          this._hub.ungetService(this._component, ref);
          refEntry.service = null;
        }
        if (this._refs.length > 0) {
          newRef = this._refs[0];
          if (!this._aggregate) {
            return this._inject(newRef);
          }
        } else {
          return this._computeDependencyState();
        }
      }
    };

    ServiceDependency.prototype._onServiceModified = function(ref) {
      var entry, refEntry, _i, _len, _ref;
      _ref = this._refs;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        if (entry.reference === ref) {
          refEntry = entry;
        }
      }
      if (!(refEntry != null)) {
        return this._onServiceArrival(ref);
      }
    };

    ServiceDependency.prototype._inject = function(entry) {
      var svc;
      svc = this._hub.getService(this._serviceComponent, entry.reference);
      entry.service = svc;
      if ((this._field != null) && this._aggregate) {
        if (!(this._component[this._field] != null)) {
          this._component[this._field] = [svc];
        } else {
          this._component[this._field].push(svc);
        }
      }
      if ((this._field != null) && !this._aggregate) {
        this._component[this._field] = svc;
      }
      if (this._bind != null) {
        return this._bind.apply(this._component, [svc, entry.reference]);
      }
    };

    ServiceDependency.prototype._deinject = function(entry) {
      if ((this._field != null) && this._aggregate) {
        HUBU.UTILS.removeElementFromArray(this._component[this._field], entry.service);
      }
      if ((this._field != null) && !this._aggregate) {
        this._component[this._field] = null;
      }
      if (this._unbind != null) {
        return this._unbind.apply(this._component, [entry.service, entry.reference]);
      }
    };

    /**
    Gets the current service object(s).
    This method returns an array of service objects.
    @method
    @memberOf HUBU.ServiceComponent
    @name #locateServices
    @returns {Array} The array of service objects. Contains only one element for scalar dependencies.
    */


    ServiceDependency.prototype.locateServices = function() {
      var ref, refs, svc, _i, _len;
      svc = [];
      refs = this._hub.getServiceReferences(this._contract, this._filter);
      for (_i = 0, _len = refs.length; _i < _len; _i++) {
        ref = refs[_i];
        svc.push(this._hub.getService(this._component, ref));
      }
      return svc;
    };

    return ServiceDependency;

  })();

  HUBU.ProvidedService = ProvidedService = (function() {

    ProvidedService.UNREGISTERED = 0;

    ProvidedService.REGISTERED = 1;

    ProvidedService.prototype._hub = null;

    ProvidedService.prototype._contract = null;

    ProvidedService.prototype._properties = null;

    ProvidedService.prototype._registration = null;

    ProvidedService.prototype._serviceComponent = null;

    ProvidedService.prototype._component = null;

    ProvidedService.prototype._preRegistration = null;

    ProvidedService.prototype._postRegistration = null;

    ProvidedService.prototype._preUnregistration = null;

    ProvidedService.prototype._postUnRegistration = null;

    function ProvidedService(component, contract, properties, preRegistration, postRegistration, preUnregistration, postUnregistration, hub) {
      this._component = component;
      this._contract = contract;
      this._hub = hub;
      this._properties = properties;
      if (preRegistration != null) {
        this._preRegistration = HUBU.UTILS.isFunction(preRegistration) ? preRegistration : this._component[preRegistration];
        if (!(this._preRegistration != null)) {
          throw new Exception("preRegistration method " + preRegistration + " not found on component");
        }
      }
      if (postRegistration != null) {
        this._postRegistration = HUBU.UTILS.isFunction(postRegistration) ? postRegistration : this._component[postRegistration];
        if (!(this._postRegistration != null)) {
          throw new Exception("postRegistration method " + postRegistration + " not found on component");
        }
      }
      if (preUnregistration != null) {
        this._preUnregistration = HUBU.UTILS.isFunction(preUnregistration) ? preUnregistration : this._component[preUnregistration];
        if (!(this._preUnregistration != null)) {
          throw new Exception("preUnregistration method " + preUnregistration + " not found on component");
        }
      }
      if (postUnregistration != null) {
        this._postUnRegistration = HUBU.UTILS.isFunction(postUnregistration) ? postUnregistration : this._component[postUnregistration];
        if (!(this._postUnRegistration != null)) {
          throw new Exception("postUnregistration method " + postUnregistration + " not found on component");
        }
      }
    }

    ProvidedService.prototype.setServiceComponent = function(sc) {
      return this._serviceComponent = sc;
    };

    ProvidedService.prototype._register = function() {
      var proxy;
      if (this._registration != null) {
        return false;
      }
      if ((this._preRegistration != null)) {
        this._preRegistration.apply(this._component, []);
      }
      proxy = HUBU.UTILS.createProxyForContract(this._contract, this._component);
      this._registration = this._hub.registerService(this._component, this._contract, this._properties, proxy);
      HUBU.logger.debug("Service from " + this._component.getComponentName() + " registered");
      if ((this._postRegistration != null)) {
        this._postRegistration.apply(this._component, [this._registration]);
      }
      return true;
    };

    ProvidedService.prototype._unregister = function() {
      if (!(this._registration != null)) {
        return false;
      }
      if (this._preUnregistration != null) {
        this._preUnregistration.apply(this._component, [this._registration]);
      }
      this._hub.unregisterService(this._registration);
      this._registration = null;
      if (this._postUnRegistration != null) {
        return this._postUnRegistration.apply(this._component, []);
      }
    };

    ProvidedService.prototype.onStart = function() {};

    ProvidedService.prototype.onStop = function() {
      return this._unregister();
    };

    ProvidedService.prototype.onValidation = function() {
      return this._register();
    };

    ProvidedService.prototype.onInvalidation = function() {
      return this._unregister();
    };

    return ProvidedService;

  })();

  /**
  @class
  @classdesc The service oriented extension. This extension handles service components, so manage provided and required services.
  @param {HUBU.Hub} the hub
  */


  HUBU.ServiceOrientation = ServiceOrientation = (function() {
    /**
    The hub
    @private
    @name HUBU.ServiceOrientation#_hub
    @type {HUBU.Hub}
    */

    ServiceOrientation.prototype._hub = null;

    /**
    The service registry
    @private
    @name HUBU.ServiceOrientation#_registry
    @type SOC.ServiceRegistry
    */


    ServiceOrientation.prototype._registry = null;

    /**
    An array of { component -> service component }.
    To keep things simple, a component can have only one service component
    @type {Array}
    @private
    @name HUBU.ServiceOrientation#_components
    */


    ServiceOrientation.prototype._components = [];

    function ServiceOrientation(hubu) {
      var registry, self;
      this._hub = hubu;
      this._registry = new SOC.ServiceRegistry(this._hub);
      this._components = [];
      registry = this._registry;
      self = this;
      /**
      Gets the service registry of the hub.
      @method
      @name HUBU.Hub#getServiceRegistry
      @return {SOC.ServiceRegistry} the service registry
      */

      this._hub.getServiceRegistry = function() {
        return registry;
      };
      /**
      Registers a service in the hub's service registry.
      @method
      @name HUBU.Hub#registerService
      @param {HUBU.AbstractComponent} component the component registering the service
      @param {Object} contract the published contract
      @param {Object} properties the service properties (optional)
      @param {Object} svcObject either the service object, or the contruction method
      @return {SOC.ServiceRegistration} the service registration
      */

      this._hub.registerService = function(component, contract, properties, svcObject) {
        return registry.registerService(component, contract, properties, svcObject);
      };
      /**
      Unregisters a service.
      @method
      @name HUBU.Hub#unregisterService
      @param {SOC.ServiceRegistration} registration the service registration of the service to unpublish.
      */

      this._hub.unregisterService = function(registration) {
        return registry.unregisterService(registration);
      };
      /**
      Looks for service references
      @method
      @name HUBU.Hub#getServiceReferences
      @param {Object} contract the service contract
      @param {Function} filter the filter method that the provider must match
      @return {Array} an array of all matching service references, empty if no services match
      */

      this._hub.getServiceReferences = function(contract, filter) {
        return registry.getServiceReferences(contract, filter);
      };
      /**
      Looks for a service reference
      @method
      @name HUBU.Hub#getServiceReference
      @param {Object} contract the service contract
      @param {Function} filter the filter method that the provider must match
      @return {SOC.ServiceReference} a matching service reference or `null` if no services match
      */

      this._hub.getServiceReference = function(contract, filter) {
        var refs;
        refs = registry.getServiceReferences(contract, filter);
        if (refs.length !== 0) {
          return refs[0];
        }
        return null;
      };
      /**
      Gets the service object of the given service reference.
      @method
      @name HUBU.Hub#getService
      @param {HUBU.AbstractComponent} component the component getting the service
      @param {SOC.ServiceReference} reference the service reference
      @return {Object} the service object
      */

      this._hub.getService = function(component, reference) {
        return registry.getService(component, reference);
      };
      /**
      Releases an used service.
      @method
      @name HUBU.Hub#ungetService
      @param {HUBU.AbstractComponent} component the component that got the service
      @param {SOC.ServiceReference} reference the service reference
      */

      this._hub.ungetService = function(component, reference) {
        return registry.ungetService(component, reference);
      };
      /**
      Registers a service listener on the service registry of the hub.
      The parameter specifies the _listener_. This parameter must contain a key `listener`  with a function as value.
      This function receives a `SOC.ServiceEvent`. The listener is called everytime a matching service event is fired.
      the parameter must also contain the `contract` specifying the targeted service contract and/or a `filter`, i.e. a
      method validating a service reference (given as parameter). For example, the following snippet illustrates a valid
      service listener registrations:
      
        var listenAllContractService = {
              bindCount: 0,
              unbindCount : 0,
              contract : contract,
              // no filter
              listener : function(event) {
                  if (event.getType() === SOC.ServiceEvent.REGISTERED) {
                      listenAllContractService.bindCount = listenAllContractService.bindCount +1;
                  } else if (event.getType() === SOC.ServiceEvent.UNREGISTERING) {
                      listenAllContractService.unbindCount = listenAllContractService.unbindCount +1;
                  }
              }
          };
      
          var listenFrContractService = {
              bindCount: 0,
              unbindCount : 0,
              contract : contract,
              filter : function(ref) {
                  return ref.getProperty("lg") === "fr";
              },
              listener : function(event) {
                  if (event.getType() === SOC.ServiceEvent.REGISTERED) {
                      listenFrContractService.bindCount = listenFrContractService.bindCount +1;
                  } else if (event.getType() === SOC.ServiceEvent.UNREGISTERING) {
                      listenFrContractService.unbindCount = listenFrContractService.unbindCount +1;
                  }
              }
          };
      
          hub.registerServiceListener(listenAllContractService);
          hub.registerServiceListener(listenFrContractService);
      
      @method
      @name HUBU.Hub#registerServiceListener
      @param {Object} listenerConfiguration the listener configuration.
      */

      this._hub.registerServiceListener = function(listenerConfiguration) {
        return registry.registerServiceListener(listenerConfiguration);
      };
      /**
      Unregisters a service listener.
      @method
      @name HUBU.Hub#unregisterServiceListener
      @param {Object} listenerConfiguration The service listener to unregister.
      */

      this._hub.unregisterServiceListener = function(listenerConfiguration) {
        return registry.unregisterServiceListener(listenerConfiguration);
      };
      /**
      Defines a service dependency. This method is used to declare a service dependency injected automatically within the
      component. Please refer to the documentation.
      @method
      @name HUBU.Hub#requireService
      @param {Object} the service dependency description.
      @return {HUBU.Hub} the current hub
      */

      this._hub.requireService = function(description) {
        self.requireService(description);
        return this;
      };
      /**
      Defines a provided service. The service is managed by h-ubu. Please refer to the documentation
      @method
      @name HUBU.Hub#provideService
      @param {Object} the provided service description.
      @return {HUBU.Hub} the current hub
      */

      this._hub.provideService = function(description) {
        self.provideService(description);
        return this;
      };
      /**
      Locates a service dependency. This method returns only the first service object on aggregate dependencies.
      @method
      @name HUBU.Hub#locateService
      @param {HUBU.AbstractComponent} component the component holding the dependency.
      @param {String} name the dependency name, or the contract is the name was omitted in the service dependency.
      @return {Object} the service object, `null` if there are no service provider.
      */

      this._hub.locateService = function(component, name) {
        var cmpEntry, dep, entry, svc, _i, _len, _ref;
        _ref = self._components;
        for (_i = 0, _len = _ref.length; _i < _len; _i++) {
          entry = _ref[_i];
          if (entry.component === component) {
            cmpEntry = entry;
          }
        }
        if (!(cmpEntry != null)) {
          return null;
        }
        dep = cmpEntry.serviceComponent.getServiceDependencyByName(name);
        if (!(dep != null)) {
          throw new Exception("No dependency " + name + " on component " + cmpEntry.component.getComponentName());
        }
        svc = dep.locateServices();
        if (svc === null || svc.length === 0) {
          return null;
        }
        return svc[0];
      };
      /**
      Locates a service dependency. This method returns all service object on aggregate dependencies, but an array of one
      element on scalar dependencies.
      @method
      @name HUBU.Hub#locateServices
      @param {HUBU.AbstractComponent} component the component holding the dependency.
      @param {String} name the dependency name, or the contract is the name was omitted in the service dependency.
      @return {Array} the service objects, empty if there are no service provider, with only one element on fulfilled scalar
      service dependencies.
      */

      this._hub.locateServices = function(component, name) {
        var cmpEntry, dep, entry, svc, _i, _len, _ref;
        _ref = self._components;
        for (_i = 0, _len = _ref.length; _i < _len; _i++) {
          entry = _ref[_i];
          if (entry.component === component) {
            cmpEntry = entry;
          }
        }
        if (!(cmpEntry != null)) {
          return null;
        }
        dep = cmpEntry.serviceComponent.getServiceDependencyByName(name);
        if (!(dep != null)) {
          throw new Exception("No dependency " + name + " on component " + cmpEntry.component.getComponentName());
        }
        svc = dep.locateServices();
        if (svc === null || svc.length === 0) {
          return [];
        }
        return svc;
      };
    }

    /* End of constructor
    */


    /*
      # The given component is unregistered from the hub. We needs to unregisters all services.
    */


    ServiceOrientation.prototype.unregisterComponent = function(cmp) {
      var entry, _i, _len, _ref;
      _ref = this._components;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        if (!((entry != null) && entry.component === cmp)) {
          continue;
        }
        entry.serviceComponent.onStop();
        HUBU.UTILS.removeElementFromArray(this._components, entry);
      }
      return this._registry.unregisterServices(cmp);
    };

    ServiceOrientation.prototype.requireService = function(description) {
      var aggregate, bind, component, contract, field, filter, name, optional, req, unbind;
      component = description.component, contract = description.contract, filter = description.filter, aggregate = description.aggregate, optional = description.optional, field = description.field, bind = description.bind, unbind = description.unbind, name = description.name;
      if (!(component != null)) {
        throw new Exception("Cannot require a service without a valid component");
      }
      if (aggregate == null) {
        aggregate = false;
      }
      if (optional == null) {
        optional = false;
      }
      if (contract == null) {
        contract = null;
      }
      if (filter == null) {
        filter = null;
      }
      if (!(field != null) && !(bind != null) && !(name != null)) {
        throw new Exception("Cannot require a service - field or bind must be set");
      }
      if (field == null) {
        field = null;
      }
      if (bind == null) {
        bind = null;
      }
      if (unbind == null) {
        unbind = null;
      }
      if (name == null) {
        name = contract;
      }
      if (!(field != null) && !(bind != null)) {
        optional = true;
      }
      req = new HUBU.ServiceDependency(component, contract, filter, aggregate, optional, field, bind, unbind, name, this._hub);
      return this._addServiceDependencyToComponent(component, req);
    };

    ServiceOrientation.prototype.provideService = function(description) {
      var component, contract, postRegistration, postUnregistration, preRegistration, preUnregistration, properties, ps;
      component = description.component, contract = description.contract, properties = description.properties, preRegistration = description.preRegistration, postRegistration = description.postRegistration, preUnregistration = description.preUnregistration, postUnregistration = description.postUnregistration;
      if (!(component != null)) {
        throw new Exception("Cannot provided a service without a valid component");
      }
      if (!(contract != null)) {
        throw new Exception("Cannot provided a service without a valid contract");
      }
      if (properties == null) {
        properties = {};
      }
      ps = new HUBU.ProvidedService(component, contract, properties, preRegistration, postRegistration, preUnregistration, postUnregistration, this._hub);
      return this._addProvidedServiceToComponent(component, ps);
    };

    ServiceOrientation.prototype._addServiceDependencyToComponent = function(comp, req) {
      var cmpEntry, entry, newComponent, _i, _len, _ref;
      newComponent = false;
      _ref = this._components;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        if (entry.component === comp) {
          cmpEntry = entry;
        }
      }
      if (!(cmpEntry != null)) {
        cmpEntry = {
          'component': comp,
          'serviceComponent': new HUBU.ServiceComponent(comp)
        };
        this._components.push(cmpEntry);
        newComponent = true;
      }
      cmpEntry.serviceComponent.addRequiredService(req);
      if (newComponent && this._hub.isStarted()) {
        return cmpEntry.serviceComponent.onStart();
      }
    };

    ServiceOrientation.prototype._addProvidedServiceToComponent = function(comp, ps) {
      var cmpEntry, entry, newComponent, _i, _len, _ref;
      newComponent = false;
      _ref = this._components;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        if (entry.component === comp) {
          cmpEntry = entry;
        }
      }
      if (!(cmpEntry != null)) {
        cmpEntry = {
          'component': comp,
          'serviceComponent': new HUBU.ServiceComponent(comp)
        };
        this._components.push(cmpEntry);
        newComponent = true;
      }
      cmpEntry.serviceComponent.addProvidedService(ps);
      if (this._hub.isStarted() && newComponent) {
        return cmpEntry.serviceComponent.onStart();
      }
    };

    ServiceOrientation.prototype.start = function() {
      var entry, _i, _len, _ref, _results;
      _ref = this._components;
      _results = [];
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        _results.push(entry.serviceComponent.onStart());
      }
      return _results;
    };

    ServiceOrientation.prototype.stop = function() {
      var entry, _i, _len, _ref, _results;
      _ref = this._components;
      _results = [];
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        entry = _ref[_i];
        _results.push(entry.serviceComponent.onStop());
      }
      return _results;
    };

    return ServiceOrientation;

  })();

  /* End of the Service Orientation Extension class
  */


  getHubuExtensions().service = ServiceOrientation;

}).call(this);
