
/*
#
# Copyright 2013 OW2 Nanoko Project
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
*/


/*
# Define the building block of the Service-Orientation of H-UBU
*/


/*
TODO Used by -> the usage graph
TODO Several contract within a registration
TODO Service Ranking
TODO Factories / Object creation strategy ?
*/


(function() {
  var SOC, ServiceEvent, ServiceReference, ServiceRegistration, ServiceRegistry, _ref;

  getGlobal().SOC = (_ref = getGlobal().SOC) != null ? _ref : {};

  SOC = getGlobal().SOC;

  /*
  # Service Registrations represents a published service from the publisher point of view.
  */


  SOC.ServiceRegistration = ServiceRegistration = (function() {

    ServiceRegistration._nextId = 1;

    ServiceRegistration.prototype._id = -1;

    ServiceRegistration.prototype._component = null;

    ServiceRegistration.prototype._contract = null;

    ServiceRegistration.prototype._hub = null;

    ServiceRegistration.prototype._registered = false;

    ServiceRegistration.prototype._properties = {};

    ServiceRegistration.prototype._reference = null;

    ServiceRegistration.prototype._registry = null;

    ServiceRegistration.prototype._svcObject = null;

    ServiceRegistration.getAndIncId = function() {
      var id;
      id = SOC.ServiceRegistration._nextId;
      SOC.ServiceRegistration._nextId = SOC.ServiceRegistration._nextId + 1;
      return id;
    };

    function ServiceRegistration(contract, component, svcObject, properties, hub, registry) {
      this._id = -1;
      if (!(component != null)) {
        throw new Exception("Cannot create a service registration without a valid component");
      }
      if (!(svcObject != null)) {
        throw new Exception("Cannot create a service registration without a valid service object");
      }
      if (!(contract != null)) {
        throw new Exception("Cannot create a service registration without a contract");
      }
      if (!(hub != null)) {
        throw new Exception("Cannot create a service registration without the hub");
      }
      if (!(registry != null)) {
        throw new Exception("Cannot create a service registration without the registry");
      }
      this._component = component;
      this._hub = hub;
      this._contract = contract;
      this._properties = properties != null ? properties : {};
      this._registry = registry;
      this._svcObject = svcObject;
      this._properties["service.contract"] = this._contract;
      this._properties["service.publisher"] = this._component;
    }

    ServiceRegistration.prototype.register = function() {
      if (!(HUBU.UTILS.isComponentPlugged(this._component, this._hub) || this._component === this._hub)) {
        throw new Exception("Invalid registration, the component is not plugged on the hub");
      }
      this._id = SOC.ServiceRegistration.getAndIncId();
      this._reference = new SOC.ServiceReference(this);
      this._properties["service.id"] = this._id;
      this._registered = this._id !== -1;
      return this._id;
    };

    ServiceRegistration.prototype.unregister = function() {
      return this._registered = false;
    };

    ServiceRegistration.prototype.isRegistered = function() {
      return this._registered;
    };

    ServiceRegistration.prototype.getReference = function() {
      if (!(HUBU.UTILS.isComponentPlugged(this._component, this._hub) || this._component === this._hub)) {
        throw new Exception("Invalid lookup, the component is not plugged on the hub");
      }
      return this._reference;
    };

    ServiceRegistration.prototype.getProperties = function() {
      return this._properties;
    };

    ServiceRegistration.prototype.getService = function(component) {
      if (!HUBU.UTILS.isFunction(this._svcObject)) {
        return this._svcObject;
      }
      return this._svcObject.apply(this._component, [component]);
    };

    ServiceRegistration.prototype.setProperties = function(properties) {
      var event, old, props;
      old = null;
      if (this.isRegistered()) {
        props = HUBU.UTILS.clone(this._properties, ["service.contract", "service.publisher"]);
        old = new SOC.ServiceRegistration(this._contract, this._component, this._svcObject, props, this._hub, this._registry);
        old._id = this._id;
        old._reference = new SOC.ServiceReference(old);
      }
      this._properties = properties != null ? properties : {};
      this._properties["service.contract"] = this._contract;
      this._properties["service.publisher"] = this._component;
      this._properties["service.id"] = this._id;
      if (this.isRegistered() && (old != null)) {
        event = new SOC.ServiceEvent(SOC.ServiceEvent.MODIFIED, this.getReference());
        return this._registry.fireServiceEvent(event, old.getReference());
      }
    };

    return ServiceRegistration;

  })();

  /*
  #  Service Reference represents a published service from the consumer point of view.
  */


  SOC.ServiceReference = ServiceReference = (function() {

    ServiceReference.prototype._registration = null;

    function ServiceReference(registration) {
      this._registration = registration;
    }

    ServiceReference.prototype.getContract = function() {
      return this._registration.getProperties()["service.contract"];
    };

    ServiceReference.prototype.getProperties = function() {
      return this._registration.getProperties();
    };

    ServiceReference.prototype.getProperty = function(key) {
      return this._registration.getProperties()[key];
    };

    ServiceReference.prototype.getId = function() {
      return this._registration.getProperties()["service.id"];
    };

    ServiceReference.prototype.isValid = function() {
      return this._registration.isRegistered;
    };

    return ServiceReference;

  })();

  SOC.ServiceEvent = ServiceEvent = (function() {

    ServiceEvent.REGISTERED = 1;

    ServiceEvent.MODIFIED = 2;

    ServiceEvent.UNREGISTERING = 4;

    ServiceEvent.MODIFIED_ENDMATCH = 8;

    ServiceEvent.prototype._type = 0;

    ServiceEvent.prototype._reference = null;

    function ServiceEvent(type, ref) {
      this._type = type;
      this._reference = ref;
    }

    ServiceEvent.prototype.getReference = function() {
      return this._reference;
    };

    ServiceEvent.prototype.getType = function() {
      return this._type;
    };

    return ServiceEvent;

  })();

  /*
  # The Service Registry class
  */


  SOC.ServiceRegistry = ServiceRegistry = (function() {

    ServiceRegistry.prototype._registrations = null;

    ServiceRegistry.prototype._hub = null;

    ServiceRegistry.prototype._listeners = null;

    function ServiceRegistry(hub) {
      this._registrations = [];
      this._listeners = [];
      if (!(hub != null)) {
        throw new Exception("Cannot initialize the service registry without a hub");
      }
      this._hub = hub;
    }

    /*
      # Gets all registered services.
      # @return the list of service references, empty if no services are registered
    */


    ServiceRegistry.prototype.getRegisteredServices = function() {
      var entry, reg, result, _i, _j, _len, _len1, _ref1, _ref2;
      result = [];
      _ref1 = this._registrations;
      for (_i = 0, _len = _ref1.length; _i < _len; _i++) {
        entry = _ref1[_i];
        _ref2 = entry.registrations;
        for (_j = 0, _len1 = _ref2.length; _j < _len1; _j++) {
          reg = _ref2[_j];
          result.push(reg.getReference());
        }
      }
      return result;
    };

    /*
      # Adds a service registration
    */


    ServiceRegistry.prototype._addRegistration = function(component, reg) {
      var cmpEntry, entry, _i, _len, _ref1;
      _ref1 = this._registrations;
      for (_i = 0, _len = _ref1.length; _i < _len; _i++) {
        entry = _ref1[_i];
        if (entry.component === component) {
          cmpEntry = entry;
        }
      }
      if (!(cmpEntry != null)) {
        cmpEntry = {
          'component': component,
          'registrations': []
        };
        this._registrations.push(cmpEntry);
      }
      return cmpEntry.registrations.push(reg);
    };

    ServiceRegistry.prototype._removeRegistration = function(reg) {
      var cmpEntry, entry, _i, _len, _ref1;
      _ref1 = this._registrations;
      for (_i = 0, _len = _ref1.length; _i < _len; _i++) {
        entry = _ref1[_i];
        if (HUBU.UTILS.indexOf(entry.registrations, reg) !== -1) {
          cmpEntry = entry;
        }
      }
      if (!(cmpEntry != null)) {
        return null;
      }
      HUBU.UTILS.removeElementFromArray(cmpEntry.registrations, reg);
      if (cmpEntry.registrations.length === 0) {
        HUBU.UTILS.removeElementFromArray(this._registrations, cmpEntry);
      }
      return cmpEntry.component;
    };

    /*
      # Registers a service
      # @return the service registration
    */


    ServiceRegistry.prototype.registerService = function(component, contract, properties, svcObject) {
      var reg;
      if (!(contract != null)) {
        throw new Exception("Cannot register a service without a proper contract");
      }
      if (!(component != null)) {
        throw new Exception("Cannot register a service without a valid component");
      }
      svcObject = svcObject != null ? svcObject : component;
      if (!HUBU.UTILS.isFunction(svcObject) && !HUBU.UTILS.isObjectConformToContract(svcObject, contract)) {
        throw new Exception("Cannot register service - the service object does not implement the contract").add("contract", contract).add("component", component);
      }
      svcObject = svcObject != null ? svcObject : component;
      reg = new ServiceRegistration(contract, component, svcObject, properties, this._hub, this);
      this._addRegistration(component, reg);
      reg.register();
      this.fireServiceEvent(new SOC.ServiceEvent(SOC.ServiceEvent.REGISTERED, reg.getReference()));
      return reg;
    };

    /*
      # Unregisters a service
    */


    ServiceRegistry.prototype.unregisterService = function(registration) {
      var component, ref;
      if (!(registration != null)) {
        throw new Exception("Cannot unregister the service - invalid registration");
      }
      component = this._removeRegistration(registration);
      if ((component != null)) {
        ref = registration.getReference();
        registration.unregister();
        this.fireServiceEvent(new SOC.ServiceEvent(SOC.ServiceEvent.UNREGISTERING, ref));
        return true;
      }
      throw new Exception("Cannot unregister service - registration not found");
    };

    ServiceRegistry.prototype.unregisterServices = function(component) {
      var cmpEntry, entry, reg, regs, _i, _j, _len, _len1, _ref1;
      if (!(component != null)) {
        throw new Exception("Cannot unregister the services - invalid component");
      }
      _ref1 = this._registrations;
      for (_i = 0, _len = _ref1.length; _i < _len; _i++) {
        entry = _ref1[_i];
        if (entry.component === component) {
          cmpEntry = entry;
        }
      }
      if (cmpEntry != null) {
        regs = cmpEntry.registrations;
        if (regs != null) {
          for (_j = 0, _len1 = regs.length; _j < _len1; _j++) {
            reg = regs[_j];
            this.unregisterService(reg);
          }
        }
        return HUBU.UTILS.removeElementFromArray(this._registrations, cmpEntry);
      }
    };

    ServiceRegistry.prototype.getServiceReferences = function(contract, filter) {
      return this._match(this._buildFilter(contract, filter));
    };

    /*
      # Traverses the registered services to select the ones matching with the given filter (method)
      # It returns an empty array if ne matching service can be found
    */


    ServiceRegistry.prototype._match = function(filter) {
      var matching, ref, refs;
      refs = this.getRegisteredServices();
      matching = (function() {
        var _i, _len, _results;
        _results = [];
        for (_i = 0, _len = refs.length; _i < _len; _i++) {
          ref = refs[_i];
          if (filter.match(ref)) {
            _results.push(ref);
          }
        }
        return _results;
      })();
      return matching;
    };

    /*
      # Build an object with a `match` function built from the contract and the filter.
    */


    ServiceRegistry.prototype._buildFilter = function(contract, filter) {
      var container,
        _this = this;
      if (!(contract != null) && !(filter != null)) {
        return {
          match: function(ref) {
            return true;
          }
        };
      } else if ((contract != null) && !(filter != null)) {
        container = {};
        container.contract = contract;
        container.match = function(ref) {
          return ref.getProperty("service.contract") === container.contract;
        };
        return container;
      } else if ((contract != null) && (filter != null)) {
        container = {};
        container.contract = contract;
        container.filter = filter;
        container.match = function(ref) {
          return (ref.getProperty("service.contract") === container.contract) && container.filter(ref);
        };
        return container;
      } else {
        return {
          filter: filter,
          match: function(ref) {
            return this.filter(ref);
          }
        };
      }
    };

    ServiceRegistry.prototype.getService = function(component, ref) {
      if (!(ref != null)) {
        throw new Exception("Cannot get service - the reference is null");
      }
      if (!ref.isValid()) {
        HUBU.logger.warn("Cannot retrieve service for " + ref + " - the reference is invalid");
        return null;
      }
      return ref._registration.getService(component);
    };

    ServiceRegistry.prototype.ungetService = function(component, ref) {};

    ServiceRegistry.prototype.registerServiceListener = function(listenerConfig) {
      var contract, filter, listener, newFilter, svcListener;
      contract = listenerConfig.contract, filter = listenerConfig.filter, listener = listenerConfig.listener;
      if (!(listener != null)) {
        throw new Exception("Can't register the service listener, the listener is not set").add("listenerConfig", listenerConfig);
      }
      newFilter = this._buildFilter(contract, filter);
      svcListener = {
        listener: listener,
        filter: newFilter,
        contract: contract
      };
      if (HUBU.UTILS.isObject(listener)) {
        if (!HUBU.UTILS.isObjectConformToContract(listener, SOC.ServiceListener)) {
          throw new Exception("Can't register the service listener, the listener is not conform to the Service Listener contract");
        }
      }
      return this._listeners.push(svcListener);
    };

    ServiceRegistry.prototype.unregisterServiceListener = function(listenerConfig) {
      var contract, filter, list, listener, _i, _len, _ref1, _results;
      contract = listenerConfig.contract, filter = listenerConfig.filter, listener = listenerConfig.listener;
      if (!(listener != null)) {
        throw new Exception("Can't unregister the service listener, the listener is not set").add("listenerConfig", listenerConfig);
      }
      _ref1 = this._listeners.slice();
      _results = [];
      for (_i = 0, _len = _ref1.length; _i < _len; _i++) {
        list = _ref1[_i];
        if (list.contract === contract && list.listener === listener) {
          _results.push(HUBU.UTILS.removeElementFromArray(this._listeners, list));
        }
      }
      return _results;
    };

    /*
      # This method should be used by the extension only.
    */


    ServiceRegistry.prototype.fireServiceEvent = function(event, oldRef) {
      var listener, matched, newEvent, _i, _len, _ref1, _results;
      _ref1 = this._listeners;
      _results = [];
      for (_i = 0, _len = _ref1.length; _i < _len; _i++) {
        listener = _ref1[_i];
        matched = !(listener.filter != null) || this._testAgainstFilter(listener, event.getReference());
        if (matched) {
          _results.push(this._invokeServiceListener(listener, event));
        } else if (event.getType() === SOC.ServiceEvent.MODIFIED && (oldRef != null)) {
          if (this._testAgainstFilter(listener, oldRef)) {
            newEvent = new SOC.ServiceEvent(SOC.ServiceEvent.MODIFIED_ENDMATCH, event.getReference());
            _results.push(this._invokeServiceListener(listener, newEvent));
          } else {
            _results.push(void 0);
          }
        } else {
          _results.push(void 0);
        }
      }
      return _results;
    };

    ServiceRegistry.prototype._testAgainstFilter = function(listener, ref) {
      return listener.filter.match(ref);
    };

    ServiceRegistry.prototype._invokeServiceListener = function(listener, event) {
      if (HUBU.UTILS.isFunction(listener.listener)) {
        return listener.listener(event);
      } else if (HUBU.UTILS.isObject(listener.listener)) {
        return listener.serviceChanged(event);
      }
    };

    return ServiceRegistry;

  })();

}).call(this);
