
/*
#
# Copyright 2013 OW2 Nanoko Project
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
*/


/*
# Utility methods used by H-UBU
*/


(function() {
  var Exception, Logger, logger, utils, _ref,
    __slice = [].slice;

  HUBU.UTILS = (_ref = HUBU.UTILS) != null ? _ref : {};

  utils = HUBU.UTILS;

  getGlobal().namespace = function(target, name, block) {
    var item, top, _i, _len, _ref1, _ref2;
    if (arguments.length < 3) {
      _ref1 = [(typeof exports !== 'undefined' ? exports : window)].concat(__slice.call(arguments)), target = _ref1[0], name = _ref1[1], block = _ref1[2];
    }
    top = target;
    _ref2 = name.split('.');
    for (_i = 0, _len = _ref2.length; _i < _len; _i++) {
      item = _ref2[_i];
      target = target[item] || (target[item] = {});
    }
    return block(target, top);
  };

  /*
  # Logger.
  # If set will happend the logger name in from of all logged messages.
  # The logger uses `window.console` or `global.console` to log messages. So if this object is not defined, the message are not logged.
  # The logger defines the common logging methods: `debug`, `info`, `warn` and `error`.
  # By default the log level is set to INFO, but can be adjusted using the `setLevel` method.
  */


  getGlobal().Logger = Logger = (function() {

    Logger.DEBUG = 0;

    Logger.INFO = 1;

    Logger.WARNING = 2;

    Logger.ERROR = 3;

    Logger.prototype._header = "";

    Logger.prototype._level = Logger.INFO;

    function Logger(name) {
      var m_header;
      if (name == null) {
        name = "";
      }
      if (name.length > 0) {
        m_header = "[" + name + "] ";
      }
    }

    Logger.prototype._getConsole = function() {
      if (((typeof window !== "undefined" && window !== null ? window.console : void 0) != null)) {
        return window.console;
      }
      if (((typeof global !== "undefined" && global !== null ? global.console : void 0) != null)) {
        return global.console;
      }
      return null;
    };

    Logger.prototype.log = function(message) {
      if ((this._getConsole() != null)) {
        this._getConsole().log(("" + this._header) + message);
        return true;
      }
      return false;
    };

    Logger.prototype.debug = function(message) {
      if (this._level <= Logger.DEBUG) {
        return this.log("DEBUG - " + message);
      }
      return false;
    };

    Logger.prototype.info = function(message) {
      if (this._level <= Logger.INFO) {
        return this.log("INFO - " + message);
      }
      return false;
    };

    Logger.prototype.warn = function(message) {
      if (this._level <= Logger.WARNING) {
        return this.log("WARN - " + message);
      }
      return false;
    };

    Logger.prototype.error = function(message) {
      if (this._level <= Logger.ERROR) {
        return this.log("ERROR - " + message);
      }
    };

    Logger.prototype.setLevel = function(level) {
      return this._level = level;
    };

    return Logger;

  })();

  /* End of Logger class
  */


  HUBU.logger = new Logger("hubu");

  logger = HUBU.logger;

  getGlobal().Exception = Exception = (function() {

    Exception.prototype.data = {};

    function Exception(message) {
      this.message = message;
    }

    Exception.prototype.add = function(key, value) {
      this.data.key = value;
      return this;
    };

    Exception.prototype.toString = function() {
      return this.message;
    };

    return Exception;

  })();

  /*
  # Contract and Reflection related methods
  */


  /**
  # This function is returning the `type` of an object. It is different from the JavaScript `typeof`, and relies on
  # the Object `toString` method.
  # Here are the different results :
  #
  # *`typeOf(1)` => "number"
  # *`typeOf({})` => "object"
  # *`typeOf([])` => "array"
  # *`typeOf(function() {})` => "function"
  # *`typeOf(null)` => "null"
  #
  */


  utils.typeOf = function(obj) {
    var classToType, myClass, name, _i, _len, _ref1;
    if (!(obj != null)) {
      return new String(obj);
    }
    classToType = new Object;
    _ref1 = "Boolean Number String Function Array Date RegExp".split(" ");
    for (_i = 0, _len = _ref1.length; _i < _len; _i++) {
      name = _ref1[_i];
      classToType["[object " + name + "]"] = name.toLowerCase();
    }
    myClass = Object.prototype.toString.call(obj);
    if (myClass in classToType) {
      return classToType[myClass];
    }
    return "object";
  };

  /**
  # Checks that the given object is conform to the given contract
  # The contract is a javascript object.
  # The conformity is computed as follow:
  #
  # `O is conform to C if and only if for all i in C where C[i] != null O[i] != null && typeof(C[i]) = typeOf(O[i])`
  #
  # This is an implementation of 'Chi':
  # `Metamodel <- chi <- Model -> mu -> System`
  # where chi is : isObjectConformToContract and mu is representationOf.
  # @param object the object to check
  # @param contract the contract
  # @return true if the object is conform with the given contract, false otherwise.
  */


  utils.isObjectConformToContract = function(object, contract) {
    var props;
    for (props in contract) {
      if (!(object[props] != null)) {
        logger.warn("Object not conform to contract - property " + props + " missing");
        return false;
      } else {
        if (this.typeOf(contract[props]) !== (this.typeOf(object[props]))) {
          logger.warn(("Object not conform to contract - the type of the property " + props + " does not match.          Expected '") + this.typeOf(contract[props]) + "' but found '" + this.typeOf(object[props]) + "'");
          return false;
        }
      }
    }
    return true;
  };

  /**
  # Utility method to check if the given object is a function.
  # @param {Object} obj the object to check
  # @returns `true` if the given object is a function, `false` otherwise
  */


  utils.isFunction = function(ref) {
    return this.typeOf(ref) === "function";
  };

  /**
  # Utility method to check if the given object is an object.
  # @param {Object} obj the object to check
  # @returns `true` if the given object is an object, `false` otherwise
  */


  utils.isObject = function(ref) {
    return this.typeOf(ref) === "object";
  };

  /**
  # Invokes the method `method` on the object `target` with the arguments `args` (Array).
  # @param obj the instance
  # @param method the method name to call
  # @param args {Array} the arguments to pass to the method.
  # @return either the result of the method. `false` if the method is not defined, or is not a function.
  */


  utils.invoke = function(target, method, args) {
    if ((target[method] != null) && this.isFunction(target[method])) {
      return target[method].apply(target, args);
    }
    return false;
  };

  /**
  # Extends the given object `obj` with the given function `func`. Basically, if the `obj[name]` is not defined, then
  # this method extends `obj` with `obj[name]=func`
  # If the method is added, the method returns `true`, `false` otherwise.
  # @param obj the object
  # @param name the name of the function to add
  # @param func the function to append to the object
  # @return {Boolean}
  */


  utils.defineFunctionIfNotExist = function(obj, name, func) {
    if (!(obj[name] != null)) {
      obj[name] = func;
      return true;
    }
    return false;
  };

  /**
  # Clone an object (deep copy).
  # @param obj {Object} the object to clone
  # @param excludes {Array} the property to exclude.
  # @return the cloned object, or the object itself if it's not an object.
  */


  utils.clone = function(obj, excludes) {
    var flags, key, newInstance;
    if (!(obj != null) || typeof obj !== 'object') {
      return obj;
    }
    if (obj instanceof Date) {
      return new Date(obj.getTime());
    }
    if (obj instanceof RegExp) {
      flags = '';
      if (obj.global != null) {
        flags += 'g';
      }
      if (obj.ignoreCase != null) {
        flags += 'i';
      }
      if (obj.multiline != null) {
        flags += 'm';
      }
      if (obj.sticky != null) {
        flags += 'y';
      }
      return new RegExp(obj.source, flags);
    }
    newInstance = new obj.constructor();
    excludes = excludes != null ? excludes : [];
    for (key in obj) {
      if (this.indexOf(excludes, key) === -1) {
        newInstance[key] = this.clone(obj[key], excludes);
      }
    }
    return newInstance;
  };

  /**
  # Creates a `bind` method. This method is calling the given `method` on the given `object`.
  # For example, `bind(foo, doSomething)` returns a method like:
  # `function() { return foo.doSomething(); }`
  # @param {Object} the object on which the method will be called
  # @param {Function} the function to call, is can be given as string too
  # @return {Function} the wrapper function.
  */


  utils.bind = function(obj, method) {
    if (this.typeOf(method) === "string") {
      if (obj[method] != null) {
        method = obj[method];
      } else {
        throw 'HUBU.bind: obj[' + method + "] is null";
      }
    }
    if (this.typeOf(method) === "function") {
      return function() {
        return method.apply(obj, Array.prototype.slice.call(arguments));
      };
    } else {
      throw 'HUBU.bind: obj[' + method + "] is not a function";
    }
  };

  /**
  # Creates a proxy hiding the given object. The proxy implements the contract (and only the contract).
  # @param {Object} contract the contract
  # @param {Object} object the object to proxy
  # @return the proxy
  */


  utils.createProxyForContract = function(contract, object) {
    var props, proxy;
    proxy = {};
    proxy.__proxy__ = object;
    for (props in contract) {
      if (this.isFunction(contract[props])) {
        proxy[props] = this.bind(object, object[props]);
      } else {
        proxy[props] = object[props];
      }
    }
    return proxy;
  };

  /**
  # Checks if the given component implements the 'component' protocol (i.e. interface).
  # @param {Object} component the component to check
  # @return `true` if this is a valid component, `false` otherwise.
  */


  utils.isComponent = function(component) {
    if (!(component != null)) {
      return false;
    }
    return this.isObjectConformToContract(component, new HUBU.AbstractComponent());
  };

  /**
  # Checks wheter the given component is plugged on the given hub.
  # The component can be given as string (component name) or as object (component object)
  # @param {Object} or {String} component the component to check
  # @param hub the hub
  @ @return `true` is the component is plugged on the hub, `false` otherwise
  */


  utils.isComponentPlugged = function(component, hub) {
    if (this.typeOf(component) === "string") {
      return hub.getComponent(component) !== null;
    }
    if (this.typeOf(component) === "object") {
      return this.indexOf(hub.getComponents(), component) !== -1;
    }
    return false;
  };

  /**
  # indexOf function.
  # This method delegates on `Array.indexOf` if it exists. If not (IE), it just implements its own indexOf with simple
  # lookup
  # @param {Object} array the array
  # @param {Object} obj the object
  # @return the index of the object 'obj' in the array or -1 if not found.
  */


  utils.indexOf = function(array, obj) {
    var i, v, _i, _len;
    if ((Array.prototype.indexOf != null)) {
      return array.indexOf(obj);
    } else {
      for (i = _i = 0, _len = array.length; _i < _len; i = ++_i) {
        v = array[i];
        if (v === obj) {
          return i;
        }
      }
      return -1;
    }
  };

  /**
  # Removes the object or value `obj` from the array `array`.
  # Even if the array is modified in place, this method returns the final array.
  # All occurence of `obj` are removed from the array
  # @param array the array
  # @param obj the reference to remove
  # @return the final array
  */


  utils.removeElementFromArray = function(array, obj) {
    var v;
    for (v in array) {
      if (array[v] === obj) {
        array.splice(v, 1);
      }
    }
    return array;
  };

  /*
  # End of the contract and reflection related methods
  */


}).call(this);
