/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.security.auth;

import com.google.common.jimfs.Configuration;
import com.google.common.jimfs.Jimfs;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.spi.FileSystemProvider;
import java.util.Arrays;
import java.util.Collection;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.neo4j.graphdb.mockfs.DelegatingFileSystem;
import org.neo4j.graphdb.mockfs.DelegatingFileSystemProvider;
import org.neo4j.kernel.impl.util.Charsets;
import org.neo4j.kernel.impl.util.TestLogger;
import org.neo4j.kernel.impl.util.TestLogging;
import org.neo4j.kernel.logging.DevNullLoggingService;
import org.neo4j.kernel.logging.Logging;
import org.neo4j.server.security.auth.Credential;
import org.neo4j.server.security.auth.FileUserRepository;
import org.neo4j.server.security.auth.User;
import org.neo4j.server.security.auth.exception.ConcurrentModificationException;

@RunWith(value=Parameterized.class)
public class FileUserRepositoryTest {
    private final FileSystem fs;
    private Path authFile;

    @Parameterized.Parameters(name="{1} filesystem")
    public static Collection<Object[]> data() {
        return Arrays.asList({Configuration.unix(), "unix"}, {Configuration.osX(), "osX"}, {Configuration.windows(), "windows"});
    }

    public FileUserRepositoryTest(Configuration fsConfig, String fsType) {
        this.fs = Jimfs.newFileSystem((Configuration)fsConfig);
        this.authFile = this.fs.getPath("dbms", "auth.db");
    }

    @Test
    public void shouldStoreAndRetriveUsersByName() throws Exception {
        FileUserRepository users = new FileUserRepository(this.authFile, DevNullLoggingService.DEV_NULL);
        User user = new User("jake", Credential.INACCESSIBLE, true);
        users.create(user);
        User result = users.findByName(user.name());
        MatcherAssert.assertThat((Object)result, (Matcher)CoreMatchers.equalTo((Object)user));
    }

    @Test
    public void shouldPersistUsers() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.authFile, DevNullLoggingService.DEV_NULL);
        User user = new User("jake", Credential.INACCESSIBLE, true);
        users.create(user);
        users = new FileUserRepository(this.authFile, DevNullLoggingService.DEV_NULL);
        users.start();
        User resultByName = users.findByName(user.name());
        MatcherAssert.assertThat((Object)resultByName, (Matcher)CoreMatchers.equalTo((Object)user));
    }

    @Test
    public void shouldNotFindUserAfterDelete() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.authFile, DevNullLoggingService.DEV_NULL);
        User user = new User("jake", Credential.INACCESSIBLE, true);
        users.create(user);
        users.delete(user);
        MatcherAssert.assertThat((Object)users.findByName(user.name()), (Matcher)CoreMatchers.nullValue());
    }

    @Test
    public void shouldNotAllowComplexNames() throws Exception {
        FileUserRepository users = new FileUserRepository(this.authFile, DevNullLoggingService.DEV_NULL);
        Assert.assertTrue((boolean)users.isValidName("neo4j"));
        Assert.assertTrue((boolean)users.isValidName("johnosbourne"));
        Assert.assertTrue((boolean)users.isValidName("john_osbourne"));
        Assert.assertFalse((boolean)users.isValidName(":"));
        Assert.assertFalse((boolean)users.isValidName(""));
        Assert.assertFalse((boolean)users.isValidName("john osbourne"));
        Assert.assertFalse((boolean)users.isValidName("john:osbourne"));
    }

    @Test
    public void shouldRecoverIfCrashedDuringMove() throws Throwable {
        final IOException exception = new IOException("simulated IO Exception on create");
        DelegatingFileSystem moveFailingFileSystem = new DelegatingFileSystem(this.fs){

            protected DelegatingFileSystemProvider createDelegate(FileSystemProvider provider) {
                return new DelegatingFileSystem.WrappedProvider(provider, this){

                    public void move(Path source, Path target, CopyOption ... options) throws IOException {
                        if (FileUserRepositoryTest.this.authFile.getFileName().toString().equals(target.getFileName().toString())) {
                            throw exception;
                        }
                        super.move(source, target, options);
                    }
                };
            }
        };
        Path authFile = moveFailingFileSystem.getPath("dbms", "auth.db");
        FileUserRepository users = new FileUserRepository(authFile, DevNullLoggingService.DEV_NULL);
        users.start();
        User user = new User("jake", Credential.INACCESSIBLE, true);
        try {
            users.create(user);
            Assert.fail((String)"Expected an IOException");
        }
        catch (IOException e) {
            Assert.assertSame((Object)exception, (Object)e);
        }
        Assert.assertFalse((boolean)Files.exists(authFile, new LinkOption[0]));
        Assert.assertFalse((boolean)Files.newDirectoryStream(authFile.getParent()).iterator().hasNext());
    }

    @Test
    public void shouldThrowIfUpdateChangesName() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.authFile, DevNullLoggingService.DEV_NULL);
        User user = new User("jake", Credential.INACCESSIBLE, true);
        users.create(user);
        User updatedUser = new User("john", Credential.INACCESSIBLE, true);
        try {
            users.update(user, updatedUser);
            Assert.fail((String)"expected exception not thrown");
        }
        catch (IllegalArgumentException e) {
            // empty catch block
        }
        MatcherAssert.assertThat((Object)users.findByName(user.name()), (Matcher)CoreMatchers.equalTo((Object)user));
    }

    @Test
    public void shouldThrowIfExistingUserDoesNotMatch() throws Throwable {
        FileUserRepository users = new FileUserRepository(this.authFile, DevNullLoggingService.DEV_NULL);
        User user = new User("jake", Credential.INACCESSIBLE, true);
        users.create(user);
        User modifiedUser = new User("jake", Credential.forPassword((String)"foo"), false);
        User updatedUser = new User("jake", Credential.forPassword((String)"bar"), false);
        try {
            users.update(modifiedUser, updatedUser);
            Assert.fail((String)"expected exception not thrown");
        }
        catch (ConcurrentModificationException e) {
            // empty catch block
        }
    }

    @Test
    public void shouldIgnoreInvalidEntries() throws Throwable {
        TestLogging testLogging = new TestLogging();
        Files.createDirectories(this.authFile.getParent(), new FileAttribute[0]);
        Files.write(this.authFile, "neo4j:fc4c600b43ffe4d5857b4439c35df88f:SHA-256,A42E541F276CF17036DB7818F8B09B1C229AAD52A17F69F4029617F3A554640F,FB7E8AE08A6A7C741F678AD22217808F:\n".getBytes(Charsets.UTF_8), new OpenOption[0]);
        FileUserRepository users = new FileUserRepository(this.authFile, (Logging)testLogging);
        users.start();
        MatcherAssert.assertThat((Object)users.numberOfUsers(), (Matcher)CoreMatchers.equalTo((Object)0));
        testLogging.getMessagesLog(FileUserRepository.class).assertExactly(new TestLogger.LogCall[]{TestLogger.LogCall.error((String)String.format("Ignoring authorization file \"%s\" (wrong number of line fields [line 1])", this.authFile.toAbsolutePath()))});
    }
}

