/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.rest.security;

import com.fasterxml.jackson.databind.JsonNode;
import java.io.IOException;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.jupiter.api.Assertions;
import org.neo4j.kernel.api.exceptions.Status;
import org.neo4j.kernel.impl.annotations.Documented;
import org.neo4j.server.rest.RESTRequestGenerator;
import org.neo4j.server.rest.domain.JsonHelper;
import org.neo4j.server.rest.domain.JsonParseException;
import org.neo4j.server.rest.security.CommunityServerTestBase;
import org.neo4j.test.TestData;
import org.neo4j.test.server.HTTP;

public class AuthenticationIT
extends CommunityServerTestBase {
    @Rule
    public TestData<RESTRequestGenerator> gen = TestData.producedThrough(RESTRequestGenerator.PRODUCER);

    @Test
    @Documented(value="Missing authorization\n\nIf an +Authorization+ header is not supplied, the server will reply with an error.")
    public void missing_authorization() throws JsonParseException, IOException {
        this.startServerWithConfiguredUser();
        RESTRequestGenerator.ResponseEntity response = ((RESTRequestGenerator)this.gen.get()).expectedStatus(401).expectedHeader("WWW-Authenticate", "Basic realm=\"Neo4j\"").get(this.dataURL());
        JsonNode data = JsonHelper.jsonNode((String)response.entity());
        JsonNode firstError = data.get("errors").get(0);
        Assert.assertThat((Object)firstError.get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)"Neo.ClientError.Security.Unauthorized"));
        Assert.assertThat((Object)firstError.get("message").asText(), (Matcher)CoreMatchers.equalTo((Object)"No authentication header supplied."));
    }

    @Test
    @Documented(value="Authenticate to access the server\n\nAuthenticate by sending a username and a password to Neo4j using HTTP Basic Auth.\nRequests should include an +Authorization+ header, with a value of +Basic <payload>+,\nwhere \"payload\" is a base64 encoded string of \"username:password\".")
    public void successful_authentication() throws JsonParseException, IOException {
        this.startServerWithConfiguredUser();
        RESTRequestGenerator.ResponseEntity response = ((RESTRequestGenerator)this.gen.get()).expectedStatus(200).withHeader("Authorization", HTTP.basicAuthHeader("neo4j", "secret")).get(this.userURL("neo4j"));
        JsonNode data = JsonHelper.jsonNode((String)response.entity());
        Assert.assertThat((Object)data.get("username").asText(), (Matcher)CoreMatchers.equalTo((Object)"neo4j"));
        Assert.assertThat((Object)data.get("password_change_required").asBoolean(), (Matcher)CoreMatchers.equalTo((Object)false));
        Assert.assertThat((Object)data.get("password_change").asText(), (Matcher)CoreMatchers.equalTo((Object)this.passwordURL("neo4j")));
    }

    @Test
    @Documented(value="Incorrect authentication\n\nIf an incorrect username or password is provided, the server replies with an error.")
    public void incorrect_authentication() throws JsonParseException, IOException {
        this.startServerWithConfiguredUser();
        RESTRequestGenerator.ResponseEntity response = ((RESTRequestGenerator)this.gen.get()).expectedStatus(401).withHeader("Authorization", HTTP.basicAuthHeader("neo4j", "incorrect")).expectedHeader("WWW-Authenticate", "Basic realm=\"Neo4j\"").post(this.dataURL());
        JsonNode data = JsonHelper.jsonNode((String)response.entity());
        JsonNode firstError = data.get("errors").get(0);
        Assert.assertThat((Object)firstError.get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)"Neo.ClientError.Security.Unauthorized"));
        Assert.assertThat((Object)firstError.get("message").asText(), (Matcher)CoreMatchers.equalTo((Object)"Invalid username or password."));
    }

    @Test
    @Documented(value="Required password changes\n\nIn some cases, like the very first time Neo4j is accessed, the user will be required to choose\na new password. The database will signal that a new password is required and deny access.\n\nSee <<rest-api-security-user-status-and-password-changing>> for how to set a new password.")
    public void password_change_required() throws JsonParseException, IOException {
        this.startServer(true);
        RESTRequestGenerator.ResponseEntity response = ((RESTRequestGenerator)this.gen.get()).expectedStatus(403).withHeader("Authorization", HTTP.basicAuthHeader("neo4j", "neo4j")).get(this.dataURL());
        JsonNode data = JsonHelper.jsonNode((String)response.entity());
        JsonNode firstError = data.get("errors").get(0);
        Assert.assertThat((Object)firstError.get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)"Neo.ClientError.Security.Forbidden"));
        Assert.assertThat((Object)firstError.get("message").asText(), (Matcher)CoreMatchers.equalTo((Object)"User is required to change their password."));
        Assert.assertThat((Object)data.get("password_change").asText(), (Matcher)CoreMatchers.equalTo((Object)this.passwordURL("neo4j")));
    }

    @Test
    @Documented(value="When auth is disabled\n\nWhen auth has been disabled in the configuration, requests can be sent without an +Authorization+ header.")
    public void auth_disabled() throws IOException {
        this.startServer(false);
        ((RESTRequestGenerator)this.gen.get()).expectedStatus(200).get(this.dataURL());
    }

    @Test
    public void shouldSayMalformedHeaderIfMalformedAuthorization() throws Exception {
        this.startServerWithConfiguredUser();
        HTTP.Response response = HTTP.withHeaders("Authorization", "This makes no sense").GET(this.dataURL());
        Assert.assertThat((Object)response.status(), (Matcher)CoreMatchers.equalTo((Object)400));
        Assert.assertThat((Object)response.get("errors").get(0).get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)"Neo.ClientError.Request.InvalidFormat"));
        Assert.assertThat((Object)response.get("errors").get(0).get("message").asText(), (Matcher)CoreMatchers.equalTo((Object)"Invalid authentication header."));
    }

    @Test
    public void shouldAllowDataAccess() throws Exception {
        this.startServerWithConfiguredUser();
        this.assertAuthorizationRequired("POST", "db/data/node", HTTP.RawPayload.quotedJson("{'name':'jake'}"), 201);
        this.assertAuthorizationRequired("GET", "db/data/node/1234", 404);
        this.assertAuthorizationRequired("POST", "db/data/transaction/commit", HTTP.RawPayload.quotedJson("{'statements':[{'statement':'MATCH (n) RETURN n'}]}"), 200);
        Assert.assertEquals((long)200L, (long)HTTP.GET(this.server.baseUri().resolve("").toString()).status());
    }

    @Test
    public void shouldNotAllowAnotherUserToAccessTransaction() throws Exception {
        this.startServerWithConfiguredUser();
        this.setupBobAndAliceUsers();
        HTTP.Response initiatingUserRequest = HTTP.withBasicAuth("bob", "secret").POST(this.server.baseUri().resolve("db/data/transaction/").toString(), AuthenticationIT.query("CREATE (n)"));
        Assertions.assertEquals((int)201, (int)initiatingUserRequest.status());
        HTTP.Response hijackingUserRequest = HTTP.withBasicAuth("alice", "secret").POST(initiatingUserRequest.location(), AuthenticationIT.query("CREATE (n)"));
        Assertions.assertEquals((int)404, (int)hijackingUserRequest.status());
        Assert.assertThat((Object)hijackingUserRequest.get("errors").get(0).get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)Status.Transaction.TransactionNotFound.code().serialize()));
        HTTP.Response initiatingUserCommitRequest = HTTP.withBasicAuth("bob", "secret").POST(initiatingUserRequest.location() + "/commit", AuthenticationIT.query("CREATE (n)"));
        Assertions.assertEquals((int)200, (int)initiatingUserCommitRequest.status());
    }

    @Test
    public void shouldNotAllowAnotherUserToCommitTransaction() throws Exception {
        this.startServerWithConfiguredUser();
        this.setupBobAndAliceUsers();
        HTTP.Response initiatingUserRequest = HTTP.withBasicAuth("bob", "secret").POST(this.server.baseUri().resolve("db/data/transaction/").toString(), AuthenticationIT.query("CREATE (n)"));
        Assertions.assertEquals((int)201, (int)initiatingUserRequest.status());
        HTTP.Response hijackingUserRequest = HTTP.withBasicAuth("alice", "secret").POST(initiatingUserRequest.location() + "/commit");
        Assertions.assertEquals((int)404, (int)hijackingUserRequest.status());
        Assert.assertThat((Object)hijackingUserRequest.get("errors").get(0).get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)Status.Transaction.TransactionNotFound.code().serialize()));
        HTTP.Response initiatingUserCommitRequest = HTTP.withBasicAuth("bob", "secret").POST(initiatingUserRequest.location() + "/commit", AuthenticationIT.query("CREATE (n)"));
        Assertions.assertEquals((int)200, (int)initiatingUserCommitRequest.status());
    }

    @Test
    public void shouldNotAllowAnotherUserToRollbackTransaction() throws Exception {
        this.startServerWithConfiguredUser();
        this.setupBobAndAliceUsers();
        HTTP.Response initiatingUserRequest = HTTP.withBasicAuth("bob", "secret").POST(this.server.baseUri().resolve("db/data/transaction/").toString(), AuthenticationIT.query("CREATE (n)"));
        Assertions.assertEquals((int)201, (int)initiatingUserRequest.status());
        HTTP.Response hijackingUserRequest = HTTP.withBasicAuth("alice", "secret").DELETE(initiatingUserRequest.location());
        Assertions.assertEquals((int)404, (int)hijackingUserRequest.status());
        Assert.assertThat((Object)hijackingUserRequest.get("errors").get(0).get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)Status.Transaction.TransactionNotFound.code().serialize()));
        HTTP.Response initiatingUserCommitRequest = HTTP.withBasicAuth("bob", "secret").POST(initiatingUserRequest.location() + "/commit", AuthenticationIT.query("CREATE (n)"));
        Assertions.assertEquals((int)200, (int)initiatingUserCommitRequest.status());
    }

    @Test
    public void shouldAllowAllAccessIfAuthenticationIsDisabled() throws Exception {
        this.startServer(false);
        Assert.assertEquals((long)201L, (long)HTTP.POST(this.server.baseUri().resolve("db/data/node").toString(), HTTP.RawPayload.quotedJson("{'name':'jake'}")).status());
        Assert.assertEquals((long)404L, (long)HTTP.GET(this.server.baseUri().resolve("db/data/node/1234").toString()).status());
        Assert.assertEquals((long)200L, (long)HTTP.POST(this.server.baseUri().resolve("db/data/transaction/commit").toString(), HTTP.RawPayload.quotedJson("{'statements':[{'statement':'MATCH (n) RETURN n'}]}")).status());
    }

    @Test
    public void shouldReplyNicelyToTooManyFailedAuthAttempts() throws Exception {
        this.startServerWithConfiguredUser();
        long timeout = System.currentTimeMillis() + 30000L;
        HTTP.Response response = null;
        while (System.currentTimeMillis() < timeout && (response = HTTP.withBasicAuth("neo4j", "incorrect").POST(this.server.baseUri().resolve("authentication").toString(), HTTP.RawPayload.quotedJson("{'username':'neo4j', 'password':'something that is wrong'}"))).status() != 429) {
        }
        Assert.assertThat((Object)response.status(), (Matcher)CoreMatchers.equalTo((Object)429));
        JsonNode firstError = response.get("errors").get(0);
        Assert.assertThat((Object)firstError.get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)"Neo.ClientError.Security.AuthenticationRateLimit"));
        Assert.assertThat((Object)firstError.get("message").asText(), (Matcher)CoreMatchers.equalTo((Object)"Too many failed authentication requests. Please wait 5 seconds and try again."));
    }

    @Test
    public void shouldNotAllowDataAccessForUnauthorizedUser() throws Exception {
        this.startServer(true);
        HTTP.Response response = HTTP.withBasicAuth("neo4j", "neo4j").POST(this.server.baseUri().resolve("authentication").toString(), HTTP.RawPayload.quotedJson("{'username':'neo4j', 'password':'neo4j'}"));
        Assert.assertEquals((long)403L, (long)HTTP.withBasicAuth("neo4j", "neo4j").POST(this.server.baseUri().resolve("db/data/node").toString(), HTTP.RawPayload.quotedJson("{'name':'jake'}")).status());
        Assert.assertEquals((long)403L, (long)HTTP.withBasicAuth("neo4j", "neo4j").GET(this.server.baseUri().resolve("db/data/node/1234").toString()).status());
        Assert.assertEquals((long)403L, (long)HTTP.withBasicAuth("neo4j", "neo4j").POST(this.server.baseUri().resolve("db/data/transaction/commit").toString(), HTTP.RawPayload.quotedJson("{'statements':[{'statement':'MATCH (n) RETURN n'}]}")).status());
    }

    private void assertAuthorizationRequired(String method, String path, int expectedAuthorizedStatus) throws JsonParseException {
        this.assertAuthorizationRequired(method, path, null, expectedAuthorizedStatus);
    }

    private void assertAuthorizationRequired(String method, String path, Object payload, int expectedAuthorizedStatus) throws JsonParseException {
        HTTP.Response response = HTTP.request(method, this.server.baseUri().resolve(path).toString(), payload);
        Assert.assertThat((Object)response.status(), (Matcher)CoreMatchers.equalTo((Object)401));
        Assert.assertThat((Object)response.get("errors").get(0).get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)"Neo.ClientError.Security.Unauthorized"));
        Assert.assertThat((Object)response.get("errors").get(0).get("message").asText(), (Matcher)CoreMatchers.equalTo((Object)"No authentication header supplied."));
        Assert.assertThat((Object)response.header("WWW-Authenticate"), (Matcher)CoreMatchers.equalTo((Object)"Basic realm=\"Neo4j\""));
        response = HTTP.withHeaders("Authorization", "This makes no sense").request(method, this.server.baseUri().resolve(path).toString(), payload);
        Assert.assertThat((Object)response.status(), (Matcher)CoreMatchers.equalTo((Object)400));
        Assert.assertThat((Object)response.get("errors").get(0).get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)"Neo.ClientError.Request.InvalidFormat"));
        Assert.assertThat((Object)response.get("errors").get(0).get("message").asText(), (Matcher)CoreMatchers.equalTo((Object)"Invalid authentication header."));
        response = HTTP.withBasicAuth("neo4j", "incorrect").request(method, this.server.baseUri().resolve(path).toString(), payload);
        Assert.assertThat((Object)response.status(), (Matcher)CoreMatchers.equalTo((Object)401));
        Assert.assertThat((Object)response.get("errors").get(0).get("code").asText(), (Matcher)CoreMatchers.equalTo((Object)"Neo.ClientError.Security.Unauthorized"));
        Assert.assertThat((Object)response.get("errors").get(0).get("message").asText(), (Matcher)CoreMatchers.equalTo((Object)"Invalid username or password."));
        Assert.assertThat((Object)response.header("WWW-Authenticate"), (Matcher)CoreMatchers.equalTo((Object)"Basic realm=\"Neo4j\""));
        response = HTTP.withBasicAuth("neo4j", "secret").request(method, this.server.baseUri().resolve(path).toString(), payload);
        Assert.assertThat((Object)response.status(), (Matcher)CoreMatchers.equalTo((Object)expectedAuthorizedStatus));
    }

    public void startServerWithConfiguredUser() throws IOException {
        this.startServer(true);
        HTTP.Response post = HTTP.withBasicAuth("neo4j", "neo4j").POST(this.server.baseUri().resolve("/user/neo4j/password").toString(), HTTP.RawPayload.quotedJson("{'password':'secret'}"));
        Assert.assertEquals((long)200L, (long)post.status());
    }

    private void setupBobAndAliceUsers() {
        HTTP.Response createBobRequest = HTTP.withBasicAuth("neo4j", "secret").POST(this.txCommitURL(), AuthenticationIT.query("CALL dbms.security.createUser('bob','secret',false)"));
        Assertions.assertEquals((int)200, (int)createBobRequest.status());
        HTTP.Response grantBobAdmin = HTTP.withBasicAuth("neo4j", "secret").POST(this.txCommitURL(), AuthenticationIT.query("CALL dbms.security.addRoleToUser('admin', 'bob')"));
        Assertions.assertEquals((int)200, (int)grantBobAdmin.status());
        HTTP.Response createAliceRequest = HTTP.withBasicAuth("neo4j", "secret").POST(this.txCommitURL(), AuthenticationIT.query("CALL dbms.security.createUser('alice','secret',false)"));
        Assertions.assertEquals((int)200, (int)createAliceRequest.status());
    }

    private static HTTP.RawPayload query(String statement) {
        return HTTP.RawPayload.rawPayload("{\"statements\":[{\"statement\":\"" + statement + "\"}]}");
    }
}

