/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.server.rest.transactional;

import java.time.Clock;
import java.util.concurrent.TimeUnit;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.junit.Assert;
import org.junit.Test;
import org.junit.jupiter.api.Assertions;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.internal.kernel.api.security.AuthSubject;
import org.neo4j.internal.kernel.api.security.LoginContext;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogProvider;
import org.neo4j.server.rest.transactional.TransactionHandle;
import org.neo4j.server.rest.transactional.TransactionHandleRegistry;
import org.neo4j.server.rest.transactional.error.InvalidConcurrentTransactionAccess;
import org.neo4j.server.rest.transactional.error.InvalidTransactionId;
import org.neo4j.server.rest.transactional.error.TransactionLifecycleException;
import org.neo4j.time.Clocks;
import org.neo4j.time.FakeClock;

public class TransactionHandleRegistryTest {
    @Test
    public void shouldGenerateTransactionId() {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)Clocks.fakeClock(), 0L, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        long id1 = registry.begin(handle);
        long id2 = registry.begin(handle);
        Assert.assertNotEquals((long)id1, (long)id2);
        logProvider.assertNoLoggingOccurred();
    }

    @Test
    public void shouldStoreSuspendedTransaction() throws Exception {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)Clocks.fakeClock(), 0L, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        long id = registry.begin(handle);
        registry.release(id, handle);
        TransactionHandle acquiredHandle = registry.acquire(id);
        Assert.assertSame((Object)handle, (Object)acquiredHandle);
        logProvider.assertNoLoggingOccurred();
    }

    @Test
    public void acquiringATransactionThatHasAlreadyBeenAcquiredShouldThrowInvalidConcurrentTransactionAccess() throws Exception {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)Clocks.fakeClock(), 0L, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        long id = registry.begin(handle);
        registry.release(id, handle);
        registry.acquire(id);
        try {
            registry.acquire(id);
            Assert.fail((String)"Should have thrown exception");
        }
        catch (InvalidConcurrentTransactionAccess invalidConcurrentTransactionAccess) {
            // empty catch block
        }
        logProvider.assertNoLoggingOccurred();
    }

    @Test
    public void acquiringANonExistentTransactionShouldThrowErrorInvalidTransactionId() throws Exception {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)Clocks.fakeClock(), 0L, (LogProvider)logProvider);
        long madeUpTransactionId = 1337L;
        try {
            registry.acquire(madeUpTransactionId);
            Assert.fail((String)"Should have thrown exception");
        }
        catch (InvalidTransactionId invalidTransactionId) {
            // empty catch block
        }
        logProvider.assertNoLoggingOccurred();
    }

    @Test
    public void transactionsShouldBeEvictedWhenUnusedLongerThanTimeout() throws Exception {
        FakeClock clock = Clocks.fakeClock();
        AssertableLogProvider logProvider = new AssertableLogProvider();
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)clock, 0L, (LogProvider)logProvider);
        TransactionHandle oldTx = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        TransactionHandle newTx = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        long txId1 = registry.begin(handle);
        long txId2 = registry.begin(handle);
        registry.release(txId1, oldTx);
        clock.forward(1L, TimeUnit.MINUTES);
        registry.release(txId2, newTx);
        registry.rollbackSuspendedTransactionsIdleSince(clock.millis() - 1000L);
        Assert.assertThat((Object)registry.acquire(txId2), (Matcher)CoreMatchers.equalTo((Object)newTx));
        try {
            registry.acquire(txId1);
            Assert.fail((String)"Should have thrown exception");
        }
        catch (InvalidTransactionId invalidTransactionId) {
            // empty catch block
        }
        logProvider.assertExactly(new AssertableLogProvider.LogMatcher[]{AssertableLogProvider.inLog(TransactionHandleRegistry.class).info("Transaction with id 1 has been automatically rolled back due to transaction timeout.")});
    }

    @Test
    public void expiryTimeShouldBeSetToCurrentTimePlusTimeout() throws Exception {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        FakeClock clock = Clocks.fakeClock();
        int timeoutLength = 123;
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)clock, (long)timeoutLength, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        long id = registry.begin(handle);
        long timesOutAt = registry.release(id, handle);
        Assert.assertThat((Object)timesOutAt, (Matcher)CoreMatchers.equalTo((Object)(clock.millis() + (long)timeoutLength)));
        clock.forward(1337L, TimeUnit.MILLISECONDS);
        registry.acquire(id);
        timesOutAt = registry.release(id, handle);
        Assert.assertThat((Object)timesOutAt, (Matcher)CoreMatchers.equalTo((Object)(clock.millis() + (long)timeoutLength)));
    }

    @Test
    public void shouldProvideInterruptHandlerForActiveTransaction() throws TransactionLifecycleException {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        FakeClock clock = Clocks.fakeClock();
        int timeoutLength = 123;
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)clock, (long)timeoutLength, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        long id = registry.begin(handle);
        registry.terminate(id);
        ((TransactionHandle)Mockito.verify((Object)handle, (VerificationMode)Mockito.times((int)1))).terminate();
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{handle});
    }

    @Test
    public void shouldProvideInterruptHandlerForSuspendedTransaction() throws TransactionLifecycleException {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        FakeClock clock = Clocks.fakeClock();
        int timeoutLength = 123;
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)clock, (long)timeoutLength, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        long id = registry.begin(handle);
        registry.release(id, handle);
        registry.terminate(id);
        ((TransactionHandle)Mockito.verify((Object)handle, (VerificationMode)Mockito.times((int)1))).terminate();
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{handle});
    }

    @Test(expected=InvalidTransactionId.class)
    public void gettingInterruptHandlerForUnknownIdShouldThrowErrorInvalidTransactionId() throws TransactionLifecycleException {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        FakeClock clock = Clocks.fakeClock();
        int timeoutLength = 123;
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)clock, (long)timeoutLength, (LogProvider)logProvider);
        registry.terminate(456L);
    }

    @Test
    public void sameUserShouldBeAbleToAcquireTransaction() throws Exception {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)Clocks.fakeClock(), 0L, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        LoginContext loginContext = this.mockLoginContext("Johannes");
        Mockito.when((Object)handle.getLoginContext()).thenReturn((Object)loginContext);
        long id = registry.begin(handle);
        registry.release(id, handle);
        TransactionHandle acquiredHandle = registry.acquire(id, loginContext);
        Assert.assertSame((Object)handle, (Object)acquiredHandle);
        logProvider.assertNoLoggingOccurred();
    }

    @Test
    public void differentUserShouldNotBeAbleToAcquireTransaction() throws Exception {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)Clocks.fakeClock(), 0L, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        LoginContext owningUser = this.mockLoginContext("Johannes");
        Mockito.when((Object)handle.getLoginContext()).thenReturn((Object)owningUser);
        long id = registry.begin(handle);
        registry.release(id, handle);
        LoginContext naughtyUser = this.mockLoginContext("Mr. Evil");
        Assertions.assertThrows(InvalidTransactionId.class, () -> registry.acquire(id, naughtyUser));
        logProvider.assertNoLoggingOccurred();
    }

    @Test
    public void shouldRetrieveHandlerLoginContext() throws Exception {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)Clocks.fakeClock(), 0L, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        LoginContext owningUser = this.mockLoginContext("Johannes");
        Mockito.when((Object)handle.getLoginContext()).thenReturn((Object)owningUser);
        long id = registry.begin(handle);
        LoginContext actualLoginContext = registry.getLoginContextForTransaction(id);
        Assert.assertSame((Object)owningUser, (Object)actualLoginContext);
    }

    @Test
    public void differentUserShouldNotBeAbleToTerminateTransaction() throws TransactionLifecycleException {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        FakeClock clock = Clocks.fakeClock();
        int timeoutLength = 123;
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)clock, (long)timeoutLength, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        LoginContext loginContext = this.mockLoginContext("Johannes");
        Mockito.when((Object)handle.getLoginContext()).thenReturn((Object)loginContext);
        long id = registry.begin(handle);
        LoginContext otherUser = this.mockLoginContext("Dr. Evil");
        Assertions.assertThrows(InvalidTransactionId.class, () -> registry.terminate(id, otherUser));
    }

    @Test
    public void sameUserShouldBeAbleToTerminateTransaction() throws TransactionLifecycleException {
        AssertableLogProvider logProvider = new AssertableLogProvider();
        FakeClock clock = Clocks.fakeClock();
        int timeoutLength = 123;
        TransactionHandleRegistry registry = new TransactionHandleRegistry((Clock)clock, (long)timeoutLength, (LogProvider)logProvider);
        TransactionHandle handle = (TransactionHandle)Mockito.mock(TransactionHandle.class);
        LoginContext loginContext = this.mockLoginContext("Johannes");
        Mockito.when((Object)handle.getLoginContext()).thenReturn((Object)loginContext);
        long id = registry.begin(handle);
        registry.terminate(id, loginContext);
        ((TransactionHandle)Mockito.verify((Object)handle, (VerificationMode)Mockito.times((int)1))).terminate();
        ((TransactionHandle)Mockito.verify((Object)handle)).getLoginContext();
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{handle});
    }

    private LoginContext mockLoginContext(String userName) {
        LoginContext loginContext = (LoginContext)Mockito.mock(LoginContext.class);
        AuthSubject authSubject = (AuthSubject)Mockito.mock(AuthSubject.class);
        Mockito.when((Object)loginContext.subject()).thenReturn((Object)authSubject);
        Mockito.when((Object)authSubject.username()).thenReturn((Object)userName);
        return loginContext;
    }
}

